<?php

defined( 'ABSPATH' ) or die( "Oops! This is a WordPress plugin and should not be called directly.\n" );

/**
 * Class for Video Blogster RSS support
 */
if ( ! class_exists( 'Video_Blogster_RSS' )) {
    class Video_Blogster_RSS {

	private $vbp = null;			// will point to main Video_Blogster instance
        private $query_fields = array();        // options for current query

	private $total = 0;			// total results of feed
        private $num_skipped = 0;               // a sum of all videos skipped for this request
        private $num_updated = 0;               // a sum of all videos updated for this request
        private $num_imported = 0;              // a sum of all videos imported for this request

	/**
	 * Create RSS video source
	 * Point back to Video Blogster object to use common functions
	 * Save the query fields for easy access
	 */
	public function __construct( $vbp, $query_fields ) {
		$this->vbp = $vbp;
		$this->query_fields = $query_fields;
		$this->vbp->info_message( sprintf( esc_html__( 'Creating %s resource', 'video-blogster' ), $query_fields[ 'videoSource' ] ), 'notice notice-warning', 'debug' );
	}

        /**
         * Extract video details into our generic videoInfo array
         */
	private function get_video_info( $item ) {
			$videoInfo = array();

			$videoInfo['videoSource'] = 'RSS';
			$videoInfo['videoID'] = $item->get_id();
			$videoInfo['orig_title'] = $videoInfo['title'] = $item->get_title();

		// that's enough to check for duplicate post
		if ( $this->vbp->check_post_duplicate( $this->query_fields, $videoInfo ) ) {
			$this->num_skipped++;
			return null;
		}

			$videoInfo['orig_desc'] = $videoInfo['desc'] = $item->get_content();
			$this->vbp->info_message( sprintf (__( 'RSS feed entry: %s', 'video-blogster' ), $videoInfo['title'] ), 'notice notice-warning', 'debug' );
			$this->vbp->info_message( sprintf (__( 'RSS feed content: %s', 'video-blogster' ), htmlentities( $videoInfo['desc'] ) ), 'notice notice-warning', 'debug' );

			$author = $item->get_author();
			$videoInfo['association'] = isset( $author ) ? $author->get_name() : '';
			$videoInfo['url'] = $item->get_link(); // default: url of feed item

			$videoInfo['videoEmbed'] = '';

			$publishedAt = TRUE == $this->query_fields['cUsePublishedDate'] ? $item->get_date() : null;
			$date = $this->vbp->getDateTime( $publishedAt, $videoInfo['url'] );
			$videoInfo['publishedAt'] = $date ? $date->format( 'Y-m-d H:i:s' ) : current_time( 'Y-m-d H:i:s' );
			$videoInfo['publishedAt8601'] = $date ? $date->format( 'c' ) : current_time( 'c' );

			$videoInfo['publishedAt8601'] = isset( $date ) ? $date->format( 'c' ) : '';

			$videoInfo['duration'] = '';
			$videoInfo['viewCount'] = $videoInfo['likeCount'] = $videoInfo['dislikeCount'] = $videoInfo['favoriteCount'] = $videoInfo['commentCount'] = 0;

			// check for item categories
			$videoInfo['sourceCategory'] = $videoInfo['tags'] = '';
			$cats = array();
			$rssCats = $item->get_categories();
			if ( ! empty( $rssCats ) ) {
				foreach ( $rssCats as $cat ) {
					$cats[] = $cat->get_label();
				}
			}

			#see if enclosure exists
			foreach ( $item->get_enclosures() as $enclosure ) {

				// grab the first thumbnail declared, if any:
				if ( empty( $videoInfo['img'] ) ) {
					$img = $enclosure->get_thumbnail(); // grabs <media:thumbnail>
					if ( ! empty( $img ) ) {
						$videoInfo['img'] = $img;
						$this->vbp->info_message( sprintf (__( '[%s] enclosure thumbnail: %s', 'video-blogster' ), $videoInfo['title'], $videoInfo['img'] ), 'notice notice-warning', 'debug' );
					}
				}

				// check all media content, if any:
				if ( $media = $enclosure->get_link() ) {
					$this->vbp->info_message( sprintf (__( '[%s] enclosure media: %s', 'video-blogster' ), $videoInfo['title'], $media ), 'notice notice-warning', 'debug' );

					$type = $enclosure->get_type();
					$this->vbp->info_message( sprintf (__( '[%s] enclosure type: %s', 'video-blogster' ), $videoInfo['title'], ! empty( $type ) ? $type : '(not found)' ), 'notice notice-warning', 'debug' );

					$medium = $enclosure->get_medium();
					$this->vbp->info_message( sprintf (__( '[%s] enclosure medium: %s', 'video-blogster' ), $videoInfo['title'], ! empty( $medium ) ? $medium : '(not found)' ), 'notice notice-warning', 'debug' );
					if ( false !== strpos( $type, 'image' ) ) { // this is an image, use it
						$medium = $media;
					}

					if ( ( $medium == 'image' || false !== strpos( $type, 'image' ) ) && empty( $videoInfo['img'] ) ) {
						$videoInfo['img'] = $media;
						$this->vbp->info_message( sprintf (__( '[%s] enclosure img %s:', 'video-blogster' ), $videoInfo['title'], $videoInfo['img'] ), 'notice notice-warning', 'debug' );
					}
					else if ( $medium == 'video' || FALSE !== strpos( $type, 'video') ) {
						$videoInfo['videoEmbed'] = $media;
						$videoInfo['duration'] = $this->vbp->seconds_to_time( $enclosure->get_duration() ); 
						$this->vbp->info_message( sprintf (__( '[%s] enclosure video detected: %s', 'video-blogster' ), $videoInfo['title'], $videoInfo['videoEmbed'] ), 'notice notice-warning', 'debug' );
						$this->vbp->info_message( sprintf (__( '[%s] enclosure duration converted from %s to %s', 'video-blogster' ), $videoInfo['title'], $enclosure->get_duration(), $videoInfo['duration'] ), 'notice notice-warning', 'debug' );
					}
					else if ( empty( $type ) && empty( $medium ) ) {
						$videoInfo['videoEmbed'] = $videoInfo['url'] = $media;
						$videoInfo['duration'] = $this->vbp->seconds_to_time( $enclosure->get_duration() ); 
						$this->vbp->info_message( sprintf (__( '[%s] enclosure video GUESS (RSS feed missing type and medium attributes): %s', 'video-blogster' ), $videoInfo['title'], $videoInfo['videoEmbed'] ), 'notice notice-warning', 'debug' );
						$this->vbp->info_message( sprintf (__( '[%s] enclosure duration converted from %s to %s', 'video-blogster' ), $videoInfo['title'], $enclosure->get_duration(), $videoInfo['duration'] ), 'notice notice-warning', 'debug' );
					}
					else {
						$videoInfo['rssEnclosure'] = $media;
					}
				}

				// check for enclosure categories
				$rssCats = $enclosure->get_categories();
				if ( ! empty( $rssCats ) ) {
					foreach ( $rssCats as $cat ) {
						$cats[] = $cat->get_label();
					}
				}

			}
			#if no enclosure thumbnail defined, look for the first image in the content
			if ( ! isset( $videoInfo['img'] ) || empty( $videoInfo['img'] ) ) {
				preg_match( '/<img([\w\W]+?)[\/]?>/i', $videoInfo['desc'], $matches );
				$imgTag = isset( $matches[0] ) ? $matches[0] : null;
				if ( $imgTag ) {
					preg_match( '/src\s*=\s*"(.+?)"/', $imgTag, $matches );
					$videoInfo['img'] = isset( $matches[1] ) ? $matches[1] : null;
				}
				if ( $imgTag && ! empty( $videoInfo['img'] ) && $this->query_fields['qOrderBy'] ) { 
					$videoInfo['desc'] = str_replace( $imgTag, '', $videoInfo['desc'] );
				}
			}

			#filter entries by given category, if any
			#make the comparison case insensitive by making both lowercase...
			if ( ! empty ( $filter_cats ) && ! empty( $cats ) ) {
				$this->vbp->info_message( sprintf (__( '%s: filter entries by categories %s', 'video-blogster' ), __FUNCTION__, $filter_cats ), 'notice notice-warning', 'debug' );
				$lc_tags = array_map( 'strtolower', $cats );
				$cat_match = false;
				$fcats = explode( ",", strtolower( $filter_cats ) );
				foreach ( $fcats as $fcat ) {
					if ( in_array( trim( $fcat ), $lc_tags ) ) {
						$cat_match = true;
						$this->vbp->info_message( sprintf (__( '%s: RSS item [%s] matches filter.', 'video-blogster' ), __FUNCTION__, $videoInfo['title'] ), 'notice notice-warning', 'debug' );
						break;
					}
				}
				if ( false === $cat_match ) {
					$this->vbp->info_message( sprintf (__( '%s: RSS item [%s] does not match filter.', 'video-blogster' ), __FUNCTION__, $videoInfo['title'] ), 'notice notice-warning', 'debug' );
					$this->num_skipped++;
					return null;
				}
			}

			if ( isset( $cats ) ) {
				$videoInfo['sourceCategoryName'] = implode( ',', $cats );
			}

			$videoInfo['feedID'] = ! empty ( $this->query_fields['id'] ) ? $this->query_fields['id'] : "";
			$videoInfo = apply_filters( 'vbp_get_video_info', $videoInfo, $item );

			$this->vbp->info_message( sprintf( esc_html__( '%s: %s', 'video-blogster' ), __FUNCTION__, htmlentities( print_r($videoInfo,true) ) ), 'notice notice-warning', 'debug' );
		return $videoInfo;
	}

	public function grab_videos() {
		$maxItems = $this->query_fields['qQueryBehavior'] == 'strict' ? $this->query_fields['qNumVideos'] : 0;
		if ( $maxItems < 0 ) $maxItems = 0;
		$assoc =  esc_url_raw( strip_tags( $this->query_fields['qAssoc'] ) );
		$filter_cats = $this->query_fields['qKeyphrase'];

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
		if ( empty( $assoc ) ) {
			return $this->vbp->info_message( sprintf( esc_html__( '%s Error: No RSS feed has been set', 'video-blogster' ), $feedID ), 'error', 'video_import' );
		}

		require_once( ABSPATH . WPINC . '/feed.php' );
		$this->vbp->info_message( sprintf( esc_html__( 'Fetching RSS feed: %s', 'video-blogster' ), $assoc ), 'notice notice-warning', 'debug' );
		$rss = fetch_feed( $assoc );
		if ( is_wp_error( $rss ) ) {
			return $this->vbp->info_message( sprintf ( esc_html__( '%s WP error %s with fetch_feed: %s', 'video-blogster' ), $feedID, $rss->get_error_code(), $rss->get_error_message() ), 'error', 'video_import' );
		}

		$this->total = $rss->get_item_quantity();
		$this->vbp->info_message( sprintf( esc_html__( 'RSS feed has %s total entries. Parsing %d of them.', 'video-blogster' ), $this->total, $maxItems ), 'notice notice-warning', 'debug' );
		$rss_items = $rss->get_items( 0, $maxItems );

		if ( empty( $rss_items ) ) {
			return $this->vbp->info_message( sprintf (__( '%s Error - no rss items found in fetch_feed: %s', 'video-blogster' ), $feedID, $assoc ), 'error', 'critical' );
		}

		foreach ( $rss_items as $item ) {

			$this->vbp->info_message( sprintf( esc_html__( 'Checking RSS feed item: [%s] from %s', 'video-blogster' ), $item->get_title(), $assoc ), 'notice notice-warning', 'debug' );

			$videoInfo = $this->get_video_info( $item );

			if ( ! $videoInfo ) continue;

			$postID = $this->vbp->save_the_video( $this->query_fields, $videoInfo );

			if ( $postID < 0 ) return 0; // user abort!
			if ( ! $postID ) { $this->num_skipped++; continue; }

			$this->vbp->publish_the_video( $postID, $this->query_fields, $videoInfo );

                        if ( $videoInfo['action'] == 'saved' ) $this->num_imported++;
			else if ( $videoInfo['action'] == 'updated' ) $this->num_updated++;

			$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
			if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) break;
		}

		$this->vbp->import_finished( 'RSS entries', $this->num_skipped, $this->num_updated, $this->num_imported, $this->total, $this->query_fields['id'], $this->query_fields['qQueryBehavior'] );
		return 1;
	}

	/**
	 * RSS feeds are fully imported - no need to check for deletions from source site
	 */
	public function check_for_deleted_videos() {
	}

    } // END class Video_Blogster_RSS
} // END if ( ! class_exists( 'Video_Blogster_RSS' ) )

?>
