<?php

defined( 'ABSPATH' ) or die( "Oops! This is a WordPress plugin and should not be called directly.\n" );

/**
 * Class for Video Blogster SoundCloud support
 */
if ( ! class_exists( 'Video_Blogster_SoundCloud' )) {
    class Video_Blogster_SoundCloud {

	private $vbp = null;			// will point to main Video_Blogster instance
        private $query_fields = array();        // options for current query
        private $client_id = null;        	// SoundCloud client_id for Video Blogster
        private $comment_num = 0;       	// Internal counter
        private $batch_limit = 200;             // Max amount of tracks we can request per API query

        private $total = 0;                	// total number of results
        private $num_skipped = 0;               // a sum of all videos skipped for this request
        private $num_updated = 0;               // a sum of all videos updated for this request
        private $num_imported = 0;              // a sum of all videos imported for this request

	/**
	 * Create SoundCloud video source
	 * Point back to Video Blogster object to use common functions
	 * Save the query fields for easy access
	 */
	public function __construct( $vbp, $query_fields, $client_id ) {
		$this->vbp = $vbp;
		$this->query_fields = $query_fields;
		$this->client_id = $client_id;
		$this->vbp->info_message( sprintf( esc_html__( 'Creating %s resource', 'video-blogster' ), $query_fields[ 'videoSource' ] ), 'notice notice-warning', 'debug' );
	}

        /**
         * Make the query and check for errors.
         */
        private function queryApi( $url ) {
		$this->vbp->info_message( sprintf( '%s : %s%s%s', 
			__FUNCTION__, 
			'<a target="_blank" href="' . esc_url( $url ) . '">',
			esc_url( $url ),
			'</a>'
			), 'notice notice-warning', 'debug' );


		$response = wp_remote_get( $url );

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
                if ( is_wp_error( $response ) ) {
			return $this->vbp->info_message( sprintf( esc_html__( '%s WP error %s in %s(%s) - %s ', 'video-blogster' ), $response->get_error_code(), $feedID, __FUNCTION__, $url, $response->get_error_message() ) );
                }

		$code = wp_remote_retrieve_response_code( $response );
		$message = wp_remote_retrieve_response_message( $response );

                if ( $code != 200 ) {
                        return $this->vbp->info_message( sprintf( esc_html__( '%s(%s) - SoundCloud API returned %s: %s', 'video-blogster' ), __FUNCTION__, $url, $code, $message ) );
		}

		$body = wp_remote_retrieve_body( $response );
                $data = json_decode( $body );

                return $data;
        }


	private function grab_comments_batch( $url, $postID ) {
		$totalComments = $this->query_fields['qNumComments'];

		$result = $this->queryApi( $url );

		$comments = isset( $result->collection ) ? $result->collection : $result;

		if ( empty( $comments ) ) {
			return null;
		}

		$comment_date = current_time( 'Y-m-d H:i:s' );
		foreach ( $comments as $comment ) {
			if ( $this->comment_num < 50 ) {
				$this->vbp->info_message( sprintf (__( 'COMMENT: %s', 'video-blogster' ), htmlentities( print_r($comment,TRUE) ) ), 'notice notice-warning', 'debug' );
			}
			$author = isset( $comment->user->username ) ? trim( $comment->user->username ) : '';
			$author_url = isset( $comment->user->permalink_url ) ? trim( $comment->user->permalink_url ) : '';
			$content = isset( $comment->body ) ? trim( $comment->body ) : '';
			if ( empty( $author ) || empty( $content ) ) {
				continue;
			}
			if ( TRUE == $this->query_fields['cUsePublishedDate'] ) {
				$commentAt = isset( $comment->created_at ) ? $comment->created_at : current_time( 'Y-m-d H:i:s' );
				$commentDate = $this->vbp->getDateTime( $commentAt, $url );
				$comment_date = isset( $commentDate ) ? $commentDate->format('Y-m-d H:i:s') : $commentAt;
			}
			$commentdata = array(
				'comment_post_ID'       => $postID,
				'comment_author'        => $author,
				'comment_author_url'    => $author_url,
				'comment_content'       => $content,
				'comment_date'          => $comment_date,
				'comment_approved'      => 1
				);
			$commentdata = apply_filters( 'vbp_soundcloud_comment', $commentdata );
			$comment_id = $this->vbp->save_the_comment( $commentdata, $this->query_fields );
			if ( $comment_id ) {
				$this->comment_num++;
			}
			if ( $totalComments > 0 && $this->comment_num >= $totalComments ) {
				return null;
			}
		}
		return isset( $result->next_href ) ? $result->next_href : null;
	}


	/**
	 * Get SoundCloud comments on a track
	 */
	private function grab_comments( $videoID, $postID ) {
		$cm = $cn = false;
		$this->vbp->bulk_comment_start( $cm, $cn );

		$url = "https://api.soundcloud.com/tracks/" . $videoID . "/comments.json?client_id=" . $this->client_id;

		$url .= "&linked_partitioning=1";

		$this->comment_num = 0;

		$url = apply_filters( 'vbp_soundcloud_comments_query', $url, $this->query_fields );

		while ( $url ) {
			$url = $this->grab_comments_batch( $url, $postID );
		}

		$this->vbp->info_message( sprintf( esc_html__( '%s comments imported successfully to post id %s.', 'video-blogster' ), $this->comment_num, $postID ), 'updated', 'video_import' );

		$this->vbp->bulk_comment_end( $cm, $cn );
	}

	private function get_best_thumbnail( $track ) {
		$img = array();

		$size = isset( $this->query_fields['qExtraParams2'] ) ? $this->query_fields['qExtraParams2'] : 't500x500';

		if ( isset( $track->user->avatar_url ) ) {
			$img[] = $track->user->avatar_url;
			$img[] = str_replace( 'large', $size, $track->user->avatar_url );
		}
		if ( isset( $track->artwork_url ) ) {
			$img[] = $track->artwork_url;
			$img[] = str_replace( 'large', $size, $track->artwork_url );
		}

		$img = array_reverse( $img );

		$image = $this->vbp->verify_thumbnails( $img );

		return $image;
	}

        /**
         * Extract video details into our generic videoInfo array
         */
        private function get_video_info( $track ) {
			$videoInfo = array();

			$videoInfo['videoSource'] = 'SoundCloud';
			$videoInfo['videoID'] = isset( $track->id ) ? $track->id : '';
			$videoInfo['orig_title'] = $videoInfo['title'] = isset( $track->title ) ? $track->title : '';

		// that's enough to check for duplicate post
		if ( $this->vbp->check_post_duplicate( $this->query_fields, $videoInfo ) ) {
			$this->num_skipped++;
			return null;
		}

			$videoInfo['orig_desc'] = $videoInfo['desc'] = isset( $track->description ) ? $track->description : '';

			$videoInfo['authorUrl'] = isset( $track->user->permalink_url ) ? $track->user->permalink_url : '';
			$videoInfo['authorTitle'] = isset( $track->user->permalink ) ? $track->user->permalink: '';

			$videoInfo['association'] = isset( $track->user->username ) ? $track->user->username : '';
			$videoInfo['url'] = isset( $track->permalink_url ) ? $track->permalink_url : '';

			$this->vbp->info_message( sprintf( esc_html__( 'Checking Soundcloud track: [%s] at %s', 'video-blogster' ), $videoInfo['title'], $videoInfo['url'] ), 'notice notice-warning', 'debug' );

			$embeddable = isset( $track->embeddable_by ) ? $track->embeddable_by : '';
			if ( $embeddable != 'all' ) {
				$this->vbp->info_message( sprintf( esc_html__( 'Track only embeddable_by \'%s\'. Skipping.', 'video-blogster' ), $embeddable ), 'notice notice-warning', 'debug' );
				$this->num_skipped++;
				return null;
			}


			$videoInfo['duration'] = isset( $track->duration ) ? $track->duration : 0;
			$videoInfo['duration'] = $this->vbp->seconds_to_time( $videoInfo['duration'] / 1000 );
			$videoInfo['viewCount'] = isset( $track->playback_count ) ? $track->playback_count : '';
			$videoInfo['likeCount'] = $videoInfo['favoriteCount'] = isset( $track->favoritings_count ) ? $track->favoritings_count : '';
			$videoInfo['dislikeCount'] = 0;
			$videoInfo['commentCount'] = isset( $track->comment_count ) ? $track->comment_count : '';
			$videoInfo['img'] = $this->get_best_thumbnail( $track );


			// Make a separate call to Soundcloud oembed for embed code 
			$url = "https://www.soundcloud.com/oembed?url=" . htmlentities( $videoInfo['url'] ) . "&format=json";
			$videoInfo['videoEmbed'] = $this->vbp->queryoEmbed( $url );

			$publishedAt = isset( $track->created_at ) ? $track->created_at : null;
			$date = $this->vbp->getDateTime( $publishedAt, $videoInfo['url'] );
			$videoInfo['publishedAt'] = $date->format( 'Y-m-d H:i:s' );
			$videoInfo['publishedAt8601'] = isset( $date ) ? $date->format( 'c' ) : '';

			$videoInfo['sourceCategory'] = '';
			if ( isset( $track->genre ) ) {
				$videoInfo['sourceCategoryName'] = $track->genre;
			}

			$videoInfo['tags'] = '';
			if ( isset( $track->tag_list ) ) {
				$videoInfo['tags'] = rtrim( $track->tag_list );
				$videoInfo['tags'] = preg_replace('/(\\s)(?=(?:[^"]|"[^"]*")*$)/', ',', $videoInfo['tags'] );
				$videoInfo['tags'] = str_replace( "\"", "", $videoInfo['tags'] );
			}

			$videoInfo['feedID'] = ! empty ( $this->query_fields['id'] ) ? $this->query_fields['id'] : "";
			$videoInfo = apply_filters( 'vbp_get_video_info', $videoInfo, $track );

			$this->vbp->info_message( sprintf( esc_html__( '%s: %s', 'video-blogster' ), __FUNCTION__, htmlentities( print_r($videoInfo,true) ) ), 'notice notice-warning', 'debug' );
		return $videoInfo;
	}

	private function save_tracks( $tracks ) {
                if ( ! $tracks ) {
                        return 0;
                }
		foreach ( $tracks as $track ) {

			$videoInfo = $this->get_video_info( $track );

			if ( ! $videoInfo ) { continue; }

			$postID = $this->vbp->save_the_video( $this->query_fields, $videoInfo );

			if ( $postID < 0 ) return 0; // user abort!
			if ( ! $postID ) { $this->num_skipped++; continue; }

			if ( ! empty( $this->query_fields['qNumComments'] ) && $videoInfo['commentCount'] ) {
				$this->grab_comments( $videoInfo['videoID'], $postID );
			}

			$this->vbp->publish_the_video( $postID, $this->query_fields, $videoInfo );

                        if ( $videoInfo['action'] == 'saved' ) $this->num_imported++;
                        else if ( $videoInfo['action'] == 'updated' ) $this->num_updated++;

			$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
			if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;
		}
		$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
		if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;
		return 1;
	}

	public function grab_videos_batch( $url ) {
		$result = $this->queryApi( $url );
		if ( false !== strpos( $url, '/tracks/' ) ) {
			$tracks = isset( $result ) ? array( $result ) : array();
		}
		else {
			$tracks = isset( $result->collection ) ? $result->collection : array();
		}

		$this->total += count( $tracks );

		if ( $result && empty( $tracks ) ) {
			$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
			return $this->vbp->info_message( sprintf( 
				esc_html__( '%s %s: No results returned for %s%s%s.%sThis could mean this resource is only available on main SoundCloud site or SoundCloud Go+, and not available to 3rd party sites.', 'video-blogster' ), 
				$feedID, 
				__FUNCTION__, 
				'<a target="_blank" href="' . $url . '">',
				$url,
				'</a>',
				'<br>' 
			), 'notice notice-warning', 'critical' );
		}

		$this->vbp->info_message( sprintf( esc_html__( 'SoundCloud response data (%s total results): %s', 'video-blogster' ), $this->total, htmlentities( print_r( $tracks,true ) ) ), 'notice notice-warning', 'debug' );

		if ( ! $this->save_tracks( $tracks ) ) return 0;

		return isset( $result->next_href ) ? $result->next_href : null;
	}

	/*
	 * https://developers.soundcloud.com/docs/api/reference
	 */
	public function grab_videos() {
                $maxItems = $this->query_fields['qQueryBehavior'] == 'strict' ? $this->query_fields['qNumVideos'] : -1;
                $limit = $maxItems > $this->batch_limit || $maxItems < 0 ? $this->batch_limit : $maxItems;


		$assoc =  $this->query_fields['qAssoc'];	# group, playlist, user, or track
		$assocType = $this->query_fields['qAssocType'];
		$keyphrase = urlencode( $this->query_fields['qKeyphrase'] );
		$filters = ! empty( $this->query_fields['qExtraParams'] ) ? "&" . ltrim( $this->query_fields['qExtraParams'], "&" ) : '';

                if ( empty( $keyphrase ) && ( empty( $assoc ) || empty( $assocType ) ) ) {
			$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
                        return $this->vbp->info_message( sprintf( esc_html__( '%s Error: Missing URL or keyphrase.', 'video-blogster' ), $feedID ), 'error', 'critical' );
                }

		if ( stripos( $assoc, 'http://' ) !== FALSE || stripos( $assoc, 'https://' ) !== FALSE ) {
			// resolve SoundCloud url to get id!
			$url = "https://api.soundcloud.com/resolve.json?" . http_build_query( array( 'url' => $assoc, 'client_id' => $this->client_id ) );
			$item = $this->queryApi( $url );
			if ( empty( $item ) ) {
				return; 
			}
			$this->vbp->info_message( sprintf (__( 'SoundCloud resolved [%s] to get item id %s', 'video-blogster' ), $assoc, $item->id), 'notice notice-warning', 'debug' );
			$assoc = $item->id;
		}

		if ( $keyphrase && ! $assoc ) {
			$url = "https://api.soundcloud.com/tracks.json?client_id=" . $this->client_id;
		}
		else if ( $assocType == 'track' ) { // fetch specific track
			$url = "https://api.soundcloud.com/tracks/" . $assoc . ".json?client_id=" . $this->client_id;
		}
		else if ( $assocType == 'user' ) {
			$url = "https://api.soundcloud.com/users/" . $assoc . "/tracks.json?client_id=" . $this->client_id;
		}
		else if ( $assocType == 'playlist' ) {
			$url = "https://api.soundcloud.com/playlists/" . $assoc . "/tracks.json?client_id=" . $this->client_id;
		}
		else if ( $assocType == 'group' ) {
			$url = "https://api.soundcloud.com/groups/" . $assoc . "/tracks.json?client_id=" . $this->client_id;
		}
		if ( $keyphrase ) {
			$url .= "&q=" . $keyphrase;
		}
		$url .= "&filter=public";
		if ( $filters ) {
			$url .= $filters;
		}

		$url .= "&limit=" . $limit;

		$url .= "&linked_partitioning=1";

		$url = apply_filters( 'vbp_soundcloud_search_query', $url, $this->query_fields );

		while ( $url ) {
			$url = $this->grab_videos_batch( $url );
		}

		if ( $this->query_fields['qQueryBehavior'] == 'strict' ) $this->total = 0; // no way to know
		$this->vbp->import_finished( 'SoundCloud tracks', $this->num_skipped, $this->num_updated, $this->num_imported, $this->total, $this->query_fields['id'], $this->query_fields['qQueryBehavior'] );
		return 0;
	}

	/**
	 * SoundCloud 
	 */
	public function check_for_deleted_videos() {

		$getMore = true;
		$offset = 0;
		$chunksize = apply_filters( 'vbp_posts_chunksize', 1000 );

		while ( $getMore ) {
		$posts = $this->vbp->get_posts_by_site( 'SoundCloud', $offset );
		if ( ! $posts ) return;

		$num_results = count( $posts );

		if ( $num_results && $offset == 0 ) {
			$this->vbp->info_message( sprintf( esc_html__( 'SoundCloud %s Start', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
		}

		foreach ( $posts as $post_id ) {
			$videoID = get_post_meta( $post_id, 'VideoID', TRUE );
			if ( ! $videoID ) {
				continue;
			}
			$url = get_post_meta( $post_id, 'VideoUrl', TRUE );
			$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: checking SoundCloud track id %s, url: %s ', 'video-blogster' ), $videoID, $url ), 'updated', 'utility_funcs' );
			$url = "https://api.soundcloud.com/tracks/" . $videoID . ".json?" . http_build_query( array( 'client_id' => $this->client_id ) );
			$track = $this->queryApi( $url );
			if ( empty( $track ) || 
				( isset( $track->errors[0]->error_message ) && false !== stripos( $track->errors[0]->error_message, '404 - Not Found' ) )
			) {
				$postID = apply_filters( 'vbp_delete_video', $post_id );
				if ( ! $postID ) {
					continue;
				}
				wp_trash_post( $postID );
				$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: SoundCloud video %s not found. Sending post %s to trash.', 'video-blogster' ), $videoID, $postID ), 'updated', 'critical' );
			}
		} // end foreach posts
			if ( $num_results < $chunksize ) break;
			$offset += $chunksize;
			$query_args['offset'] = $offset;
		} // end getMore
		if ( $num_results || $offset ) $this->vbp->info_message( sprintf( esc_html__( 'SoundCloud %s End', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
	}

    } // END class Video_Blogster_SoundCloud
} // END if ( ! class_exists( 'Video_Blogster_SoundCloud' ) )


?>
