<?php

defined( 'ABSPATH' ) or die( "Oops! This is a WordPress plugin and should not be called directly.\n" );

/**
 * Class for Video Blogster Spotify support
 * https://developer.spotify.com/
 * https://developer.spotify.com/documentation/web-api/reference/#/
 */
if ( ! class_exists( 'Video_Blogster_Spotify' )) {
    class Video_Blogster_Spotify {

	private $vbp = null;			// will point to main Video_Blogster instance
        private $query_fields = array();        // options for current query

	private $batch_limit = 50;             // max amount we can request per query
	private $access_token = null;

        private $total = 0;                	// total results
        private $num_skipped = 0;               // a sum of all videos skipped for this request
        private $num_updated = 0;               // a sum of all videos updated for this request
        private $num_imported = 0;              // a sum of all videos imported for this request

	/**
	 * Create Spotify video source
	 * Point back to Video Blogster object to use common functions
	 * Save the query fields for easy access
	 */
	public function __construct( $vbp, $query_fields ) {
		$this->vbp = $vbp;
		$this->query_fields = $query_fields;
		$this->vbp->info_message( sprintf( esc_html__( 'Creating %s resource', 'video-blogster' ), $query_fields[ 'videoSource' ] ), 'notice notice-warning', 'debug' );
	}

        /**
         * Make the query and check for errors.
         */
        private function getAccessToken() {
		$transient = "vbp_spotify_token";
		if ( false !== ( $data = get_transient( $transient ) ) ) {
			$this->access_token = $data;
			return;
		}
		$clientID = "56c7be9615ff4ba9b137699158d104dd";
		$clientSecret = "ca2b7717a53c44fbbee4ec40659b5407";
		$headers = array(
			'Authorization'		=> 'Basic ' . $this->vbp->vbp_encode( $clientID . ":" . $clientSecret )
		);
		$body = "grant_type=client_credentials";
		$url = "https://accounts.spotify.com/api/token";

		$response = wp_remote_post( $url, array( 'body' => $body, 'headers' => $headers ) );

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
                if ( is_wp_error( $response ) ) {
			return $this->vbp->info_message( sprintf( esc_html__( '%s WP error %s in %s(%s) - %s ', 'video-blogster' ), $response->get_error_code(), $feedID, __FUNCTION__, $url, $response->get_error_message() ) );
                }
                $data = json_decode( wp_remote_retrieve_body( $response ) );
		$this->access_token = isset( $data->access_token ) ? $data->access_token : null;
		$expires = isset( $data->expires_in ) ? $data->expires_in : 600;
		if ( ! empty( $this->access_token ) ) {
			set_transient( $transient, $this->access_token, $expires );
			$this->vbp->info_message( sprintf( esc_html__( 'Saved transient [%s] with access_token [%s] for [%s] seconds', 'video-blogster' ), $transient, $this->access_token, $expires ), 'notice notice-warning', 'debug' );
		}
	}

        /**
         * Make the query and check for errors.
         */
        private function queryApi( $url ) {
		$this->vbp->info_message( sprintf( '%s : %s%s%s', 
			__FUNCTION__, 
			'<a target="_blank" href="' . esc_url( $url ) . '">',
			esc_url( $url ),
			'</a>'
			), 'notice notice-warning', 'debug' );


		$this->getAccessToken();
		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
		if ( empty( $this->access_token ) ) {
			return $this->vbp->info_message( sprintf( esc_html__( '%s Unable to retrieve Spotify access token. Aborting.', 'video-blogster' ), $feedID ) );
		}
		$headers = array(
			'Authorization'		=> 'Bearer ' . $this->access_token
		);
		$this->vbp->info_message( sprintf( '%s : Bearer Token: %s', __FUNCTION__, $this->access_token ), 'notice notice-warning', 'debug' );
		$response = wp_remote_get( $url, array( 'headers' => $headers ) );
                if ( is_wp_error( $response ) ) {
			return $this->vbp->info_message( sprintf( esc_html__( '%s WP error %s in %s(%s) - %s ', 'video-blogster' ), $feedID, $response->get_error_code(), __FUNCTION__, $url, $response->get_error_message() ) );
                }
		else if ( isset( $response['response']['message'] ) && $response['response']['message'] === 'Not Found' ) {
                        return $this->vbp->info_message( sprintf( esc_html__( '%s %s(%s) - Spotify API returned: %s', 'video-blogster' ), $feedID, __FUNCTION__, $url, $response['response']['message'] ) );
		}
                $body = wp_remote_retrieve_body( $response );
                $data = json_decode( $body );

                if ( isset( $data->error ) ) {
                        return $this->vbp->info_message( sprintf( esc_html__( '%s %s(%s) - Spotify API returned: %s', 'video-blogster' ), $feedID, __FUNCTION__, $url, $data->error->message ) );
                }
		return $data;
        }

        /**
         * Takes an array of video details to process and create posts
         */
        private function save_tracks( $items ) {

                if ( empty( $items ) ) {
                        return 0;
                }
		$emptyList = true;

		foreach ( $items as $item ) {
			$videoInfo = $this->get_video_info( $item );

			$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
			if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;

			if ( ! $videoInfo ) continue;

			$emptyList = false;
			$this->vbp->info_message( sprintf( esc_html__( 'Checking Spotify track: [%s] at %s', 'video-blogster' ), $videoInfo['title'], $videoInfo['url'] ), 'notice notice-warning', 'debug' );

			$postID = $this->vbp->save_the_video( $this->query_fields, $videoInfo );

			if ( $postID < 0 ) return 0; // user abort!
			if ( ! $postID ) { $this->num_skipped++; continue; }

			$this->vbp->publish_the_video( $postID, $this->query_fields, $videoInfo );

                        if ( $videoInfo['action'] == 'saved' ) $this->num_imported++;
                        else if ( $videoInfo['action'] == 'updated' ) $this->num_updated++;

			$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
			if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;
		}
		if ( true === $emptyList ) return 0;
		$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
		if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;
		return 1;
	}

        /**
         * Find the best match thumbnail to what the user requested. 
	 * If it is not defined this code will cascade to the closest match.
         */
	private function get_best_thumbnail( $item ) {
		$img = array();
		$match = false;

		if ( isset( $item->images[0]->url ) ) $img[] = $item->images[0]->url;
		if ( $this->query_fields['qExtraParams2'] == 'large' ) $match = true;

		if ( ! $match && isset( $item->images[1]->url ) ) $img[] = $item->images[1]->url;
		if ( $this->query_fields['qExtraParams2'] == 'medium' ) $match = true;

		if ( ! $match && isset( $item->images[2]->url ) ) $img[] = $item->images[2]->url;

		$img = array_reverse( $img );

		$image = $this->vbp->verify_thumbnails( $img );

		return $image;
	}

        /**
         * Extract album,artist,playlist, or track details into our generic videoInfo array
         */
        private function get_video_info( $item ) {
		if ( empty( $item ) ) return null;
		$assocType = $this->query_fields['qAssocType'];
		$videoInfo = array();

		if ( $assocType == 'playlistID' ) {
			$videoInfo['association'] = isset( $item->added_by->id ) ? $item->added_by->id : '';
			$item = isset( $item->track ) ? $item->track : null;
		}

		$videoInfo['videoSource'] = 'Spotify';
		$videoInfo['videoID'] = isset( $item->uri ) ? $item->uri : '';	// spotify:track:0N3UfH89walSn7dwjVUl08
		$videoID = isset( $item->id ) ? $item->id : ''; // 0N3UfH89walSn7d
		$videoInfo['orig_title'] = $videoInfo['title'] = isset( $item->name ) ? $item->name : '';

		// that's enough to check for duplicate post
		if ( $this->vbp->check_post_duplicate( $this->query_fields, $videoInfo ) ) {
			$this->num_skipped++;
			return null;
		}

		$tags = $artists = array();
		if ( isset( $item->album->name ) ) 
			$tags[] = $item->album->name;
		if ( isset( $item->artists ) ) {
			foreach ( $item->artists as $tag ) {
				$artists[] = $tag->name;
				$tags[] = $tag->name;
			}
		}
		$videoInfo['artists'] = implode( ", ", $artists );
		$videoInfo['tags'] = implode( ",", $tags );

		$videoInfo['authorUrl'] = isset( $item->artists[0]->external_urls->spotify ) ? $item->artists[0]->external_urls->spotify : '';
		$videoInfo['authorTitle'] = isset( $item->artists[0]->name ) ? $item->artists[0]->name : '';

		if ( $assocType == 'album' ) {
			$videoInfo['association'] = isset( $item->name ) ? $item->name : '';
			$videoInfo['img'] = $this->get_best_thumbnail( $item );
		}
		else if ( $assocType == 'artist' ) {
			$videoInfo['association'] = isset( $item->name ) ? $item->name : '';
			$videoInfo['img'] = $this->get_best_thumbnail( $item );
		}
		else if ( $assocType == 'playlist' ) {
			$videoInfo['association'] = isset( $item->owner->id ) ? $item->owner->id : '';
			$videoInfo['img'] = $this->get_best_thumbnail( $item );
		}
		else if ( $assocType == 'track' ) {
			$videoInfo['association'] = implode( ", ", $artists );
			$videoInfo['img'] = $this->get_best_thumbnail( $item->album );
		}
		else if ( $assocType == 'show' ) {
			$videoInfo['img'] = $this->get_best_thumbnail( $item );
		}
		else if ( $assocType == 'playlistID' ) { 
			$videoInfo['association'] = isset( $item->album->name ) ? $item->album->name : '';
			$videoInfo['img'] = $this->get_best_thumbnail( $item->album );
		}
		else if ( $assocType == 'showID' ) { 
			$videoInfo['img'] = $this->get_best_thumbnail( $item );
		}
		else if ( $assocType == 'artistID' ) { 
			$videoInfo['association'] = isset( $item->album->name ) ? $item->album->name : '';
			$videoInfo['img'] = $this->get_best_thumbnail( $item->album );
		}
		else {
			$videoInfo['img'] = null; // albumID - no img
		}

		$videoInfo['orig_desc'] = $videoInfo['desc'] = isset( $item->description ) ? $item->description : '';
		$videoInfo['url'] = isset( $item->external_urls->spotify ) ? $item->external_urls->spotify : '';

		$this->vbp->info_message( sprintf( esc_html__( 'Checking Spotify item: [%s] at %s, item: %s', 'video-blogster' ), $videoInfo['title'], $videoInfo['url'], htmlentities( print_r($item,true) ) ), 'notice notice-warning', 'debug' );

		$videoInfo['duration'] = isset( $item->duration_ms ) ? $item->duration_ms : 0; 
		$videoInfo['duration'] = $this->vbp->seconds_to_time( $videoInfo['duration'] / 1000 );
		$videoInfo['viewCount'] = 0;
		$videoInfo['likeCount'] = 0;
		$videoInfo['dislikeCount'] = 0;
		$videoInfo['commentCount'] = 0;

	

		// Make a separate call to Spotify oembed for embed code 
		$url = "https://embed.spotify.com/oembed?url=" . htmlentities( $videoInfo['url'] );
		$videoInfo['videoEmbed'] = $this->vbp->queryoEmbed( $url );

		if ( empty( $videoInfo['videoEmbed'] ) ) {
			$this->vbp->info_message( sprintf( esc_html__( 'Spotify oEmbed call failed. Creating default embed.', 'video-blogster' ), ), 'updated', 'critical' );
			if ( $assocType === 'albumID' ) {
				$embedType = 'embed/track';
			}
			else if ( $assocType === 'artistID' ) {
				$embedType = 'embed/track';
			}
			else if ( $assocType === 'playlistID' ) {
				$embedType = 'embed/track';
			}
			else if ( $assocType === 'showID' ) {
				$embedType = 'embed-podcast/episode';
			}
			else {
				$embedType = 'embed/' . $assocType;
			}
			$videoInfo['videoEmbed'] =
				sprintf( '<iframe src="https://open.spotify.com/%s/%s" width="300" height="380" frameborder="0" allowtransparency="true" allow="encrypted-media"></iframe>', $embedType, $videoID );
		}

		// spotify does not have a published time, except for Show IDs
		if ( isset( $item->release_date ) ) {
			$date = $this->vbp->getDateTime( $item->release_date, $videoInfo['url'] );
		}
		// if not a show then use current time.
		else {
			$date = $this->vbp->getDateTime( null, $videoInfo['url'] );
		}
		$videoInfo['publishedAt'] = $date->format( 'Y-m-d H:i:s' );
		$videoInfo['publishedAt8601'] = isset( $date ) ? $date->format( 'c' ) : '';

		$videoInfo['feedID'] = ! empty ( $this->query_fields['id'] ) ? $this->query_fields['id'] : "";
		$videoInfo = apply_filters( 'vbp_get_video_info', $videoInfo, $item );

		$this->vbp->info_message( sprintf( esc_html__( '%s: %s', 'video-blogster' ), __FUNCTION__, htmlentities( print_r($videoInfo,true) ) ), 'notice notice-warning', 'debug' );

		return $videoInfo;
	}


	public function grab_videos_batch( $url ) {

		$result = $this->queryApi( $url );

		if ( empty( $result ) ) return 0;

		$assocType = $this->query_fields['qAssocType'];

		if ( $assocType === 'album' ) {
			$data = isset( $result->albums ) ? $result->albums : array();
		}
		else if ( $assocType === 'artist' ) {
			$data = isset( $result->artists ) ? $result->artists : array();
		}
		else if ( $assocType === 'playlist' ) {
			$data = isset( $result->playlists ) ? $result->playlists : array();
		}
		else if ( $assocType === 'track' ) {
			$data = isset( $result->tracks ) ? $result->tracks : array();
		}
		else if ( $assocType === 'show' ) {
			// not API returns empty items list. last checked 5/11/2020
			$data = isset( $result->shows ) ? $result->shows : array();
		}
		else if ( $assocType === 'albumID' ) {
			$data = $result;
		}
		else if ( $assocType === 'artistID' ) {
			$data = $result;
		}
		else if ( $assocType === 'playlistID' ) {
			$data = $result;
		}
		else if ( $assocType === 'showID' ) {
			$data = isset( $result->episodes ) ? $result->episodes : array();
			// batches after the first drop the episodes wrapper for some reason...
			$data = empty( $data ) ? $result : $data;
		}
		else if ( $assocType === 'episodeID' ) {
			// episodes do not have anything to embed...
			$data = $result;
		}
		else {
			$data = null; // should not happen
		}

		$items = isset( $data->items ) ? $data->items : array();

		if ( $assocType === 'artistID' ) {	// Spotify is so screwed up
			$items = isset( $data->tracks ) ? $data->tracks : array();
		}
		$next = isset( $data->next ) ? $data->next : null;

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
		if ( ! $this->total ) {
			$this->total = isset( $data->total ) ? $data->total : count( $items );
			if ( $this->total ) {
				$this->vbp->info_message( sprintf( esc_html__( '%s Spotify API returned %s total results.', 'video-blogster' ), $feedID, $this->total ), 'updated', 'video_import' );
			}
		}

                if ( $this->total == 0 && empty( $items ) )  {
			return $this->vbp->info_message( sprintf( esc_html__( '%s No results for Spotify query (%s)', 'video-blogster' ), $feedID, htmlentities( $url ) ), 'notice notice-warning', 'critical' );
                }

		$this->vbp->info_message( sprintf( esc_html__( 'Spotify response data (%s total results): %s', 'video-blogster' ), $this->total, htmlentities( print_r( $items,true ) ) ), 'notice notice-warning', 'debug' );

		if ( ! $this->save_tracks( $items ) ) return 0;

		return $next;
	}

	public function grab_videos() {
		$totalVids = $this->query_fields['qQueryBehavior'] == 'strict' ? $this->query_fields['qNumVideos'] : -1;
		$keyphrase = urlencode( $this->query_fields['qKeyphrase'] );
		$assocType = $this->query_fields['qAssocType'];
		$assoc = $this->query_fields['qAssoc'];
		$filters = ! empty( $this->query_fields['qExtraParams'] ) ? "&" . ltrim( $this->query_fields['qExtraParams'], "&" ) : '';

		if ( false === strpos( $assocType, 'ID' ) && empty( $keyphrase ) ) {
			$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] : "";
			return $this->vbp->info_message( sprintf( esc_html__( '%s Error: No Spotify keyphrase or association not set properly.', 'video-blogster' ), $feedID ), 'error', 'critical' );
		}

		$limit = $totalVids > $this->batch_limit || $totalVids < 0 ? $this->batch_limit : $totalVids;
		if ( $assocType === 'albumID' ) {
			$base_url = "https://api.spotify.com/v1/albums/" . $assoc . "/tracks";
			$base_url .= "?limit={$limit}";
		}
		else if ( $assocType === 'artistID' ) {
			$base_url = "https://api.spotify.com/v1/artists/" . $assoc . "/top-tracks";
			$base_url .= "?limit={$limit}";
		}
		else if ( $assocType === 'playlistID' ) {
			$base_url = "https://api.spotify.com/v1/playlists/" . $assoc . "/tracks";
			$base_url .= "?limit={$limit}";
		}
		else if ( $assocType === 'showID' ) {
			$base_url = "https://api.spotify.com/v1/shows/" . $assoc;
			$base_url .= "?limit={$limit}";
		}
		else if ( $assocType === 'episodeID' ) {
			$base_url = "https://api.spotify.com/v1/episodes/" . $assoc;
			$base_url .= "?limit={$limit}";
		}
		else {	// search
			// Spotify wants raw url encoding
			$keyphrase = urldecode( $keyphrase );
			$base_url = "https://api.spotify.com/v1/search/?q=" . rawurlencode( $keyphrase );
			$base_url .= "&type={$assocType}";
			$base_url .= "&limit={$limit}";
		}

		if ( ! empty( $filters ) ) {
			$base_url .= $filters;
		}

		if ( $assocType === 'artistID' && false === stripos( $base_url, 'country=' ) ) {
			$base_url .= "&country=us";	// default
		}
		if ( $assocType === 'showID' && false === stripos( $base_url, 'market=' ) ) {
			$base_url .= "&market=us";	// default
		}

		$url = apply_filters( 'vbp_spotify_search_query', $base_url, $this->query_fields );

		while ( $url ) {
			$url = $this->grab_videos_batch( $url );
		}
		$this->vbp->import_finished( 'Spotify ' . $assocType . 's', $this->num_skipped, $this->num_updated, $this->num_imported, $this->total, $this->query_fields['id'], $this->query_fields['qQueryBehavior'] );
	}

	/**
	 * Spotify 
	 */

	private function check_for_deleted_batch( $spotIDs, $type ) {
		if ( empty( $spotIDs ) ) return 0;  // shouldn't happen

		$chunks = array_chunk( $spotIDs, $this->batch_limit );
		foreach ( $chunks as $chunk ) {

			foreach ( $chunk as $key => $value ) {
				$list[] = $value;
			}
			$commaList = implode( ',', $list );
			$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: checking Spotify %s IDs: %s ', 'video-blogster' ), $type, $commaList ), 'updated', 'utility_funcs' );

			if ( $type == 'album' ) {
				$url = "https://api.spotify.com/v1/albums/?" . http_build_query( array( 'ids' => $commaList ) );
				$result = $this->queryApi( $url );
				$items = isset( $result->albums ) ? $result->albums : array();
			}
			else if ( $type == 'artist' ) {
				$url = "https://api.spotify.com/v1/artists/?" . http_build_query( array( 'ids' => $commaList ) );
				$result = $this->queryApi( $url );
				$items = isset( $result->artists ) ? $result->artists : array();
			}
			else if ( $type == 'track' ) {
				$url = "https://api.spotify.com/v1/tracks/?" . http_build_query( array( 'ids' => $commaList ) );
				$result = $this->queryApi( $url );
				$items = isset( $result->tracks ) ? $result->tracks : array();
			}
			else {
				continue; // this shouldn't happen
			}
			// returned items are GOOD. remove them from the list.
			foreach ( $items as $item ) {
				if ( isset( $item->id ) && ( $key = array_search( $item->id, $spotIDs ) ) !== FALSE ) {
					unset( $spotIDs[$key] );
				}
			}
		}

		// any remaining IDs means they weren't found by Spotify
		foreach ( $spotIDs as $key => $value ) {
			$key = apply_filters( 'vbp_delete_video', $key );
			if ( ! $key ) continue;
			wp_trash_post( $key );
			$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: Spotify %s ID %s not found. Sending post %s to trash.', 'video-blogster' ), $type, $value, $key ), 'updated', 'critical' );
		}
	}


	public function check_for_deleted_videos() {

		$getMore = true;
		$offset = 0;
		$chunksize = apply_filters( 'vbp_posts_chunksize', 1000 );

		while ( $getMore ) {
		$posts = $this->vbp->get_posts_by_site( 'Spotify', $offset );
		if ( ! $posts ) return;

		$num_results = count( $posts );

		if ( $num_results && $offset == 0 ) {
			$this->vbp->info_message( sprintf( esc_html__( 'Spotify %s Start', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
		}

		$albumIDs = $artistIDs = $trackIDs = array();
		foreach ( $posts as $post_id ) {
			$spotifyID = get_post_meta( $post_id, 'VideoID', TRUE );
			if ( ! $spotifyID ) continue;
			$fields = explode( ':', $spotifyID );	// spotify:[TYPE]:[ID]
			if ( $fields[0] != 'spotify' ) continue;	// should never happen

			if ( $fields[1] == 'album' ) 	$albumIDs[$post_id] 	= $fields[2];
			if ( $fields[1] == 'artist' ) 	$artistIDs[$post_id] 	= $fields[2];
			if ( $fields[1] == 'track' ) 	$trackIDs[$post_id] 	= $fields[2];
			if ( isset( $fields[3] ) && $fields[3] == 'playlist' ) {
				// Spotify requires OAuth for some reason. @#$% that
			}
		} // end foreach posts
		if ( ! empty( $albumIDs ) ) {
			$this->check_for_deleted_batch( $albumIDs, 'album' );
		}
		if ( ! empty( $artistIDs ) ) {
			$this->check_for_deleted_batch( $artistIDs, 'artist' );
		}
		if ( ! empty( $trackIDs ) ) {
			$this->check_for_deleted_batch( $trackIDs, 'track' );
		}
			if ( $num_results < $chunksize ) break;
			$offset += $chunksize;
			$query_args['offset'] = $offset;

		} // end getMore
		if ( $num_results || $offset ) $this->vbp->info_message( sprintf( esc_html__( 'Spotify %s End', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
	}

    } // END class Video_Blogster_Spotify
} // END if ( ! class_exists( 'Video_Blogster_Spotify' ) )


?>
