<?php

defined( 'ABSPATH' ) or die( "Oops! This is a WordPress plugin and should not be called directly.\n" );

/**
 * Class for Video Blogster Pro Vimeo support
 * https://developer.vimeo.com/
 */
if ( ! class_exists( 'Video_Blogster_Vimeo' )) {
    class Video_Blogster_Vimeo {

	private $vbp = null;			// will point to main Video_Blogster instance
        private $query_fields = array();        // options for current query
	private $access_token = null;		// Vimeo unauthorized access token
	private $APIs = array(
		'video.data' 		=> 'https://api.vimeo.com/videos/',
		'videos.list' 		=> 'https://api.vimeo.com/videos',
		'user.list' 		=> 'https://api.vimeo.com/users/',
		'categories.list' 	=> 'https://api.vimeo.com/categories/',
		'channels.list' 	=> 'https://api.vimeo.com/channels/',
		'groups.list' 		=> 'https://api.vimeo.com/groups/',
		'token.authorize' 	=> 'https://api.vimeo.com/oauth/authorize/client',
        );
	private $abort = false;			// if Vimeo rate limits us, abort further requests

	private $batch_limit = 100;             // max amount we can request per query

        private $total = 0;              	// number of total results in query
        private $num_skipped = 0;               // a sum of all videos skipped for this request
        private $num_updated = 0;               // a sum of all videos skipped for this request
        private $num_imported = 0;              // a sum of all videos imported for this request

	/**
	 * Create Vimeo video source
	 * Point back to Video Blogster Pro object to use common functions
	 * Save the query fields for easy access
	 * Requires an app access token
	 */
	public function __construct( $vbp, $query_fields, $access_token ) {
		$this->vbp = $vbp;
		$this->query_fields = $query_fields;
		$this->access_token = $access_token;
		$this->vbp->info_message( sprintf( esc_html__( 'Creating %s resource', 'video-blogster' ), $query_fields[ 'videoSource' ] ), 'notice notice-warning', 'debug' );
	}

	/**
	 * Shortcut to proper API url
	 */
	private function getApi( $type ) {
		return $this->APIs[$type];
	}

	/**
	 * Make the query and check for errors.
	 */
	private function queryApi( $url ) {
		$this->vbp->info_message( sprintf( '%s : %s%s%s', 
			__FUNCTION__, 
			'<a target="_blank" href="' . esc_url( $url ) . '">',
			esc_url( $url ),
			'</a>'
			), 'notice notice-warning', 'debug' );


		$headers = array (
			'Accept' 	=> 'application/vnd.vimeo.*+json; version=3.2',
			'User-Agent' 	=> 'video blogster pro 1.0; (http://developer.vimeo.com/api/docs)',
			'Authorization'	=> 'Bearer ' . $this->access_token
			);

		$response = wp_remote_get( $url, array( 'headers' => $headers ) );

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
		if ( is_wp_error( $response ) ) {
			return $this->vbp->info_message( sprintf( esc_html__( '%s WP error %s in %s(%s) - %s ', 'video-blogster' ), $feedID, $response->get_error_code(), __FUNCTION__, $url, $response->get_error_message() ) );
		}
		$data = json_decode( wp_remote_retrieve_body( $response ) );
		if ( isset( $data->error ) && $data->error != 'The requested video could not be found' ) {
			$this->vbp->info_message( sprintf( esc_html__( '%s(%s) - Vimeo API returned: %s', 'video-blogster' ), __FUNCTION__, $url, $data->error ) );
			if ( $data->error == 'The requested page could not be found' ) { #JPH waiting on Vimeo
				$this->vbp->info_message( sprintf( esc_html__( 'This is likely related to the Vimeo bug of not returning the correct "total" field and an empty "next page" value. Still waiting to hear from them.', 'video-blogster' ) ) );
			}
			$this->vbp->info_message( sprintf( esc_html__( '%s(%s) - Vimeo API data: %s', 'video-blogster' ), __FUNCTION__, $url, print_r($data,true) ), 'notice notice-warning', 'debug' );

			// check for Vimeo RateLimit Ban code
			if ( 429 == wp_remote_retrieve_response_code( $response ) ) {
				$this->abort = true;
				$this->vbp->info_message( sprintf( esc_html__( '%s Vimeo returned code 429 (RateLimit). Aborting further Vimeo API requests...', 'video-blogster' ), $feedID ) );
			}
		}

		$headers = wp_remote_retrieve_headers( $response );
		if ( isset( $headers['x-ratelimit-limit'] ) ) {
			$this->vbp->info_message( sprintf( esc_html__( 'Vimeo API ratelimit %s left out of %s. Next reset at %s. Current time is %s', 'video-blogster' ), $headers['x-ratelimit-remaining'], $headers['x-ratelimit-limit'], $headers['x-ratelimit-reset'], gmdate( DATE_ISO8601) ), 'notice notice-warning', 'debug' );
		}
		if ( isset( $headers['X-Banned-IP'] ) ) {
			$this->vbp->info_message( sprintf( esc_html__( '%s Vimeo API has banned IP %s for too many requests', 'video-blogster' ), $feedID, $headers['X-Banned-IP'] ) );
		}

		return $data;
	}

	private function get_best_thumbnail( $video ) {
		$img = array();

		if ( ! isset( $video->pictures->sizes ) ) {
			$this->vbp->info_message( sprintf( esc_html__( 'Could not find pictures->sizes in video: %s', 'video-blogster' ), $print_r($video,true) ) );
			return null;
		}
		foreach ( $video->pictures->sizes as $pic ) {
			$img[] = $pic->link;
			$width = $pic->width . "px";
			if ( $this->query_fields['qExtraParams2'] == $width ) {
				break;
			}
		}

		$img = array_reverse( $img );

		$image = $this->vbp->verify_thumbnails( $img );

		return $image;
	}

	/**
	 * Extract video details into our generic videoInfo array
	 */
	private function get_video_info( $video ) {
		$videoInfo = array();

		$this->vbp->info_message( sprintf( esc_html__( '%s - video: %s', 'video-blogster' ), __FUNCTION__, htmlentities( print_r($video,true) ) ), 'notice notice-warning', 'debug' );

		$videoInfo['videoSource'] = 'Vimeo';
		$uri = isset( $video->uri ) ? $video->uri : 0;
		if ( preg_match( "/\s*(\d+)$/", $uri, $matches ) ) {
			$videoInfo['videoID'] = $matches[1];
		}
		if ( empty( $videoInfo['videoID'] ) ) {
			$this->num_skipped++;
			return null;
		}
		$videoInfo['orig_title'] = $videoInfo['title'] = isset( $video->name ) ? $video->name : '';

		// that's enough to check for duplicate post
		if ( $this->vbp->check_post_duplicate( $this->query_fields, $videoInfo ) ) {
			$this->num_skipped++;
			return null;
		}

		$videoInfo['orig_desc'] = $videoInfo['desc'] = isset( $video->description ) ? $video->description : '';
		$videoInfo['association'] = isset( $video->user->name ) ? $video->user->name : '';

		$videoInfo['authorUrl'] = isset( $video->user->link ) ? $video->user->link : '';
		$videoInfo['authorTitle'] = isset( $video->user->name ) ? $video->user->name : '';

		$videoInfo['img'] = $this->get_best_thumbnail( $video );
		$videoInfo['url'] = isset( $video->link ) ? $video->link : '';

		$videoInfo['view'] = isset( $video->privacy->view ) ? $video->privacy->view : 'anybody';
		$videoInfo['embed'] = isset( $video->privacy->embed ) ? $video->privacy->embed : 'public';

                // Make a separate call to Vimeo oembed for embed code
                $url = "http://vimeo.com/api/oembed.json?url=" . htmlentities( $videoInfo['url'] );
		$videoInfo['videoEmbed'] = $this->vbp->queryoEmbed( $url );

		$publishedAt = isset( $video->created_time ) ?  $video->created_time : current_time( 'Y-m-d H:i:s' );
		$date = $this->vbp->getDateTime( $publishedAt, $videoInfo['url'] );
                $videoInfo['publishedAt'] = isset( $date ) ? $date->format( 'Y-m-d H:i:s' ) : $publishedAt;
		$videoInfo['publishedAt8601'] = isset( $date ) ? $date->format( 'c' ) : '';

		$d = isset( $video->duration ) ? $video->duration : 0;
		$videoInfo['duration'] = '';
		if ( $d >= 3600 ) {
			$videoInfo['duration'] .= sprintf( '%02d:', ($d/3600) );
		}
		$videoInfo['duration'] .= sprintf('%02d:%02d', ($d/60%60), $d%60);

		foreach ( $video->tags as $tag ) {
			$tags[] = $tag->name;
		}
		$videoInfo['tags'] = ! empty( $tags ) ? implode( ',', $tags ) : '';

		$videoInfo['channel'] = isset( $video->user->name ) ? $video->user->name : '';

		$videoInfo['sourceCategory'] = '';

		$videoInfo['viewCount'] = isset( $video->stats->plays ) ? $video->stats->plays : 0;
		$videoInfo['likeCount'] = isset( $video->metadata->connections->likes->total ) ? $video->metadata->connections->likes->total : 0;

		$videoInfo['comments'] = isset( $video->metadata->connections->comments->total ) ? $video->metadata->connections->comments->total : 0;


		$videoInfo['feedID'] = ! empty ( $this->query_fields['id'] ) ? $this->query_fields['id'] : "";
		$videoInfo = apply_filters( 'vbp_get_video_info', $videoInfo, $video );

		$this->vbp->info_message( sprintf( esc_html__( '%s: %s', 'video-blogster' ), __FUNCTION__, htmlentities( print_r($videoInfo,true) ) ), 'notice notice-warning', 'debug' );

		return $videoInfo;
	}

	/**
	 * Do a batch query search for up to $this->batch_limit comments at a time (Vimeo limit)
	 */
	private function grab_comments_batch( $url, $postID ) {
		$this->vbp->info_message(__FUNCTION__ . ': ' . htmlentities( $url ), 'notice notice-warning', 'debug');
		$response = $this->queryApi( $url );
		$total = 0;
		$comment_date = current_time( 'Y-m-d H:i:s' );
		foreach ( $response->data as $comment ) {
			$author = isset( $comment->user->name ) ? $comment->user->name : '';
			$content = isset( $comment->text ) ? $comment->text : '';
			if ( TRUE == $this->query_fields['cUsePublishedDate'] ) {
				$commentAt = isset( $comment->created_on ) ? $comment->created_on : current_time( 'Y-m-d H:i:s' );
				$comment_date = $this->vbp->getDateTime( $commentAt, $url );
                		$comment_date = isset( $comment_date ) ? $comment_date->format( 'Y-m-d H:i:s' ) : $commentAt;
			}

			$commentdata = array(
				'comment_post_ID'       => $postID,
				'comment_author'        => $author,
				'comment_content'       => $content,
				'comment_date'          => $comment_date,
				'comment_approved'      => 1
			);

			$commentdata = apply_filters( 'vbp_vimeo_comment', $commentdata );
			$comment_id = $this->vbp->save_the_comment( $commentdata, $this->query_fields );
			if ( $comment_id ) {
				$total++;
			}
		}
		$this->vbp->info_message( sprintf( esc_html__( '%s comments imported successfully to post id %s.', 'video-blogster' ), $total, $postID ), 'updated', 'video_import' );

	return isset( $data->paging->next ) ? $data->paging->next : 0;
	}

	/**
	 * Grab and save comments for a video
	 * Toggle comment approval and email notification if set
	 * Break up process into batches
	 */
	private function grab_comments( $videoID, $postID ) {
		$cm = $cn = false;
		$this->vbp->bulk_comment_start( $cm, $cn );		

		$totalComments = $this->query_fields['qNumComments'];
        	$query_args = array(
                	'per_page'	=> $totalComments > $this->batch_limit || $totalComments < 0 ? $this->batch_limit : $totalComments,
                	'direction'	=> 'desc',
			'page'		=> 1,
			'fields'	=> 'user.name,text,created_on'
        	);

		$query_args = apply_filters( 'vbp_vimeo_comments_query', $query_args, $this->query_fields );

		$next = $this->getApi( 'video.data' ) . $videoID . '/comments?' . http_build_query( $query_args );
		while ( $totalComments ) {
			$next = $this->grab_comments_batch( $next, $postID );
			if ( ! $next ) {
				break;	// no more comments
			}
			if ( true == $this->abort ) {
				break;	// hit Vimeo Rate Limit :(
			}
			$totalComments -= $query_args['per_page'];
                	$query_args['per_page'] = $totalComments > $this->batch_limit || $totalComments < 0 ? $this->batch_limit : $totalComments;
		}

		$this->vbp->bulk_comment_end( $cm, $cn );
	}

	/**
	 * Takes an array of video details to process and create posts
	 */
	private function save_videos( $videoDetails ) {

                if ( ! $videoDetails ) {
			return 0;
                }

		foreach ( $videoDetails as $video ) {
			$videoInfo = $this->get_video_info( $video );
			if ( ! $videoInfo ) continue;

			$this->vbp->info_message( sprintf( esc_html__( 'Checking Vimeo video: [%s] at %s', 'video-blogster' ), $videoInfo['title'], $videoInfo['url'] ), 'notice notice-warning', 'debug' );

/* JPH
			if ( isset( $videoInfo['view'] ) && $videoInfo['view'] != 'anybody' ) {
				$this->vbp->info_message( sprintf( esc_html__( 'Vimeo video privacy view set to [%s]. Skipping.', 'video-blogster' ), $videoInfo['view'] ),  'notice notice-warning', 'video_skip' );
				$this->num_skipped++;
				continue;
			}
*/
			if ( isset( $videoInfo['embed'] ) && $videoInfo['embed'] != 'public' ) {
				$this->vbp->info_message( sprintf( esc_html__( 'Vimeo video privacy embed set to [%s]. Skipping.', 'video-blogster' ), $videoInfo['embed'] ),  'notice notice-warning', 'video_skip' );
				$this->num_skipped++;
				continue;
			}

			$postID = $this->vbp->save_the_video( $this->query_fields, $videoInfo );

			if ( $postID < 0 ) return 0; // user abort!
			if ( ! $postID ) { $this->num_skipped++; continue; }

			if ( ! empty( $this->query_fields['qNumComments'] ) && $videoInfo['comments'] ) {
				$this->grab_comments( $videoInfo['videoID'], $postID );
			}

			$this->vbp->publish_the_video( $postID, $this->query_fields, $videoInfo );

                        if ( $videoInfo['action'] == 'saved' ) $this->num_imported++;
                        else if ( $videoInfo['action'] == 'updated' ) $this->num_updated++;

			$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
			if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'], $this->total ) ) return 0;
		}
		$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
		if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'], $this->total ) ) return 0;
		return 1;
	}

	/**
	 * Grab a single video by the video ID stored in qAssoc.
	 */
	private function grab_videos_by_ids() {

		$ids = explode( PHP_EOL, $this->query_fields['qAssoc'] );
		foreach ( $ids as $id ) {
			$url = $this->getApi( 'video.data' ) . $id;
			$response = $this->queryApi( $url );
			if ( ! isset( $response->error ) ) { 
				$this->save_videos( array( $response ) );
			}
		}
		$this->vbp->import_finished( 'Vimeo videos', $this->num_skipped, $this->num_updated, $this->num_imported, $this->total, $this->query_fields['id'], $this->query_fields['qQueryBehavior'] );
	}

	/**
	 * Make a batch request for videos
	 */
	private function grab_videos_batch( $url ) {
		$response = $this->queryApi( $url );

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
		if ( ! $this->total ) {
			$this->total = isset( $response->data ) ? $response->total : 0;
			if ( $this->total ) {
				$this->vbp->info_message( sprintf( esc_html__( '%s Vimeo API returned %s total results.', 'video-blogster' ), $feedID, $this->total ), 'updated', 'video_import' );
			}
		}

		$data = isset( $response->data ) ? $response->data : null;
		$next = isset( $response->paging->next ) ? $response->paging->next : null;
		if ( ! empty( $data ) ) {
			$this->vbp->info_message( sprintf( esc_html__( 'Vimeo response data (%s total results): %s', 'video-blogster' ), $this->total, htmlentities( print_r( $data,true ) ) ), 'notice notice-warning', 'debug' );
			if ( ! $this->save_videos( $data ) ) return 0;
		}
		else {
			return $this->vbp->info_message( sprintf( esc_html__( '%s No results returned for given query. Full response: %s', 'video-blogster' ), $feedID, htmlentities( print_r($response,true) ) ), 'notice notice-warning', 'critical' );
		}
		return $next;
	}

	/**
	 * Process the video query and break down into batches
	 * https://developer.vimeo.com/api/endpoints/videos#
	 */
	public function grab_videos() {
		$totalVids = $this->query_fields['qQueryBehavior'] == 'strict' ? $this->query_fields['qNumVideos'] : -1;
		$keyphrase = $this->query_fields['qKeyphrase'];
		$assoc = $this->query_fields['qAssoc'];
		$assocType = $this->query_fields['qAssocType'];
		$sort = $this->query_fields['qOrderBy'];
		$direction = $this->query_fields['qOrderDirection'];
		$filters = ! empty( $this->query_fields['qExtraParams'] ) ? "&" . ltrim( $this->query_fields['qExtraParams'], "&" ) : '';
/* Examples:
DOES NOT WORK
filter=minimum_likes&filter_minimum_likes=10
filter=categories&filter_categories=northamerica
SEEMS TO WORK
filter=content_rating&filter_content_rating=safe,unrated
filter=embeddable&filter_embeddable=true
SEE ALSO
https://developer.vimeo.com/api/common-formats
*/

		$embeddable = $filters ? null : 'embeddable';// any user filters overrides default embeddable filters

		if ( empty( $keyphrase ) && ( empty( $assoc ) || empty( $assocType ) ) ) {
			$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] : "";
                        return $this->vbp->info_message( sprintf( esc_html__( '%s Error: No Vimeo keyphrase or association not set properly.', 'video-blogster' ), $feedID ), 'error', 'critical' );
                }

		if ( $assoc && $assocType == 'category' ) {
			$base_url = $this->getApi( 'categories.list' ) . $assoc . '/videos';
			$keyphrase = $sort = $direction = $embeddable = null; // not allowed
		}
		else if ( $assoc && $assocType == 'channel' ) {
			$base_url = $this->getApi( 'channels.list' ) . $assoc . '/videos';
		}
		else if ( $assoc && $assocType == 'group' ) {
			$base_url = $this->getApi( 'groups.list' ) . $assoc . '/videos';
		}
		else if ( $assoc && $assocType == 'user' ) {
			$base_url = $this->getApi( 'user.list' ) . $assoc . '/videos';
		}
		else if ( $assoc && $assocType == 'video' ) {
			return $this->grab_videos_by_ids();
		}
		else {
			$base_url = $this->getApi( 'videos.list' );
		}

		$query_args = array(
			'per_page'		=> $totalVids > $this->batch_limit || $totalVids < 0 ? $this->batch_limit : $totalVids,
			'page'			=> 1,
			'fields'		=> "name,description,user.name,pictures.sizes.link,pictures.sizes.width,link,uri,created_time,duration,stats.plays,metadata.connections.likes.total,metadata.connections.comments.total"
		);
		if ( $sort ) {
			$query_args['sort'] = $sort;
		}
		if ( $direction && $sort != 'relevant' ) {
			$query_args['direction'] = $direction;
		}
		if ( $embeddable ) {
			$query_args['filter'] = 'embeddable';
			$query_args['filter_embeddable'] = 'true';
		}
		$query_args['fields'] .= ",tags.name";

		if ( $keyphrase ) {
			$query_args['query'] = $keyphrase;
		}

		$query_args = apply_filters( 'vbp_vimeo_search_query', $query_args, $this->query_fields  );

		$url = $base_url . '?' . http_build_query( $query_args );
		if ( $filters) $url .= $filters;
		while ( $url ) {
			$url = $this->grab_videos_batch( $url );
			if ( $url ) $url = 'https://api.vimeo.com' . $url;

			if ( true == $this->abort ) {
				break;	// hit Vimeo Rate Limit :(
			}
		}

		$this->vbp->import_finished( 'Vimeo videos', $this->num_skipped, $this->num_updated, $this->num_imported, $this->total, $this->query_fields['id'], $this->query_fields['qQueryBehavior'] );
		return 1;
	}

	/**
	 * Checks video ids to see if they're been deleted from Vimeo
	 * No batch checking allowed - have to query each video individually :(
	 */
	public function check_for_deleted_videos() {
		// get comma delimited videoIDs from post metadata
		$getMore = true;
		$offset = 0;
		$chunksize = apply_filters( 'vbp_posts_chunksize', 1000 );

		while ( $getMore ) {
		$posts = $this->vbp->get_posts_by_site( 'Vimeo', $offset );
		if ( ! $posts ) return;

		$num_results = count( $posts );

		if ( $num_results && $offset == 0 ) { 
			$this->vbp->info_message( sprintf( esc_html__( 'Vimeo %s Start', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
		}

		foreach ( $posts as $post_id ) {
			$videoID = get_post_meta( $post_id, 'VideoID', TRUE );
			if ( ! $videoID ) {
				continue;
			}
			$url = get_post_meta( $post_id, 'VideoUrl', TRUE );
			$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: checking Vimeo video ID %s, url: %s', 'video-blogster' ), $videoID, $url ), 'updated', 'utility_funcs' );
			$url = $this->getApi( 'video.data' ) . $videoID;
			$data = $this->queryApi( $url );
			if ( true == $this->abort ) {
				break;	// hit Vimeo Rate Limit :(
			}
			if ( isset( $data->error ) && $data->error == 'The requested video could not be found' ) {
				$postID = apply_filters( 'vbp_delete_video', $post_id );
				if ( ! $postID ) {
					continue;
				}
				wp_trash_post( $postID );
				$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: Vimeo video %s not found. Sending post %s to trash.', 'video-blogster' ), $url , $postID ), 'updated', 'critical' );
			}
		} // end foreach posts
			if ( true == $this->abort ) {
				break;	// hit Vimeo Rate Limit :(
			}
			if ( $num_results < $chunksize ) break;
			$offset += $chunksize;
			$query_args['offset'] = $offset;
		} // end getMore
		if ( $num_results || $offset ) $this->vbp->info_message( sprintf( esc_html__( 'Vimeo %s End', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
	}

    } // END class Video_Blogster_Vimeo
} // END if ( ! class_exists( 'Video_Blogster_Vimeo' ) )
?>
