<?php

defined( 'ABSPATH' ) or die( "Oops! This is a WordPress plugin and should not be called directly.\n" );

/**
 * Class for Video Blogster Pro YouTube support
 */
if ( ! class_exists( 'Video_Blogster_YouTube' )) {
    class Video_Blogster_YouTube {

	private $vbp=0;	// will point to main Video_Blogster instance
	private $apiKey=0;
	private $youtube_categories=0;		// fetch once per instance only when needed
	private $youtube_regions=0;		// fetch once per instance only when needed
	private $query_fields = array();	// options for current query
    	private $APIs = array(
		'videos.list' 		=> 'https://www.googleapis.com/youtube/v3/videos',
		'search.list' 		=> 'https://www.googleapis.com/youtube/v3/search',
		'channels.list' 	=> 'https://www.googleapis.com/youtube/v3/channels',
		'playlists.list' 	=> 'https://www.googleapis.com/youtube/v3/playlists',
		'playlistItems.list' 	=> 'https://www.googleapis.com/youtube/v3/playlistItems',
		'categories.list' 	=> 'https://www.googleapis.com/youtube/v3/videoCategories',
		'regions.list' 		=> 'https://www.googleapis.com/youtube/v3/i18nRegions',
		'commentThreads.list' 	=> 'https://www.googleapis.com/youtube/v3/commentThreads'
	);
	private $batch_limit = 50;             // max amount of videos we can request per API query
	private $comments_limit = 100;         // max amount of comments we can request per API query

	private $total = 0;			// total number of results
	private $num_skipped = 0;		// a sum of all videos skipped for this request
	private $num_updated = 0;		// a sum of all videos updated for this request
	private $num_imported = 0;		// a sum of all videos imported for this request

	private $metadata = false;		// if verifying metadata every 30 days

	/**
	 * Create YouTube video source
	 * Point back to Video Blogster Pro object to use common functions
	 * Save the query fields for easy access
	 * Requires app API key
	 */
        public function __construct( $vbp, $query_fields, $key ) {
		$this->vbp = $vbp;
		$this->query_fields = $query_fields;
		$this->apiKey = $key;
		$this->vbp->info_message( sprintf( esc_html__( 'Creating %s resource', 'video-blogster' ), $query_fields[ 'videoSource' ] ), 'notice notice-warning', 'debug' );
	}

	/**
	 * Shortcut to proper API url
	 */
	private function getApi( $type ) {
		return $this->APIs[$type];
	}

	/**
	 * Make the query and check for errors.
	 */
	private function queryApi( $url, $json = true ) {
		$this->vbp->info_message( sprintf( '%s : %s%s%s', 
			__FUNCTION__, 
			'<a target="_blank" href="' . esc_url( $url ) . '">',
			esc_url( $url ),
			'</a>'
			), 'notice notice-warning', 'debug' );

                $response = wp_remote_get( $url );

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
    		if ( is_wp_error( $response ) ) {
			return $this->vbp->info_message( sprintf( esc_html__( '%s WP error %s in %s(%s) - %s ', 'video-blogster' ), $feedID, $response->get_error_code(), __FUNCTION__, $url, $response->get_error_message() ) );
		}
                $data = $json ? json_decode( wp_remote_retrieve_body( $response ) ) : wp_remote_retrieve_body( $response );
		if ( isset( $data->error ) ) {
#JPH should I check quota?
			return $this->vbp->info_message( sprintf( esc_html__( '%s %s (%s) - YouTube API returned %s', 'video-blogster' ), $feedID, __FUNCTION__, htmlentities( $url ), $data->error->message ) );
		}
		return $data;
	}

	/**
	 * Query YouTube for existing categories
	 */
	public function grab_categories( $region = 'US' ) {
		if ( $this->youtube_categories )	{
			return $this->youtube_categories;
		}

		// if feed is using a regionCode, use that to determine available categories instead:
		$region = ! empty( $this->query_fields['qRegionCode'] ) ? $this->query_fields['qRegionCode'] : $region;
		$this->youtube_categories = $this->vbp->transientChecker( 'vb_youtube_categories_' . $region );
		if ( ! empty( $this->youtube_categories ) ) return $this->youtube_categories;

        	$query_args = array(
                	'part'		=> 'snippet',
                	'regionCode'	=> $region,
                	'key'		=> $this->apiKey
        	);
		$url = $this->getApi( 'categories.list' ) . '?' . http_build_query( $query_args );
		$data = $this->queryApi( $url );
		$this->youtube_categories = isset( $data->items ) ? $data->items : array();
		if ( ! empty( $this->youtube_categories ) ) {
			usort($this->youtube_categories, function($a, $b) {return strcmp($a->snippet->title, $b->snippet->title);});
			$this->vbp->transientSave( 'vb_youtube_categories_' . $region, $this->youtube_categories );
		}

		return $this->youtube_categories;
	}

	/**
	 * Query YouTube for i18nRegions.list
	 */
	public function grab_regions() {
		if ( $this->youtube_regions )	{
			return $this->youtube_regions;
		}

		$this->youtube_regions = $this->vbp->transientChecker( 'vb_youtube_regions' );
		if ( ! empty( $this->youtube_regions ) ) return $this->youtube_regions;
		
        	$query_args = array(
                	'part'		=> 'snippet',
                	'key'		=> $this->apiKey
        	);
		$url = $this->getApi( 'regions.list' ) . '?' . http_build_query( $query_args );
		$data = $this->queryApi($url);
		$this->youtube_regions = isset( $data->items ) ? $data->items : array();
		if ( ! empty( $this->youtube_regions ) ) {
			usort($this->youtube_regions, function($a, $b) {return strcmp($a->snippet->name, $b->snippet->name);});
			$this->vbp->transientSave( 'vb_youtube_regions', $this->youtube_regions );
		}
		return $this->youtube_regions;
	}


	/**
	 * Do a batch query request for up to $this->comments_limit comments at a time (YouTube limit)
	 */
	private function grab_comments_batch( $url, $postID ) {
		$this->vbp->info_message(__FUNCTION__ . ': ' . htmlentities( $url ), 'notice notice-warning', 'debug' );
		$data = $this->queryApi( $url );
		if ( ! $data ) { // error occurred - probably timeout. Try again another time
			return $this->vbp->info_message( esc_html__( 'Comment Query Error: Skipping this batch.', 'video-blogster' ) );
		}
		$items = isset( $data->items ) ? $data->items : array();

		$total = $ctotal = 0;
		$comment_date = current_time( 'Y-m-d H:i:s' );
		foreach ( $items as $item )  {
			$ctotal++;
			$topLevelComment = $item->snippet->topLevelComment;
			$author = isset( $topLevelComment->snippet->authorDisplayName ) ? trim( $topLevelComment->snippet->authorDisplayName ) : '';
			$content = isset( $topLevelComment->snippet->textDisplay ) ? trim( $topLevelComment->snippet->textDisplay ) : '';
			$parentId = isset( $topLevelComment->id ) ? trim( $topLevelComment->id ) : '';
			if ( empty( $author ) || empty( $content ) ) {
				continue;
			}
			if ( TRUE == $this->query_fields['cUsePublishedDate'] ) {
				$commentAt = isset( $topLevelComment->snippet->publishedAt ) ? $topLevelComment->snippet->publishedAt : current_time( 'Y-m-d H:i:s' );
				$commentDate = $this->vbp->getDateTime( $commentAt, $url );
				$comment_date = isset( $commentDate ) ? $commentDate->format('Y-m-d H:i:s') : $commentAt;
			}
			$commentdata = array(
				'comment_post_ID'       => $postID,
				'comment_author'        => $author,
				'comment_content'       => $content,
				'comment_date'          => $comment_date,
				'comment_approved'      => 1
			);
			$commentdata = apply_filters( 'vbp_youtube_comment', $commentdata );
			if ( empty( $commentdata ) ) continue;
			$comment_id = $this->vbp->save_the_comment( $commentdata, $this->query_fields );
			if ( $comment_id ) {
				if ( $comment_id > 0 ) $total++;	// the id of a new comment
				else $comment_id = abs( $comment_id ); 	// the id of an existing comment
			}
			$comment_parent = $comment_id;

			// any replies?
			$this->vbp->info_message( sprintf( esc_html__( ' ==> Comment total reply count: %d', 'video-blogster' ), $item->snippet->totalReplyCount ), 'notice notice-warning', 'debug' );
			if ( isset( $item->snippet->totalReplyCount ) && ! $item->snippet->totalReplyCount ) continue;	

			// any replies to this top level comment?
			$replies = isset( $item->replies->comments ) ? $item->replies->comments : array();
			$this->vbp->info_message( sprintf( esc_html__( ' ==> Comment item replies count: %d', 'video-blogster' ), count( $replies ) ), 'notice notice-warning', 'debug' );
			foreach ( $replies as $reply )  {
				$ctotal++;
				$author = isset( $reply->snippet->authorDisplayName ) ? trim( $reply->snippet->authorDisplayName ) : '';
				$content = isset( $reply->snippet->textDisplay ) ? trim( $reply->snippet->textDisplay ) : '';
				if ( empty( $author ) || empty( $content ) ) {
					continue;
				}
				if ( TRUE == $this->query_fields['cUsePublishedDate'] ) {
					$commentAt = isset( $reply->snippet->publishedAt ) ? $reply->snippet->publishedAt : current_time( 'Y-m-d H:i:s' );
					$comment_date = $this->vbp->getDateTime( $commentAt, $url );
					$comment_date = isset( $comment_date ) ? $comment_date->format('Y-m-d H:i:s') : $commentAt;
				}
				$commentdata = array(
					'comment_post_ID'       => $postID,
					'comment_author'        => $author,
					'comment_content'       => $content,
					'comment_parent'	=> $comment_parent,
					'comment_date'          => $comment_date,
					'comment_approved'      => 1
				);
				$commentdata = apply_filters( 'vbp_youtube_comment', $commentdata );
				if ( empty( $commentdata ) ) continue;
				$comment_id = $this->vbp->save_the_comment( $commentdata, $this->query_fields );
				if ( $comment_id ) {
					if ( $comment_id > 0 ) $total++;	// the id of a new comment
					else $comment_id = abs( $comment_id ); 	// the id of an existing comment
				}
			}
		}
		$this->vbp->info_message( sprintf( esc_html__( '%s out of %s total comments imported successfully to post id %s.', 'video-blogster' ), $total, $ctotal, $postID ), 'updated', 'video_import' );

		return isset( $data->nextPageToken ) ? $data->nextPageToken : '';
	}

	/**
	 * Grab and save comments for a video
	 */
	private function grab_comments( $videoID, $postID ) {
		$this->vbp->info_message( sprintf( esc_html__( 'Importing comments for video ID %s, post ID %s', 'video-blogster' ), $videoID, $postID ), 'notice notice-warning', 'debug' );
		$cm = $cn = false;
		$this->vbp->bulk_comment_start( $cm, $cn );

		$totalComments = $this->query_fields['qNumComments'];
		$part = 'snippet';
		if ( true == $this->query_fields['qCommentReplies'] ) {
			$part .= ', replies';
		}
        	$query_args = array(
                	'part'		=> $part,
                	'maxResults'	=> $totalComments > $this->comments_limit || $totalComments < 0 ? $this->comments_limit : $totalComments,
                	'videoId'	=> $videoID,
                	'order'		=> $this->query_fields['qCommentOrderBy'],
                	'textFormat'	=> 'plainText',
                	'key'		=> $this->apiKey
        	);

		if ( ! empty( $this->query_fields['qCommentSearchTerms'] ) ) {
			$query_args['searchTerms'] = $this->query_fields['qCommentSearchTerms'];
		}

		$query_args = apply_filters( 'vbp_youtube_comments_query', $query_args, $this->query_fields );

		while ( $totalComments ) {
			$request = $this->getApi( 'commentThreads.list' ) . '?' . http_build_query( $query_args );
			$next = $this->grab_comments_batch( $request, $postID );
			if ( ! $next ) {
				break;	// no more comments
			}
			$query_args['pageToken'] = $next;
			$totalComments -= $query_args['maxResults'];
                	$query_args['maxResults'] = $totalComments > $this->comments_limit || $totalComments < 0 ? $this->comments_limit : $totalComments;
		}

		$this->vbp->bulk_comment_end( $cm, $cn );
	}


	/**
	 * Checks a batch of video ids to see if they're been deleted from YouTube
	 * called in batches of $this->batch_limit (YouTube limit). Query result list without id means that video has been deleted.
	 * Query result list without id means that video has been deleted.
	 */
	private function check_for_deleted_batch( $videoIDs ) {
		if ( empty( $videoIDs ) ) {
			return 0;
		}

		foreach ( $videoIDs as $key => $value ) {
			$list[] = $value;
		}
		$commaList = implode( ',', $list );

		$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: checking YouTube video IDs: %s ', 'video-blogster' ), $commaList ), 'updated', 'utility_funcs' );

        	$query_args = array(
                	'part'		=> 'id,status',
                	'id'		=> $commaList,
                	'key'		=> $this->apiKey
        	);
		$url = $this->getApi( 'videos.list' ) . '?' . http_build_query( $query_args );
		$data = $this->queryApi( $url );

		if ( ! $data ) { // error occurred - probably timeout. Try again another time
			return $this->vbp->info_message( esc_html__( 'DelCheck Query Error, no data returned from API: Skipping this batch.', 'video-blogster' ) );
		}
		$videoDetails = isset( $data->items ) ? $data->items : array();

		// remove each ID found from query from our original list of IDs to check...
		foreach ( $videoDetails as $video )  {

			// check if video is still valid on YouTube
			// if so, remove from the GOOD list, leaving only ID's of videos that WILL be trashed
			if ( ( $key = array_search( $video->id, $videoIDs ) ) !== FALSE && ( $video->status->uploadStatus == 'processed' || $video->status->uploadStatus == 'uploaded' ) ) {
				unset( $videoIDs[$key] );
			}
		}

		// any remaining IDs in our list means they weren't found on YouTube
		foreach ( $videoIDs as $key => $value ) {
			$key = apply_filters( 'vbp_delete_video', $key );
			if ( ! $key ) {
				continue;
			}
			wp_trash_post( $key );
			$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: YouTube video %s not found. Sending post %s to trash.', 'video-blogster' ), $value, $key ), 'updated', 'critical' );
		}
		return 0;
	}

	/**
	 * Creates batches of YouTube ids from save posts to check
	 */
	public function check_for_deleted_videos() {
		// get comma delimited videoIDs from post metadata
		$getMore = true;
		$offset = 0;
		$chunksize = apply_filters( 'vbp_posts_chunksize', 1000 );

		while ( $getMore ) {
		$posts = $this->vbp->get_posts_by_site( 'YouTube', $offset );
		if ( ! $posts ) return;

		$num_results = count( $posts );

		if ( $num_results && $offset == 0 ) {
			$this->vbp->info_message( sprintf( esc_html__( 'YouTube %s Start', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
		}

		$num=0;
		foreach ( $posts as $post_id ) {
			if ( ! $num ) {
				// use this to reset array after each batch
				$videoIDs = array();
			}
			$videoID = get_post_meta( $post_id, 'VideoID', TRUE );
			if ( ! $videoID ) {     // mvb compatability check:
                                $videoID = get_post_meta( $post_id, 'mvb_vid_code', TRUE );
                        }
			if ( ! $videoID ) {
				continue;
			}
			$videoIDs[$post_id] = $videoID;
			if ( ++$num >= $this->batch_limit ) {
				$num = $this->check_for_deleted_batch( $videoIDs );
			}
		} // end foreach posts
			if ( $num_results < $chunksize ) break;
			$offset += $chunksize;
			$query_args['offset'] = $offset;
		} // end getMore
		if ( $num ) {
			$this->check_for_deleted_batch( $videoIDs );	// still ids some left to check
		}
		if ( $num_results || $offset ) $this->vbp->info_message( sprintf( esc_html__( 'YouTube %s End', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
	}

	/**
	 * Update video metadata for each video after 30 days
	 */
	public function update_metadata() {

		$cutoff = date( "Y-m-d", strtotime( "-30 days", strtotime( "now" ) ) );
		$meta = array(
			'relation'		=> 'OR',
			array(
				'key'		=> 'VideoLastChecked',
				'compare'	=> 'NOT EXISTS'
			),
			array(
				'key'		=> 'VideoLastChecked',
				'compare'	=> '<=',
				'type'		=> 'DATE',
				'value'		=> $cutoff
			),
		);

		$this->metadata = true;
		$this->query_fields['qAssocType'] = 'video';
		$this->query_fields['qNumVideos'] = 50;
		$this->query_fields['qQueryBehavior'] = 'strict';
		$this->query_fields['pUpdateExisting'] = true;
		$this->query_fields['cPostStatus'] = 'inherit';

		$this->vbp->info_message( sprintf( esc_html__( '%s start check with args: %s', 'video-blogster' ), __FUNCTION__, print_r($meta,true) ), 'notice notice-warning', 'debug' );

		$postIDs = $this->vbp->get_posts_by_site( 'YouTube', 0, $meta );

		$num = 0;

		// get the videoIDs from these posts and check for meta updates
		if ( ! empty( $postIDs ) ) {

			$videoIDs = array();
			foreach ( $postIDs as $postID ) {
				$videoID = get_post_meta( $postID, 'VideoID', TRUE );
				if ( $videoID ) {
					$videoIDs[$postID] = $videoID;
				}
			}

			$this->vbp->info_message( sprintf( esc_html__( 'Updating YouTube metadata for post/video IDs: %s', 'video-blogster' ), print_r( $videoIDs, true ) ), 'notice notice-warning', 'critical' );
 
			$this->query_fields['qAssoc'] = $videoIDs;
			$this->grab_videos_by_ids();
			$num = count( $videoIDs );
		}

		$yt_meta_check_time = $this->vbp->transientChecker( 'vbp_youtube_metadata', 60 );

		if ( $num >= 50 )
			$yt_meta_check_time /= 2;
		else
			$yt_meta_check_time *= 2;

		if ( $yt_meta_check_time < 60 )
			$yt_meta_check_time = 60;
		else if ( $yt_meta_check_time > DAY_IN_SECONDS )
			$yt_meta_check_time = DAY_IN_SECONDS;

		$yt_meta_check_time = apply_filters( 'vbp_youtube_metadata_checkfreq', $yt_meta_check_time );

		$this->vbp->transientSave( 'vbp_youtube_metadata', $yt_meta_check_time, DAY_IN_SECONDS );
		$this->vbp->info_message( sprintf( esc_html__( '%s finished.', 'video-blogster' ), __FUNCTION__, ), 'notice notice-warning', 'debug' );
	}

	/**
	 * Have to make a separate query for the video details and stats after a search query
	 * accepts a search result from query and/or a specific videoID
	 * https://developers.google.com/youtube/v3/docs/videos/list
	 */
	private function grab_videolist_details( $videos, $videoIDs='' ) {
		$this->vbp->info_message( sprintf( esc_html__( 'Getting video details from results... %s', 'video-blogster' ), $videoIDs ), 'notice notice-warning', 'debug' );
		$base_url = $this->getApi( 'videos.list' );
		if ( ! empty( $videoIDs ) ) {
			$commaList = $videoIDs;
		}
		else if ( ! empty( $videos ) ) {
			$ids = array();
			foreach ( $videos as $video )  {
				if ( isset( $video->id->videoId ) ) {	// a search result
					$ids[] = $video->id->videoId;
				}
				else if ( isset( $video->id ) ) {	// a video result
					$ids[] = $video->id;
				}
			}
			$commaList = implode( ',', $ids );
		}
		else { // empty search query result
			return 0;
		}

                $parts = 'snippet,contentDetails,statistics';

		if ( true === $this->metadata )
			$parts .= ',status';

		$filters = ! empty( $this->query_fields['qExtraParams'] ) ? "&" . ltrim( $this->query_fields['qExtraParams'], "&" ) : '';
		if ( false !== stripos( $filters, 'eventType=live' ) )
			$parts .= ',liveStreamingDetails';


		// if using a chart, we need to verify video is embeddable with status check
		if ( $this->query_fields['qAssocType'] === 'mostPopular' )
			$parts .= ',status';

                $parts = apply_filters( 'vbp_youtube_video_parts', $parts );

        	$query_args = array(
                	'part'		=> $parts,
                	'id'		=> $commaList,
                	'key'		=> $this->apiKey
        	);
		$url = $base_url . '?' . http_build_query( $query_args );
		$data = $this->queryApi( $url );
		$videoDetails = isset( $data->items ) ? $data->items : 0;
		if ( ! $videoDetails ) {
			$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] : "";
			$this->vbp->info_message( sprintf( esc_html__( '%s YouTube (%s) unable to get videoDetails. Full response was: %s', 'video-blogster' ), $feedID, $url, htmlentities( print_r( $data,true ) ) ) );
		}
		return $videoDetails;
	}

	private function get_best_thumbnail( $video ) {
		$img = array();
		$match = false;

                $sizeRequest = $this->query_fields['qExtraParams2'];
		#$hd = isset( $video->contentDetails->definition ) ? $video->contentDetails->definition : '';

		if ( $sizeRequest == 'largest' ) {
			// default if not HD:
			if ( isset( $video->snippet->thumbnails->high ) ) $img[] = $video->snippet->thumbnails->high->url;
			// HD videos should have one of these:
			// 640x480, ratio 4:3
			if ( isset( $video->snippet->thumbnails->standard ) ) $img[] = $video->snippet->thumbnails->standard->url;
			// 1280x720, ratio 16:9
			if ( isset( $video->snippet->thumbnails->maxres ) ) $img[] = $video->snippet->thumbnails->maxres->url;
			else if ( isset( $video->snippet->thumbnails->default ) ) {
				// fake a maxres entry. should be there.
				$img[] = str_replace( 'default', 'maxresdefault', $video->snippet->thumbnails->default->url );
			}
		}
		else if ( in_array( $sizeRequest, array( 'default', 'high', 'standard' ) ) ) {
			// look for best 4:3 ratio thumb

			// 120x90, ratio 4:3
			if ( isset( $video->snippet->thumbnails->default ) ) $img[] = $video->snippet->thumbnails->default->url;
			if ( $sizeRequest == 'default' ) $match = true;

			// 480x360, ratio 4:3
			if ( ! $match && isset( $video->snippet->thumbnails->high ) ) $img[] = $video->snippet->thumbnails->high->url;
			if ( $sizeRequest == 'high' ) $match = true;

			// 640x480, ratio 4:3
			if ( ! $match && isset( $video->snippet->thumbnails->standard ) ) $img[] = $video->snippet->thumbnails->standard->url;
			if ( $sizeRequest == 'standard' ) $match = true;
		}
		else {
			// look for best 16:9 ratio thumb

			// 320x180, ratio 16:9
			if ( ! $match && isset( $video->snippet->thumbnails->medium ) ) $img[] = $video->snippet->thumbnails->medium->url;
			if ( $sizeRequest == 'medium' ) $match = true;
			error_log("JPH here - sizeRequest is {$sizeRequest}, match is {$match}");

			// 1280x720, ratio 16:9
			if ( ! $match && isset( $video->snippet->thumbnails->maxres ) ) $img[] = $video->snippet->thumbnails->maxres->url;
			else if ( ! $match && isset( $video->snippet->thumbnails->default ) ) {
				// fake a maxres entry. should be there.
				$img[] = str_replace( 'default', 'maxresdefault', $video->snippet->thumbnails->default->url );
			}
		}

		$img = array_reverse( $img );

		if ( empty( $img ) ) {
                        return $this->vbp->info_message( sprintf( esc_html__( '%s Error: no img match for %s.', 'video-blogster' ), __FUNCTION__, $sizeRequest ), 'error', 'critical' );
		}

		$img = apply_filters( 'vbp_youtube_thumbnail_array', $img );

		$image = $this->vbp->verify_thumbnails( $img );

		return $image;
	}

	/**
	 * Extract video details into our generic videoInfo array
	 */
	private function get_video_info( $video ) {
		$videoInfo = array();

		// snippet info: ===================================================

		$videoInfo['videoSource'] = 'YouTube';
		$videoInfo['videoID'] = isset( $video->id ) ? $video->id : 0;
		$videoInfo['orig_title'] = $videoInfo['title'] = isset( $video->snippet->title ) ? $video->snippet->title : '';

		// that's enough to check for duplicate post
		if ( $this->vbp->check_post_duplicate( $this->query_fields, $videoInfo ) ) {
			$this->num_skipped++;
			return null;
		}
		// mostPopular chart - requested status because chart returns videos not embeddable off YouTube
		// playlists - requested status because playlist returns videos not embeddable off YouTube
		if ( isset( $video->status->embeddable ) && true !== $video->status->embeddable ) {
			$this->num_skipped++;
			return null;
		}

		$videoInfo['orig_desc'] = $videoInfo['desc'] = isset( $video->snippet->description ) ? $video->snippet->description : '';
		$videoInfo['channel'] = isset( $video->snippet->channelTitle ) ? $video->snippet->channelTitle : '';
		$videoInfo['channelID'] = isset( $video->snippet->channelId ) ? $video->snippet->channelId : '';
		$videoInfo['authorUrl'] = isset( $video->snippet->channelId ) ? 'https://www.youtube.com/channel/' . $video->snippet->channelId : '';
		$videoInfo['authorTitle'] = isset( $video->snippet->channelTitle ) ? $video->snippet->channelTitle : '';
		$videoInfo['association'] = $videoInfo['channel'];
		$videoInfo['categoryID'] = isset( $video->snippet->categoryId ) ? $video->snippet->categoryId : 0;
		if ( isset( $video->snippet->playlistId ) ) {	// this is from a playlist
			$videoInfo['videoID'] = $video->snippet->resourceId->videoId;
		}
		$videoInfo['url'] = 'https://www.youtube.com/watch?v=' . $videoInfo['videoID'];

                // Make a separate call to YouTube oembed for embed code
                $url = "https://www.youtube.com/oembed?url=" . htmlentities( $videoInfo['url'] ) . "&format=json";
		if ( false === $this->metadata )
			$videoInfo['videoEmbed'] = $this->vbp->queryoEmbed( $url );

		// if oEmbed call fails, try to create embed from scratch
		if ( empty( $videoInfo['videoEmbed'] ) ) {
			$videoInfo['videoEmbed'] = '<iframe width="560" height="315" src="https://www.youtube.com/embed/' . $videoInfo['videoID'] . '" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe>';
		}

		$publishedAt = isset( $video->snippet->publishedAt ) ? $video->snippet->publishedAt : current_time( 'Y-m-d H:i:s' );
		$date = $this->vbp->getDateTime( $publishedAt, $videoInfo['url'] );
		$videoInfo['publishedAt'] = isset( $date ) ? $date->format( 'Y-m-d H:i:s' ) : $publishedAt;
		$videoInfo['publishedAt8601'] = isset( $date ) ? $date->format( 'c' ) : '';

		if ( isset( $video->liveStreamingDetails->scheduledStartTime ) ) {
			$scheduledAt = $video->liveStreamingDetails->scheduledStartTime;
			$date = $this->vbp->getDateTime( $scheduledAt, $videoInfo['url'] );
			$videoInfo['scheduledAt'] = isset( $date ) ? $date->format( 'Y-m-d H:i:s' ) : null;
		}
		else if ( isset( $video->liveStreamingDetails ) ) {
			$videoInfo['scheduledAt'] = null;
		}
		if ( isset( $video->liveStreamingDetails->concurrentViewers ) ) 
			$videoInfo['concurrentViewers'] = $video->liveStreamingDetails->concurrentViewers;
		else
			$videoInfo['concurrentViewers'] = 0;

		$videoInfo['sourceCategory'] = '';
		// get the YouTube video category?
		if ( isset( $video->snippet->playlistId ) ) {	// this is from a playlist
			$this->vbp->info_message( esc_html__( 'Notice: YouTube does not provide a category for Playlist items', 'video-blogster' ), 'updated', 'video_import' );
		}
		else if ( $videoInfo['categoryID'] ) { #JPH?
			$categories = $this->grab_categories();
			foreach ( $categories as $cat ) {
				if ( $cat->id == $videoInfo['categoryID'] && isset( $cat->snippet->title ) ) {
					$videoInfo['sourceCategoryName'] = $cat->snippet->title;
				}
			}
		}

		if ( false === $this->metadata )
			$videoInfo['img'] = $this->get_best_thumbnail( $video );
		else
			$videoInfo['img'] = null;

		// contentDetails info: =============================================
		$videoInfo['duration'] = '';
		if ( isset( $video->snippet->playlistId ) ) {	// this is from a playlist
			$this->vbp->info_message( esc_html__( 'Notice: YouTube does not provide a video duration for Playlist items', 'video-blogster' ), 'updated', 'video_import' );
		}
		else if ( isset( $video->contentDetails->duration ) ) {
			// let's try to parse the given ISO 8601 time ( ie: PT1H11M11S )
			preg_match_all('|([0-9]+)([a-z])|Ui', $video->contentDetails->duration, $matches );
			if( isset( $matches[2] ) ){
				$videoInfo['duration'] = '';
				$hours=$minutes=$seconds=0;
				foreach( $matches[2] as $key => $unit ) {
					if ( $unit == 'H' ) $hours = $matches[1][$key];
					else if ( $unit == 'M' ) $minutes = $matches[1][$key];
					else if ( $unit == 'S' ) $seconds = $matches[1][$key];
				}
				$videoInfo['duration'] = $hours ? sprintf( "%02s:%02s:%02s", $hours, $minutes, $seconds ) : sprintf( "%02s:%02s", $minutes, $seconds );
				$this->vbp->info_message( sprintf( esc_html__( 'Converted duration %s to %s.', 'video-blogster' ), $video->contentDetails->duration, $videoInfo['duration'] ), 'notice notice-warning', 'debug' );
			}
		}

		// statistics info: ==================================================
		// not available on playlists :(
		if ( isset( $video->snippet->playlistId ) ) {	// this is from a playlist
			$this->vbp->info_message( esc_html__( 'Notice: YouTube does not provide statistics for Playlist items', 'video-blogster' ), 'updated', 'video_import' );
		}
		$videoInfo['viewCount'] = isset( $video->statistics->viewCount ) ? $video->statistics->viewCount : 0;
		$videoInfo['likeCount'] = isset( $video->statistics->likeCount ) ? $video->statistics->likeCount : 0;
		$videoInfo['dislikeCount'] = isset( $video->statistics->dislikeCount ) ? $video->statistics->dislikeCount : 0;
		$videoInfo['favoriteCount'] = isset( $video->statistics->favoriteCount ) ? $video->statistics->favoriteCount : 0;
		$videoInfo['commentCount'] = isset( $video->statistics->commentCount ) ? $video->statistics->commentCount : 0;

		// external info: ====================================================
		$videoInfo['tags'] = '';

		// 08/3/15: tags now included in YouTube data? this is new. check for it but keep fallback method too.
		if ( isset( $video->snippet->tags ) ) { 
			$videoInfo['tags'] = implode( ',', $video->snippet->tags );
			$this->vbp->info_message( sprintf( esc_html__( 'Found snippet tags: %s.', 'video-blogster' ), $videoInfo['tags'] ), 'notice notice-warning', 'debug' );
		}
		else {
			$videoInfo['tags'] = $this->vbp->get_meta_tags( $videoInfo['url'] );
			$this->vbp->info_message( sprintf( esc_html__( 'Found meta tags: %s.', 'video-blogster' ), $videoInfo['tags'] ), 'notice notice-warning', 'debug' );
		}

		$videoInfo['feedID'] = ! empty ( $this->query_fields['id'] ) ? $this->query_fields['id'] : "";
		$videoInfo = apply_filters( 'vbp_get_video_info', $videoInfo, $video );

		$this->vbp->info_message( sprintf( esc_html__( '%s: %s', 'video-blogster' ), __FUNCTION__, htmlentities( print_r($videoInfo,true) ) ), 'notice notice-warning', 'debug' );

		return $videoInfo;
	}

	/**
	 * Takes an array of video details to process and create posts
	 */
	private function remove_deleted_videos( $ids, $videoDetails ) {

		if ( empty( $ids ) )
			return;

		foreach ( $videoDetails as $v => $video )  {
			// unset videoIDs that are still valid, leaving only videoIDs to delete
			$valid = true;
			if ( ( $key = array_search( $video->id, $ids ) ) !== FALSE 
				&& 
				( $video->status->uploadStatus == 'processed' 
				|| $video->status->uploadStatus == 'uploaded' ) 
				&& $video->status->privacyStatus != 'public'
			) {
				$valid = false;
			}
			else if ( ( $key = array_search( $video->id, $ids ) ) !== FALSE 
				&& isset( $video->status->embeddable )
				&& true !== $video->status->embeddable
			) {
				$valid = false;
			}

			if ( true === $valid ) {
				unset( $ids[$key] );
			}
		}

		$ids = apply_filters( 'vbp_remove_deleted_videos', $ids, 'YouTube' );

		if ( empty( $ids ) ) {
			$ids = array();
		}

		foreach ( $ids as $postID => $videoID ) {
			$this->vbp->video_blogster_metadata_unpublish( $postID, $videoID, 'YouTube' );
		}
	}

	/**
	 * Takes an array of video details to process and create posts
	 */
	private function save_videos( $videoDetails ) {

		if ( ! $videoDetails ) {
			return 0;
		}

		$getChannelDetails = apply_filters( 'vbp_youtube_get_channel_list_parts', false );

		foreach ( $videoDetails as $video )  {

			$videoInfo = $this->get_video_info( $video );
			if ( ! $videoInfo ) {
				continue;
			}
			$this->vbp->info_message( sprintf( esc_html__( 'Checking YouTube video: [%s] at %s', 'video-blogster' ), $videoInfo['title'], $videoInfo['url'] ), 'notice notice-warning', 'debug' );
			if ( $videoInfo['title'] == 'Deleted video' ) {
				$this->vbp->info_message( esc_html__( 'Deleted video detected. Skipping.', 'video-blogster' ), 'notice notice-warning', 'video_skip' );
				$this->num_skipped++;
				continue;
			}

			if ( false !== $getChannelDetails && $videoInfo['channelID'] ) {
				// user will use hooks in function to get data
				$channel = $this->grab_channel_info( array( 'id' => $videoInfo['channelID'], 'part' => $getChannelDetails ) );
				$videoInfo = apply_filters( 'vbp_youtube_channel_list_info', $videoInfo, $channel );
			}

			$postID = $this->vbp->save_the_video( $this->query_fields, $videoInfo, $this->metadata );

			if ( $postID < 0 ) return 0; // user abort!
			if ( ! $postID )  { $this->num_skipped++; continue; }

			if ( ! empty( $this->query_fields['qNumComments'] ) && $videoInfo['commentCount'] > 0 ) {
				$this->grab_comments( $videoInfo['videoID'], $postID );
			}

			$this->vbp->publish_the_video( $postID, $this->query_fields, $videoInfo );

			if ( $videoInfo['action'] == 'saved' ) $this->num_imported++;
			else if ( $videoInfo['action'] == 'updated' ) $this->num_updated++;

			$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
			if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;
		}
		$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
		if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;
		return 1;
	}

	/**
	 * Will grab up to $this->batch_limit vids from a playlist at a time
	 */
	private function grab_playlist_batch( $url ) {
		$data = $this->queryApi( $url );

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] : "";
		if ( ! $this->total ) {
			$this->total = isset( $data->pageInfo->totalResults ) ? $data->pageInfo->totalResults : 0;
			if ( $this->total ) {
				$this->vbp->info_message( sprintf( esc_html__( '%s YouTube API returned %s total results.', 'video-blogster' ), $feedID, $this->total ), 'updated', 'video_import' );
			}
		}

                if ( $this->total == 0 && empty( $data->items ) ) {
                        return $this->vbp->info_message( sprintf( esc_html__( '%s No results for YouTube query (%s)', 'video-blogster' ), $feedID, htmlentities( $url ) ), 'notice notice-warning', 'critical' );
                }
		$videoDetails = isset( $data->items ) ? $data->items : 0;
		$nextPage = isset( $data->nextPageToken ) ? $data->nextPageToken : 0;

		// playlistItems do not provide video category, duration or statistics
		// so grab all video ids from playlist and grab video list details instead. a bit more quota cost :(

		if ( empty( $videoDetails ) ) {
			$this->vbp->info_message( sprintf( esc_html__( '%s YouTube (%s) unable to get playlist items. Full response was: %s', 'video-blogster' ), $feedID, $url, htmlentities( print_r( $data,true ) ) ) );
		}

		$this->vbp->info_message( sprintf( esc_html__( 'YouTube response data (%s total results): %s', 'video-blogster' ), $this->total, htmlentities( print_r( $videoDetails,true ) ) ), 'notice notice-warning', 'debug' );

		unset( $data );

		$videoIDs = array();
		foreach ( $videoDetails as $video ) {
			$videoInfo = array();
			$videoInfo['videoSource'] = 'YouTube';

			$videoInfo['videoID'] = isset( $video->contentDetails->videoId ) ? $video->contentDetails->videoId : 0;
			$videoInfo['title'] = $videoInfo['videoID']; // snippet costs extra quota. just use id.
			// that's enough to check for duplicate post
			if ( $this->vbp->check_post_duplicate( $this->query_fields, $videoInfo ) ) {
				$this->num_skipped++;
				continue;
			}
			else if ( ! empty( $videoInfo['videoID'] ) ) $videoIDs[] = $videoInfo['videoID'];
		}
		if ( empty( $videoIDs ) ) return $nextPage;	// all previously imported 

		$commaList = implode( ',', $videoIDs );
		$this->vbp->info_message( sprintf( esc_html__( 'Fetching playlist videoIDs: %s', 'video-blogster' ), $commaList ), 'updated', 'video_import' );
		$videoDetails = $this->grab_videolist_details( '', $commaList );

		if ( empty( $videoDetails ) ) return $nextPage;	// YouTube has no data for deleted vids in playlists

		if ( ! $this->save_videos( $videoDetails ) ) return 0;

		unset( $videoDetails );

		return $nextPage;
	}

	/**
	 * Query for videos on a playlist
	 * https://developers.google.com/youtube/v3/docs/playlistItems/list
	 */
	private function grab_playlist() {
		$this->vbp->info_message( sprintf( esc_html__( 'Importing from playlist: %s', 'video-blogster' ), $this->query_fields['qAssoc'] ), 'updated', 'video_import' );
		$totalVids = $this->query_fields['qQueryBehavior'] == 'strict' ? $this->query_fields['qNumVideos'] : -1;
        	$query_args = array(
                	'part'		=> 'id,contentDetails',
                	'playlistId'	=> $this->query_fields['qAssoc'],
			'maxResults'	=> $totalVids > $this->batch_limit || $totalVids < 0 ? $this->batch_limit : $totalVids,
                	'key'		=> $this->apiKey
        	);
		
		$query_args = apply_filters( 'vbp_youtube_playlist_query', $query_args, $this->query_fields );

		while ( $totalVids ) {
			$url = $this->getApi( 'playlistItems.list' ) . '?' . http_build_query( $query_args );
			$query_args['pageToken'] = $this->grab_playlist_batch( $url );
                        if ( ! $query_args['pageToken'] ) break; // No need to request more.
			if ( $this->query_fields['qQueryBehavior'] == 'strict' && $totalVids > 0 ) {
                        	$totalVids -= $query_args['maxResults'];
				if ( $totalVids <= 0 ) break;	// failsafe for strict behavior
			}
                        $query_args['maxResults'] = $totalVids > $this->batch_limit || $totalVids < 0 ? $this->batch_limit : $totalVids;
                }
		$this->vbp->import_finished( 'YouTube videos', $this->num_skipped, $this->num_updated, $this->num_imported, $this->total, $this->query_fields['id'], $this->query_fields['qQueryBehavior'] );
	}

	/**
	 * Grab videos by the video IDs stored in qAssoc.
	 */
	private function grab_videos_by_ids() {

		if ( false === $this->metadata )
			$ids = explode( PHP_EOL, $this->query_fields['qAssoc'] );
		else
			$ids = $this->query_fields['qAssoc'];

		$this->vbp->info_message( sprintf( esc_html__( '%s: IDs are %s ', 'video-blogster' ), __FUNCTION__, print_r($ids,true) ), 'notice notice-warning', 'debug' );

		$chunks = array_chunk( $ids, $this->batch_limit );
		foreach( $chunks as $chunk ) {
			$commaIDs = implode( ',', $chunk );
			$videoDetails = $this->grab_videolist_details( '', $commaIDs );
			if ( true === $this->metadata )
				$this->remove_deleted_videos( $ids, $videoDetails );
			$this->save_videos( $videoDetails );
		}

		if ( false === $this->metadata )
			$this->vbp->import_finished( 'YouTube videos', $this->num_skipped, $this->num_updated, $this->num_imported, $this->total, $this->query_fields['id'], $this->query_fields['qQueryBehavior'] );
	}

	/**
	 * Do a batch query search for up to $this->batch_limit vids at a time
	 */
	private function grab_videos_batch( $url ) {
		$data = $this->queryApi( $url );

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] : "";
		if ( empty( $data ) ) {
			// let's try again one time
			sleep( 1 );
			$data = $this->queryApi( $url );
			if ( empty( $data ) ) {
				$this->vbp->info_message( sprintf( esc_html__( '%s YouTube (%s) sent an empty page response (request tried twice).', 'video-blogster' ), $feedID, $url) );
			}
		}

		if ( ! $this->total ) {
			$this->total = isset( $data->pageInfo->totalResults ) ? $data->pageInfo->totalResults : 0;
			if ( $this->total ) {
				$this->vbp->info_message( sprintf( esc_html__( '%s YouTube API returned %s total results.', 'video-blogster' ), $feedID, $this->total ), 'updated', 'video_import' );
			}
		}

                if ( $this->total == 0 && empty( $data->items ) ) {
                        return $this->vbp->info_message( sprintf( esc_html__( '%s No results for YouTube query (%s)', 'video-blogster' ), $feedID, htmlentities( $url ) ), 'updated', 'critical' );
                }
		$items = isset( $data->items ) ? $data->items : 0;
		$nextPage = isset( $data->nextPageToken ) ? $data->nextPageToken : 0;

		$this->vbp->info_message( sprintf( esc_html__( 'YouTube nextPageToken is %s', 'video-blogster' ), $nextPage ), 'notice notice-warning', 'debug' );


		if ( $items ) {
		$this->vbp->info_message( sprintf( esc_html__( 'YouTube response page (%s page results): %s', 'video-blogster' ), count( $items ), htmlentities( print_r( $items,true ) ) ), 'notice notice-warning', 'debug' );

		// initial search query has limited data, so make a separate query to get video details
		$videoDetails = ( isset( $items ) ) ? $this->grab_videolist_details( $items, '' ) : 0;

		if ( ! $videoDetails ) {
			return $nextPage;	// YouTube has no data for deleted vids in playlists
		}


		unset( $data );

		if ( ! $this->save_videos( $videoDetails ) ) return 0;
		unset( $videoDetails);
		}

		return $nextPage;
	}

////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * Process the video query and break down into batches
	 * https://developers.google.com/youtube/v3/docs/search/list
	 */
	public function grab_videos() {
		$keyphrase = $this->query_fields['qKeyphrase'];
		$assoc = $this->query_fields['qAssoc'];
		$assocType = $this->query_fields['qAssocType'];
		$query_type = 'search.list';
		$channelID = null;
		$filters = ! empty( $this->query_fields['qExtraParams'] ) ? "&" . ltrim( $this->query_fields['qExtraParams'], "&" ) : '';

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
		$this->vbp->info_message( sprintf( esc_html__( '%s: query_fields are %s ', 'video-blogster' ), __FUNCTION__, print_r($this->query_fields,true) ), 'notice notice-warning', 'debug' );

                if ( empty( $keyphrase ) 
			&& ( empty( $assoc ) || empty( $assocType ) || $assocType == 'search' ) 
			&& $assocType != 'mostPopular' 
			&& empty( $this->query_fields['qCategory'] )
		) {
                        return $this->vbp->info_message( sprintf( esc_html__( '%s Error: No YouTube keyphrase or association not set properly.', 'video-blogster' ), $feedID ), 'error', 'critical' );
                }

		if ( ! $assoc || $assocType == 'search' ) { // use keyphrase only
			$channelID = '';
		}
		else if ( $assocType == 'video' ) {
			return $this->grab_videos_by_ids();
		}
		else if ( $assocType == 'playlist' ) {
			return $this->grab_playlist();
		}
		else if ( $assocType == 'channel' ) {
			$channelID = $assoc;
			// if channel does not start with 'UC' then assume custom page
			$channelID = $this->check_for_custom_page( $channelID );
		}
		else if ( $assocType == 'user' ) {
			$channel = $this->grab_channel_info( array( 'forUsername' => $assoc, 'part' => 'id' ) );
			$channelID = $channel ? $channel->id : null;
			if ( ! $channelID ) {	# no such user/channel found - ABORT
				return $this->vbp->info_message( sprintf( esc_html__( '%s Error: No channelID returned by YouTube for user %s.', 'video-blogster' ), $feedID, $assoc ) );
			 }
		}
		else if ( $assocType == 'mostPopular' ) {
		}
		else { // shouldn't happen
			return $this->vbp->info_message( sprintf( esc_html__( '%s Error: association [%s] not recognized.', 'video-blogster' ), $feedID, $assocType ) );
		}

		$totalVids = $this->query_fields['qQueryBehavior'] == 'strict' ? $this->query_fields['qNumVideos'] : -1;
		
        	$query_args = array(
			'part' 			=> 'snippet',
			'maxResults' 		=> $totalVids > $this->batch_limit || $totalVids < 0 ? $this->batch_limit : $totalVids,
			'order'			=> ! empty( $this->query_fields['qOrderBy'] ) ? $this->query_fields['qOrderBy'] : 'date',
			'videoDuration'		=> ! empty( $this->query_fields['qDuration'] ) ? $this->query_fields['qDuration'] : 'any',
			'videoDefinition'	=> ! empty( $this->query_fields['qDefinition'] ) ? $this->query_fields['qDefinition'] : 'any',
			'type' 			=> 'video',
			'videoEmbeddable'       => 'TRUE',	// must set type to video
			'videoSyndicated'       => 'TRUE',	// only videos allowed outside of YouTube
			'safeSearch'       	=> ! empty( $this->query_fields['qSafeSearch'] ) ? $this->query_fields['qSafeSearch'] : 'moderate',
			'videoLicense'       	=> ! empty( $this->query_fields['qLicense'] ) ? $this->query_fields['qLicense'] : 'any',
			'videoCaption'       	=> ! empty( $this->query_fields['qCaption'] ) ? $this->query_fields['qCaption'] : 'any',
                	'key'			=> $this->apiKey,
		);

		// optional args:
		if ( $channelID ) {
			$query_args['channelId'] = $channelID;
		}
		if ( $keyphrase ) {
			$query_args['q'] = $keyphrase;
		}
		if ( $this->query_fields['qPublishedAfter'] ) {
			$date = $this->vbp->getDateTime( $this->query_fields['qPublishedAfter'], 'API query', false, false );
			if ( $date ) {
				$query_args['publishedAfter'] = $date->format( 'Y-m-d\TH:i:s\Z' );
			}
			else {
				$this->vbp->info_message( sprintf( esc_html__( '%s Error: publishedAfter date %s not recognized.', 'video-blogster' ), $feedID, $query_args['publishedAfter'] ) );
			}
		}
		if ( $this->query_fields['qPublishedBefore'] ) {
			$date = $this->vbp->getDateTime( $this->query_fields['qPublishedBefore'], 'API query', false, false );
			if ( $date ) {
				$query_args['publishedBefore'] = $date->format( 'Y-m-d\TH:i:s\Z' );
			}
			else {
				$this->vbp->info_message( sprintf( esc_html__( '%s Error: publishedBefore date %s not recognized.', 'video-blogster' ), $feedID, $query_args['publishedBefore'] ) );
			}
		}
		if ( $assocType == 'mostPopular' ) {	// special case with specific parameters
			$query_type = 'videos.list';
        		$query_args = array(
				'part' 			=> 'snippet,status',
				'maxResults' 		=> $totalVids > $this->batch_limit || $totalVids < 0 ? $this->batch_limit : $totalVids,
				'chart'			=> 'mostPopular',
				'type' 			=> 'video',
				'videoEmbeddable'       => 'TRUE',	// must set type to video
				'videoSyndicated'       => 'TRUE',	// only videos allowed outside of YouTube
                		'key'			=> $this->apiKey
			);
		}
		if ( $this->query_fields['qCategory'] ) {
			$query_args['videoCategoryId'] = $this->query_fields['qCategory'];
		}
		if ( $this->query_fields['qRegionCode'] ) {
			$query_args['regionCode'] = $this->query_fields['qRegionCode'];
		}
		if ( $this->query_fields['qLanguage'] && $this->query_fields['qLanguage'] != 'u8' ) {
			$query_args['relevanceLanguage'] = $this->query_fields['qLanguage'];
		}

		$query_args = apply_filters( 'vbp_youtube_search_query', $query_args, $this->query_fields );

		while ( $totalVids ) {
			$url = $this->getApi( $query_type ) . '?' . http_build_query( $query_args );
			if ( $filters ) $url .= $filters;
			$query_args['pageToken'] = $this->grab_videos_batch( $url );
			if ( ! $query_args['pageToken'] ) break; // No need to request more.
			if ( $this->query_fields['qQueryBehavior'] == 'strict' && $totalVids > 0 ) {
				$totalVids -= $query_args['maxResults'];
				if ( $totalVids <= 0 ) break;	// failsafe for strict behavior
			}
			$query_args['maxResults'] = $totalVids > $this->batch_limit || $totalVids < 0 ? $this->batch_limit : $totalVids;
		}

		$this->vbp->import_finished( 'YouTube videos', $this->num_skipped, $this->num_updated, $this->num_imported, $this->total, $this->query_fields['id'], $this->query_fields['qQueryBehavior'] );

		if ( $this->num_skipped + $this->num_updated + $this->num_imported  >= 500 ) {
			$this->vbp->info_message( esc_html__( 'Note: You have reached the max search results (500 videos) returned from YouTube. Each search result may be slightly different depending on ordering.', 'video-blogster' ), 'notice notice-warning', 'critical' );
		}

		return $query_args['pageToken'];
	}

	/**
	 * Gets the channel id from a custom page, if any
	 * save in transient for 10 days
	 */
	private function check_for_custom_page( $channelID ) {

		if ( 'UC' !== substr( $channelID, 0, 2 ) ) {

			$this->vbp->info_message( sprintf( esc_html__( 'Custom channel page detected: %s', 'video-blogster' ), $channelID ), 'notice notice-warning', 'debug' );

			// check for transient 1st
			$key = "vbp_custom_channel_" . $channelID;
			$results = $this->vbp->transientChecker( $key );
			if ( ! empty ( $results  ) ) {
				$this->vbp->info_message( sprintf( esc_html__( 'Custom channel page transient mapped to channel: %s', 'video-blogster' ), $results ), 'notice notice-warning', 'debug' );
				return $results;
			}

			// search channels for possible match
        		$query_args = array(
				'part' 			=> 'id',
				'type'			=> 'channel',
				'q'			=> $channelID,
				'maxResults'		=> $this->batch_limit,
                		'key'			=> $this->apiKey
			);
			$url = $this->getApi( 'search.list' ) . '?' . http_build_query( $query_args );
			$data = $this->queryApi( $url );

			$this->vbp->info_message( sprintf( esc_html__( 'Custom channel search result: %s', 'video-blogster' ), print_r($data->items,true) ), 'notice notice-warning', 'debug' );

			$item = isset( $data->items ) ? current( $data->items ) : array();

			// should be only 1 result, if any.
			if ( isset( $item->id->channelId ) )  {
				$channelID = $item->id->channelId;
				$this->vbp->info_message( sprintf( esc_html__( 'Custom channel page mapped to channel: %s', 'video-blogster' ), $channelID ), 'notice notice-warning', 'debug' );
				$this->vbp->transientSave( $key, $channelID, 864000 );
			}
		}

		return $channelID;
	}

	/**
	 * Gets the channel id from a user handle
	 * save in transient for 10 days
	 */
	private function grab_channel_info( $query_args ) {
		// check transient data 
		$key = "vbp_" . implode( "_", $query_args );
		$results = $this->vbp->transientChecker( $key );
		if ( ! empty ( $results  ) ) {
			return $results;
		}

		if ( ! isset( $query_args['key'] ) ) {
			$query_args['key'] = $this->apiKey;
		}

		$url = $this->getApi( 'channels.list' ) . '?' . http_build_query( $query_args );
		$data = $this->queryApi( $url );
		$channel = isset( $data->items[0] ) ? $data->items[0] : null;

		$this->vbp->transientSave( $key, $channel, 864000 );
		return $channel;

	}

    } // END class Video_Blogster_YouTube
} // END if ( ! class_exists( 'Video_Blogster_YouTube' ) )
?>
