/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps } from '@wordpress/block-editor';
import { TextControl, CheckboxControl, Button, ButtonGroup, ExternalLink } from '@wordpress/components';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( { attributes, setAttributes } ) {
	const { title, showMapHeading, mapHeading, showAddressHeading, addressHeading, mapWidth, mapHeight, zoomLevel, apiKey, hideMap, hideHeading } = attributes;
	const limitMapZoom = ( zoomLevel ) => {
		if ( zoomLevel > 25 ) {
			zoomLevel =  25;
		}
		setAttributes( { zoomLevel } );
	}
	return (
		<div { ...useBlockProps() }>
			<TextControl
				label={ __( 'Title', 'wcvendors-pro' ) }
				value={ title }
				onChange={ ( title ) => setAttributes( { title } ) 
			} />
			
			<CheckboxControl
				label={ __( 'Show map heading', 'wcvendors-pro' ) }
				checked={ showMapHeading }
				onChange={ ( showMapHeading ) => setAttributes( { showMapHeading } )
			} />

			{
				showMapHeading && (
					<TextControl
						label={ __( 'Map heading', 'wcvendors-pro' ) }
						value={ mapHeading }
						onChange={ ( mapHeading ) => setAttributes( { mapHeading } )
					} />
				)
			}

			<CheckboxControl
				label={ __( 'Show address heading', 'wcvendors-pro' ) }
				checked={ showAddressHeading }
				onChange={ ( showAddressHeading ) => setAttributes( { showAddressHeading } )
			} />

			{
				showAddressHeading && (
					<TextControl
						label={ __( 'Address heading', 'wcvendors-pro' ) }
						value={ addressHeading }
						onChange={ ( addressHeading ) => setAttributes( { addressHeading } )
					} />
				)
			}

			<TextControl
				label={ __( 'Map height', 'wcvendors-pro' ) }
				value={ mapHeight }
				onChange={ ( mapHeight ) => setAttributes( { mapHeight } )
			} />

			<TextControl
				label={ __( 'Map width', 'wcvendors-pro' ) }
				value={ mapWidth }
				onChange={ ( mapWidth ) => setAttributes( { mapWidth } ) }
			/>

			<TextControl
				label={ __( 'Zoom level', 'wcvendors-pro' ) }
				value={ zoomLevel }
				type='number'
				max={ 25 }
				onChange={ limitMapZoom } 
				help={ __( 'Max zoom level is 25', 'wcvendors-pro' ) }
			/>

			<TextControl
				label={ __( 'Google Map API Key', 'wcvendors-pro' ) }
				value={ apiKey }
				onChange={ ( apiKey ) => setAttributes( { apiKey } ) }
				help={ __( 'Required for your map to work', 'wcvendors-pro' )  }
			 />

			<h4> { }</h4>

			<ButtonGroup>
				<Button variant="secondary" href="admin.php?page=wcv-settings&tab=forms" target="_blank">
					{ __( 'Setup Google Maps API Key', 'wcvendors-pro' ) }
				</Button>
				<Button variant="secondary" href="https://developers.google.com/maps/documentation/javascript/get-api-key" target="_blank">
					{ __( 'Get Google Maps API Key', 'wcvendors-pro' ) }
				</Button>
			</ButtonGroup>

			<p><i>{ __( 'Select the maps and places options when creating your key', 'wcvendors-pro' ) }</i></p>

			<CheckboxControl
				label={ __( 'Hide map', 'wcvendors-pro' ) }
				checked={ hideMap }
				onChange={ ( hideMap ) => setAttributes( { hideMap } )
			} />

			<CheckboxControl
				label={ __( 'Hide heading', 'wcvendors-pro' ) }
				checked={ hideHeading }
				onChange={ ( hideHeading ) => setAttributes( { hideHeading } )
			} />
		</div>
	);
}
