/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps } from '@wordpress/block-editor';
import { TextControl, CheckboxControl, RangeControl, Button, ColorPicker } from '@wordpress/components';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( { attributes, setAttributes } ) {
	const { title, showApplyButton, railColor, trackColor } = attributes;
	return (
		<div { ...useBlockProps() }>
			<TextControl
				label={ __( 'Title', 'wcvendors-pro' ) }
				value={ title }
				onChange={ ( title ) => setAttributes( { title } ) }
			/>
			<TextControl
				label={ __( 'Rail Color', 'wcvendors-pro' ) }
				value={ railColor }
				onChange={ ( railColor ) => setAttributes( { railColor } ) }
				type='color'
				style={ { width: '100px', display: 'block' } }
			/>
			<TextControl
				label={ __( 'Track Color', 'wcvendors-pro' ) }
				value={ trackColor }
				onChange={ ( trackColor ) => setAttributes( { trackColor } ) }
				type='color'
				style={ { width: '100px', display: 'block' } }
			/>

			<CheckboxControl
				label={ __( 'Show Apply Button', 'wcvendors-pro' ) }
				checked={ showApplyButton }
				onChange={ ( showApplyButton ) => setAttributes( { showApplyButton } ) }
			/>

			<RangeControl
				label={ __( 'Preview', 'wcvendors-pro' ) }
				min={ 0 }
				max={ 100 }
				railColor= { railColor }
				allowReset={ false }
				handleColor={ trackColor }
				trackColor={trackColor}
				withInputField={ false }
			/>
			{ showApplyButton && (
				<Button isPrimary>
					{ __( 'Apply', 'wcvendors-pro' ) }
				</Button>

			) }


		</div>
	);
}
