/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps } from '@wordpress/block-editor';
import { TextControl, CheckboxControl } from '@wordpress/components';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( { attributes, setAttributes } ) {
	const { title, number, showRatingTitle, showCustomer, showDate, showComment, showProduct } = attributes;
	return (
		<div { ...useBlockProps() }>
			<TextControl
				label={ __( 'Title', 'wcvendors-pro' ) }
				value={ title }
				onChange={ ( title ) => setAttributes( { title } ) }
			/>
			<TextControl
				type='number'
				min={1}
				max={100}
				maxlength='3'
				label={ __( 'Number of ratings to show', 'wcvendors-pro' ) }
				value={ number }
				onChange={ ( number ) => setAttributes( { number : parseInt( number ) } ) }
			/>

			<CheckboxControl
				label={ __( 'Show rating title', 'wcvendors-pro' ) }
				checked={ showRatingTitle }
				onChange={ ( showRatingTitle ) => setAttributes( { showRatingTitle } ) }
			/>

			<CheckboxControl
				label={ __( 'Show customer name', 'wcvendors-pro' ) }
				checked={ showCustomer }
				onChange={ ( showCustomer ) => setAttributes( { showCustomer } ) }
			/>
			<CheckboxControl
				label={ __( 'Show date', 'wcvendors-pro' ) }
				checked={ showDate }
				onChange={ ( showDate ) => setAttributes( { showDate } ) }
			/>
			<CheckboxControl
				label={ __( 'Show comment', 'wcvendors-pro' ) }
				checked={ showComment }
				onChange={ ( showComment ) => setAttributes( { showComment } ) }
			/>
			<CheckboxControl
				label={ __( 'Show product', 'wcvendors-pro' ) }
				checked={ showProduct }
				onChange={ ( showProduct ) => setAttributes( { showProduct } ) }
			/>
		</div>
	);
}
