/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps } from '@wordpress/block-editor';
import { TextControl, SelectControl, CheckboxControl } from '@wordpress/components';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( { attributes, setAttributes } ) {
	const { title, numberOfProducts, showRating, showImage, imageSize, showExcerpt, excerptLength } = attributes;
	console.log( attributes );
	return (
		<div { ...useBlockProps() }>
			<TextControl
				label={ __( 'Title', 'wcvendors-pro' ) }
				value={ title }
				onChange={ ( title ) => setAttributes( { title } ) }
			/>

			<TextControl
				label={ __( 'Number of products', 'wcvendors-pro' ) }
				type='number'
				min={ 1 }
				max={ 100 }
				maxlength='3'
				value={ numberOfProducts }
				onChange={ ( numberOfProducts ) => setAttributes( { numberOfProducts: parseInt(numberOfProducts) } ) }
			/>

			<CheckboxControl
				label={ __( 'Show rating', 'wcvendors-pro' ) }
				checked={ showRating }
				onChange={ ( showRating ) => setAttributes( { showRating } ) }
			/>

			<CheckboxControl
				label={ __( 'Show image', 'wcvendors-pro' ) }
				checked={ showImage }
				onChange={ ( showImage ) => setAttributes( { showImage } ) }
			/>
			{ showImage && (
				<SelectControl
					label={ __( 'Image szie', 'wcvendors-pro' ) }
					value={ imageSize }
					options={ [
						{ label: __( 'Thumbnail', 'wcvendors-pro' ), value: 'thumbnail' },
						{ label: __( 'Medium', 'wcvendors-pro' ), value: 'medium' },
						{ label: __( 'Medium large', 'wcvendors-pro' ), value: 'medium-large' },
						{ label: __( 'Large', 'wcvendors-pro' ), value: 'large' },
						{ label: __( 'Full', 'wcvendors-pro' ), value: 'full' },
					] }
					onChange={ ( imageSize ) => setAttributes( { imageSize } ) }
				/>
			) }
			<CheckboxControl
				label={ __( 'Show excerpt', 'wcvendors-pro' ) }
				checked={ showExcerpt }
				onChange={ ( showExcerpt ) => setAttributes( { showExcerpt } ) }
			/>
			{ showExcerpt && (
				<TextControl
					label={ __( 'Excerpt length', 'wcvendors-pro' ) }
					value={ excerptLength }
					onChange={ ( excerptLength ) => setAttributes( { excerptLength } ) }
				/>
			) }
		</div>
	);
}
