/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps } from '@wordpress/block-editor';
import { TextControl, CheckboxControl } from '@wordpress/components';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( { attributes, setAttributes }) {
	const { title, numberBeforeLabel, label, showLabel } = attributes;
	const tempLabel = showLabel ? label : '';
	return (
		<div { ...useBlockProps() }>
			<TextControl
				label={ __( 'Title' ) }
				value={ title }
				onChange={ ( value ) => setAttributes( { title: value } ) }
			/>
			<TextControl
				label={ __( 'Label' ) }
				value={ label }
				onChange={ ( value ) => setAttributes( { label: value } ) }
			/>
			<CheckboxControl
				label={ __( 'Show label' ) }
				checked={ showLabel }
				onChange={ ( value ) => setAttributes( { showLabel: value } ) }
			/>
			<CheckboxControl
				label={ __( 'Display number before label' ) }
				checked={ numberBeforeLabel }
				onChange={ ( value ) => setAttributes( { numberBeforeLabel: value } ) }
				disabled={ ! showLabel }
			/>
			<div>{ numberBeforeLabel ? `0 ${ tempLabel }` : `${ tempLabel } 0` }</div>
		</div>
	);
}
