<?php

use WC_Vendors\Classes\Front\WCV_Form_Helper;

/**
 * Form Helper Class
 *
 * Defines relevant static methods for generating form elements for public facing forms.
 *
 * @package    WCVendors_Pro
 * @subpackage WCVendors_Pro/public
 * @author     Jamie Madden <support@wcvendors.com>
 */
class WCVendors_Pro_Form_Helper extends WCV_Form_Helper {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $wcvendors_pro The ID of this plugin.
     */
    private $wcvendors_pro;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $version The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     *
     * @param    string $wcvendors_pro The name of the plugin.
     * @param    string $version       The version of this plugin.
     */
    public function __construct( $wcvendors_pro, $version ) {

        $this->wcvendors_pro = $wcvendors_pro;
        $this->version       = $version;
    }

    /**
     * Countries array
     *
     * @version 1.9.4
     * @since   1.9.4
     */
    public static function countries() {
        if ( ! function_exists( 'WC' ) ) {

            return array();
        }

        if ( ! isset( WC()->countries ) ) {

            return array();
        }

        $countries = WC()->countries->get_allowed_countries();

        if ( 'specific' !== get_option( 'woocommerce_ship_to_countries' ) ) {
            if ( ! isset( $countries['EWE'] ) ) {
                $countries['EWE'] = __( 'Everywhere else', 'wc-vendors' );
            }
        }

        if ( ! is_array( $countries ) ) {
            return array();
        }

        return apply_filters( 'wcv_countries_list', $countries );
    }


    /**
     * Output a file upload link
     *
     * @since   2.0.2
     * @version 2.0.2
     *
     * @param   array  $field file uploader arguments.
     * @param   string $type  Upload file type.
     *
     * @todo    add filters to allow the field to be hooked into this should not echo html but return it.
     */
    public static function pro_file_uploader( $field, $type = 'image' ) {

        if ( 'image' === $type ) {
            self::pro_image_uploader( $field );

            return;
        }

        $field['header_text']   = isset( $field['header_text'] ) ? $field['header_text'] : __( 'File', 'wc-vendors' );
        $field['add_text']      = isset( $field['add_text'] ) ? $field['add_text'] : __( 'Add file', 'wc-vendors' );
        $field['remove_text']   = isset( $field['remove_text'] ) ? $field['remove_text'] : __( 'Remove file', 'wc-vendors' );
        $field['file_meta_key'] = isset( $field['file_meta_key'] ) ? $field['file_meta_key'] : '_wcv_file_id';
        $field['save_button']   = isset( $field['save_button'] ) ? $field['save_button'] : __( 'Add File', 'wc-vendors' );
        $field['window_title']  = isset( $field['window_title'] ) ? $field['window_title'] : __( 'Select a File', 'wc-vendors' );
        $field['value']         = isset( $field['value'] ) ? $field['value'] : 0;
        $field['class']         = isset( $field['class'] ) ? $field['class'] : '';
        $field['wrapper_start'] = isset( $field['wrapper_start'] ) ? $field['wrapper_start'] : '';
        $field['wrapper_end']   = isset( $field['wrapper_end'] ) ? $field['wrapper_end'] : '';
        $field['required']      = isset( $field['required'] ) ? $field['required'] : '';
        $field['id']            = isset( $field['id'] ) ? $field['id'] : $field['file_meta_key'];

        $file_url = '';
        $has_file = false;
        $required = '' !== $field['required'] ? 'required' : '';

        if ( $field['value'] ) {
            $file_url = wp_get_attachment_url( $field['value'] );
        }

        if ( $file_url ) {
            $has_file = true;
        }

        do_action( 'wcv_form_file_uploader_before_' . $field['id'], $field );

        // Container wrapper start if defined start & end required otherwise no output is shown.
        if ( ! empty( $field['wrapper_start'] ) && ! empty( $field['wrapper_end'] ) ) {
            echo $field['wrapper_start']; //phpcs:ignore
        }

        echo '<div class="wcv-file-uploader' . esc_attr( $field['file_meta_key'] ) . ' ' . esc_attr( $field['class'] ) . '">';

        if ( $file_url ) {
            $escaped_url = esc_url( $file_url );
            switch ( $type ) {
                case 'video':
                    echo do_shortcode( '[video src="' . esc_url( $file_url ) . '"]' );
                    break;
                case 'audio':
                    echo do_shortcode( '[audio src="' . esc_url( $file_url ) . '"]' );
                    break;
                case 'document':
                    printf( '<a href="%1$s" target="_blank">%2$s</a>', esc_url( $file_url ), esc_html( $file_url ) );
                    break;
                default:
                    echo '<div class="control-group"><div class="control"><input value="' . esc_attr( $escaped_url ) . '" type="text" readonly /></div></div>';
            }
        }

        echo '</div>';

        echo '<a class="wcv-file-uploader-add' . esc_attr( $field['file_meta_key'] ) . ' ' . ( $has_file ? 'hidden' : '' ) . '" href="#">' . esc_html( $field['add_text'] ) . '</a><br />';
        echo '<a class="wcv-file-uploader-delete' . esc_attr( $field['file_meta_key'] ) . ' ' . ( ! $has_file ? 'hidden' : '' ) . '" href="#" >' . esc_html( $field['remove_text'] ) . '</a><br />';
        echo '<input class="wcv-img-id wcv-file-uploader" name="' . esc_attr( $field['file_meta_key'] ) . '" id="' . esc_attr( $field['file_meta_key'] ) . '" type="hidden" value="' . esc_attr( $field['value'] ) . '" data-file_meta_key="' . esc_attr( $field['file_meta_key'] ) . '" data-save_button="' . esc_attr( $field['save_button'] ) . '" data-window_title="' . esc_attr( $field['window_title'] ) . '" data-type="' . esc_attr( $type ) . '" data-msg-id="' . esc_attr( $field['file_meta_key'] ) . '_msg" ' . esc_attr( $required ) . ' />';
        echo '<span id="' . esc_attr( $field['file_meta_key'] ) . '_msg"></span>';

        // container wrapper end if defined.
        if ( ! empty( $field['wrapper_start'] ) && ! empty( $field['wrapper_end'] ) ) {
            echo $field['wrapper_end']; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        }

        do_action( 'wcv_form_file_uploader_after_' . $field['id'], $field );
    }

    /**
     * Output a file upload link
     * This method is the original of file_uploader method
     *
     * @since      2.0.2
     *
     * @param      array $field file uploader arguments.
     *
     * @todo       add filters to allow the field to be hooked into this should not echo html but return it.
     */
    public static function pro_image_uploader( $field ) {

        $field['header_text']    = isset( $field['header_text'] ) ? $field['header_text'] : __( 'Image', 'wc-vendors' );
        $field['add_text']       = isset( $field['add_text'] ) ? $field['add_text'] : __( 'Add Image', 'wc-vendors' );
        $field['remove_text']    = isset( $field['remove_text'] ) ? $field['remove_text'] : __( 'Remove Image', 'wc-vendors' );
        $field['image_meta_key'] = isset( $field['image_meta_key'] ) ? $field['image_meta_key'] : '_wcv_image_id';
        $field['save_button']    = isset( $field['save_button'] ) ? $field['save_button'] : __( 'Add Image', 'wc-vendors' );
        $field['window_title']   = isset( $field['window_title'] ) ? $field['window_title'] : __( 'Select an Image', 'wc-vendors' );
        $field['value']          = isset( $field['value'] ) ? $field['value'] : 0;
        $field['size']           = isset( $field['size'] ) ? $field['size'] : 'full';
        $field['class']          = isset( $field['class'] ) ? $field['class'] : '';
        $field['wrapper_start']  = isset( $field['wrapper_start'] ) ? $field['wrapper_start'] : '';
        $field['wrapper_end']    = isset( $field['wrapper_end'] ) ? $field['wrapper_end'] : '';
        $field['required']       = isset( $field['required'] ) ? $field['required'] : '';
        $field['id']             = isset( $field['id'] ) ? $field['id'] : $field['image_meta_key'];

        // Get the image src.
        $image_src = wp_get_attachment_image_src( $field['value'], $field['size'] );

        // See if the array is valid.
        $has_image = is_array( $image_src );

        // Get attachment alt text if available.
        $alt_text = '';
        if ( $has_image && $field['value'] ) {
            $alt_text = get_post_meta( $field['value'], '_wp_attachment_image_alt', true );
        }

        do_action( 'wcv_form_file_uploader_before_' . $field['id'], $field );

        // Container wrapper start if defined start & end required otherwise no output is shown.
        if ( ! empty( $field['wrapper_start'] ) && ! empty( $field['wrapper_end'] ) ) {
            echo $field['wrapper_start']; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        }

        echo '<div class="wcv-file-uploader' . esc_attr( $field['image_meta_key'] ) . ' ' . esc_attr( $field['class'] ) . '">';

        if ( $has_image ) {
            echo '<img src="' . esc_url( $image_src[0] ) . '" alt="' . esc_attr( $alt_text ) . '" style="max-width:100%; margin-bottom: 16px;" />';
        }

        echo '</div>';

        $required = '' !== $field['required'] ? 'required' : '';

        echo '<button class="wcv-pro-file-uploader-add' . esc_attr( $field['image_meta_key'] ) . ' ' . ( $has_image ? 'hidden' : '' ) . ' wcv-button wcv-button-outline text-blue">' . wp_kses( wcv_get_icon( 'wcv-icon wcv-icon-24 wcv-icon-middle wcv-icon-left', 'wcv-icon-photo' ), wcv_allowed_html_tags() ) . esc_html( $field['add_text'] ) . '</button> ';
        echo '<button class="wcv-pro-file-uploader-delete' . esc_attr( $field['image_meta_key'] ) . ' ' . ( ! $has_image ? 'hidden' : '' ) . '  wcv-button wcv-button-cancel" >' . esc_html( $field['remove_text'] ) . '</button><br />';
        echo '<input class="wcv-img-id wcv-file-uploader" name="' . esc_attr( $field['image_meta_key'] ) . '" id="' . esc_attr( $field['image_meta_key'] ) . '" type="hidden" value="' . esc_attr( $field['value'] ) . '" data-image_meta_key="' . esc_attr( $field['image_meta_key'] ) . '" data-save_button="' . esc_attr( $field['save_button'] ) . '" data-window_title="' . esc_attr( $field['window_title'] ) . '" data-type="image" data-msg-id="' . esc_attr( $field['image_meta_key'] ) . '_msg" ' . esc_attr( $required ) . ' />';
        echo '<span id="' . esc_attr( $field['image_meta_key'] ) . '_msg"></span>';
        // container wrapper end if defined.
        if ( ! empty( $field['wrapper_start'] ) && ! empty( $field['wrapper_end'] ) ) {
            echo $field['wrapper_end']; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        }

        do_action( 'wcv_form_file_uploader_after_' . $field['id'], $field );
    }
}
