<?php
/**
 * Handles product CSV export. Modelled on the WooCommerce Exporter.
 *
 * WooCommerce Export Class version 3.1.0.
 *
 * @package WCVendors\Public\Import-Export\Export
 * @version 1.8.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Include dependencies.
 */
if ( ! class_exists( 'WCV_CSV_Batch_Exporter', false ) ) {
    include_once 'abstract-wcv-csv-batch-exporter.php';
}

/**
 * WCV_Product_CSV_Exporter Class.
 */
class WCV_Product_CSV_Exporter extends WCV_CSV_Batch_Exporter {

    /**
     * Type of export used in filter names.
     *
     * @var string
     */
    protected $export_type = 'product';

    /**
     * Should meta be exported?
     *
     * @var boolean
     */
    protected $enable_meta_export = false;

    /**
     * Which product types are being exported.
     *
     * @var array
     */
    protected $product_types_to_export = array();

    /**
     * Products belonging to what category should be exported.
     *
     * @var string
     */
    protected $product_category_to_export = array();

    /**
     * Constructor.
     */
    public function __construct() {
        parent::__construct();
        $this->set_product_types_to_export( array_keys( WC_Admin_Exporters::get_product_types() ) );
        $this->log( 'Product CSV Exporter initialized', 'info' );
    }

    /**
     * Should meta be exported?
     *
     * @param bool $enable_meta_export Should meta be exported.
     *
     * @since 1.8.0
     */
    public function enable_meta_export( $enable_meta_export ) {
        $this->enable_meta_export = (bool) $enable_meta_export;
        $this->log( 'Meta export ' . ( $enable_meta_export ? 'enabled' : 'disabled' ), 'info' );
    }

    /**
     * Product types to export.
     *
     * @param array $product_types_to_export List of types to export.
     *
     * @since 1.8.0
     */
    public function set_product_types_to_export( $product_types_to_export ) {
        $this->product_types_to_export = array_map( 'wc_clean', $product_types_to_export );
        $this->log( 'Product types to export set: ' . implode( ', ', $this->product_types_to_export ), 'info' );
    }

    /**
     * Product category to export
     *
     * @param array|string $product_category_to_export Product category slugs to export, empty array exports all.
     *
     * @since  3.5.0
     * @return void
     */
    public function set_product_category_to_export( $product_category_to_export ) {
        if ( ! is_array( $product_category_to_export ) ) {
            $this->log( 'Product category parameter should be an array, received: ' . gettype( $product_category_to_export ), 'warning' );
            $product_category_to_export = is_string( $product_category_to_export ) ? array( $product_category_to_export ) : array();
        }
        $this->product_category_to_export = array_map( 'sanitize_title_with_dashes', $product_category_to_export );
        $this->log( 'Product categories to export set: ' . print_r( $this->product_category_to_export, true ), 'info' ); // phpcs:ignore
    }

    /**
     * Return an array of columns to export.
     *
     * @since  1.8.0
     * @return array
     */
    public function get_default_column_names() {
        return apply_filters(
            "wcvendors_product_export_{$this->export_type}_default_columns",
            array(
                'id'                 => __( 'ID', 'wcvendors-pro' ),
                'type'               => __( 'Type', 'wcvendors-pro' ),
                'sku'                => __( 'SKU', 'wcvendors-pro' ),
                'name'               => __( 'Name', 'wcvendors-pro' ),
                'published'          => __( 'Published', 'wcvendors-pro' ),
                'featured'           => __( 'Is featured?', 'wcvendors-pro' ),
                'catalog_visibility' => __( 'Visibility in catalog', 'wcvendors-pro' ),
                'short_description'  => __( 'Short description', 'wcvendors-pro' ),
                'description'        => __( 'Description', 'wcvendors-pro' ),
                'date_on_sale_from'  => __( 'Date sale price starts', 'wcvendors-pro' ),
                'date_on_sale_to'    => __( 'Date sale price ends', 'wcvendors-pro' ),
                'tax_status'         => __( 'Tax status', 'wcvendors-pro' ),
                'tax_class'          => __( 'Tax class', 'wcvendors-pro' ),
                'stock_status'       => __( 'In stock?', 'wcvendors-pro' ),
                'stock'              => __( 'Stock', 'wcvendors-pro' ),
                'low_stock_amount'   => __( 'Low stock amount', 'wcvendors-pro' ),
                'backorders'         => __( 'Backorders allowed?', 'wcvendors-pro' ),
                'sold_individually'  => __( 'Sold individually?', 'wcvendors-pro' ),
                /* translators: %s: weight */
                'weight'             => sprintf( __( 'Weight (%s)', 'wcvendors-pro' ), get_option( 'woocommerce_weight_unit' ) ),
                /* translators: %s: length */
                'length'             => sprintf( __( 'Length (%s)', 'wcvendors-pro' ), get_option( 'woocommerce_dimension_unit' ) ),
                /* translators: %s: width */
                'width'              => sprintf( __( 'Width (%s)', 'wcvendors-pro' ), get_option( 'woocommerce_dimension_unit' ) ),
                /* translators: %s: Height */
                'height'             => sprintf( __( 'Height (%s)', 'wcvendors-pro' ), get_option( 'woocommerce_dimension_unit' ) ),
                'reviews_allowed'    => __( 'Allow customer reviews?', 'wcvendors-pro' ),
                'purchase_note'      => __( 'Purchase note', 'wcvendors-pro' ),
                'sale_price'         => __( 'Sale price', 'wcvendors-pro' ),
                'regular_price'      => __( 'Regular price', 'wcvendors-pro' ),
                'category_ids'       => __( 'Categories', 'wcvendors-pro' ),
                'tag_ids'            => __( 'Tags', 'wcvendors-pro' ),
                'shipping_class_id'  => __( 'Shipping class', 'wcvendors-pro' ),
                'images'             => __( 'Images', 'wcvendors-pro' ),
                'download_limit'     => __( 'Download limit', 'wcvendors-pro' ),
                'download_expiry'    => __( 'Download expiry days', 'wcvendors-pro' ),
                'parent_id'          => __( 'Parent', 'wcvendors-pro' ),
                'grouped_products'   => __( 'Grouped products', 'wcvendors-pro' ),
                'upsell_ids'         => __( 'Upsells', 'wcvendors-pro' ),
                'cross_sell_ids'     => __( 'Cross-sells', 'wcvendors-pro' ),
                'product_url'        => __( 'External URL', 'wcvendors-pro' ),
                'button_text'        => __( 'Button text', 'wcvendors-pro' ),
                'menu_order'         => __( 'Position', 'wcvendors-pro' ),
            )
        );
    }

    /**
     * Prepare data for export. Overridden in WCVendors_Pro_Product_CSV_Exporter
     *
     * @since 1.8.0
     */
    public function prepare_data_to_export() {
        $this->log( 'Starting data preparation for export', 'info' );

        $args = array(
            'status'   => array( 'private', 'publish', 'draft', 'future', 'pending' ),
            'author'   => get_current_user_id(),
            'type'     => $this->product_types_to_export,
            'limit'    => $this->get_limit(),
            'page'     => $this->get_page(),
            'orderby'  => array(
                'ID' => 'ASC',
            ),
            'return'   => 'objects',
            'paginate' => true,
        );

        if ( ! empty( $this->product_category_to_export ) ) {
            $args['category'] = $this->product_category_to_export;
        }

        $this->log( 'Query args: ' . wp_json_encode( $args ), 'info' );

        $products = wc_get_products( apply_filters( "wcvendors_product_export_{$this->export_type}_query_args", $args ) );

        if ( ! $products || ! isset( $products->products ) ) {
            $this->log( 'No products found for export', 'warning' );
            $this->total_rows = 0;
            $this->row_data   = array();
            return;
        }

        $this->total_rows  = $products->total;
        $this->row_data    = array();
        $variable_products = array();

        $this->log( "Found {$this->total_rows} products for export", 'info' );

        foreach ( $products->products as $product ) {
            if ( ! $product || ! is_a( $product, 'WC_Product' ) ) {
                $this->log( 'Invalid product object encountered, skipping', 'warning' );
                continue;
            }

            // Check if the category is set, this means we need to fetch variations seperately as they are not tied to a category.
            if ( ! empty( $args['category'] ) && $product->is_type( 'variable' ) ) {
                $variable_products[] = $product->get_id();
            }

            try {
                $this->row_data[] = $this->generate_row_data( $product );
            } catch ( Exception $e ) {
                $this->log( 'Error generating row data for product ID ' . $product->get_id() . ': ' . $e->getMessage(), 'error' );
                continue;
            }
        }

        // If a category was selected we loop through the variations as they are not tied to a category so will be excluded by default.
        if ( ! empty( $variable_products ) ) {
            $this->log( 'Processing variations for ' . count( $variable_products ) . ' variable products', 'info' );

            foreach ( $variable_products as $parent_id ) {
                $products = wc_get_products(
                    array(
                        'parent' => $parent_id,
                        'type'   => array( 'variation' ),
                        'return' => 'objects',
                        'limit'  => -1,
                    )
                );

                if ( ! $products ) {
                    $this->log( "No variations found for product ID {$parent_id}", 'warning' );
                    continue;
                }

                foreach ( $products as $product ) {
                    if ( ! $product || ! is_a( $product, 'WC_Product' ) ) {
                        $this->log( "Invalid variation product object for parent ID {$parent_id}, skipping", 'warning' );
                        continue;
                    }

                    try {
                        $this->row_data[] = $this->generate_row_data( $product );
                    } catch ( Exception $e ) {
                        $this->log( 'Error generating row data for variation ID ' . $product->get_id() . ': ' . $e->getMessage(), 'error' );
                        continue;
                    }
                }
            }
        }

        $this->log( 'Data preparation completed. Generated ' . count( $this->row_data ) . ' rows for export', 'info' );
    }

    /**
     * Take a product and generate row data from it for export.
     *
     * @param WC_Product $product WC_Product object.
     *
     * @return array
     */
    protected function generate_row_data( $product ) {
        if ( ! $product || ! is_a( $product, 'WC_Product' ) ) {
            $this->log( 'Invalid product object passed to generate_row_data', 'error' );
            return array();
        }

        $columns = $this->get_column_names();
        $row     = array();

        foreach ( $columns as $column_id => $column_name ) {
            $column_id = strstr( $column_id, ':' ) ? current( explode( ':', $column_id ) ) : $column_id;
            $value     = '';

            // Skip some columns if dynamically handled later or if we're being selective.
            if ( in_array( $column_id, array( 'downloads', 'attributes', 'meta' ), true ) || ! $this->is_column_exporting( $column_id ) ) {
                continue;
            }

            try {
                if ( has_filter( "wcvendors_product_export_{$this->export_type}_column_{$column_id}" ) ) {
                    // Filter for 3rd parties.
                    $value = apply_filters( "wcvendors_product_export_{$this->export_type}_column_{$column_id}", '', $product, $column_id );

                } elseif ( is_callable( array( $this, "get_column_value_{$column_id}" ) ) ) {
                    // Handle special columns which don't map 1:1 to product data.
                    $value = $this->{"get_column_value_{$column_id}"}( $product );

                } elseif ( is_callable( array( $product, "get_{$column_id}" ) ) ) {
                    // Default and custom handling.
                    $value = $product->{"get_{$column_id}"}( 'edit' );
                }

                if ( 'description' === $column_id || 'short_description' === $column_id ) {
                    $value = $this->filter_description_field( $value );
                }

                $row[ $column_id ] = $value;
            } catch ( Exception $e ) {
                $this->log( "Error processing column '{$column_id}' for product ID {$product->get_id()}: " . $e->getMessage(), 'error' );
                $row[ $column_id ] = '';
            }
        }

        try {
            $this->prepare_downloads_for_export( $product, $row );
            $this->prepare_attributes_for_export( $product, $row );
            $this->prepare_meta_for_export( $product, $row );
        } catch ( Exception $e ) {
            $this->log( "Error preparing additional data for product ID {$product->get_id()}: " . $e->getMessage(), 'error' );
        }

        return apply_filters( 'wcvendors_product_export_row_data', $row, $product );
    }

    /**
     * Get published value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return int
     */
    protected function get_column_value_published( $product ) {
        $statuses = array(
            'draft'   => -1,
            'private' => 0,
            'publish' => 1,
        );

        // Fix display for variations when parent product is a draft.
        if ( 'variation' === $product->get_type() ) {
            $parent_id = $product->get_parent_id();
            if ( $parent_id ) {
                $parent = wc_get_product( $parent_id );
                if ( $parent ) {
                    $status = 'draft' === $parent->get_status() ? $parent->get_status() : $product->get_status( 'edit' );
                } else {
                    $this->log( "Parent product not found for variation ID {$product->get_id()}", 'warning' );
                    $status = $product->get_status( 'edit' );
                }
            } else {
                $this->log( "No parent ID found for variation ID {$product->get_id()}", 'warning' );
                $status = $product->get_status( 'edit' );
            }
        } else {
            $status = $product->get_status( 'edit' );
        }

        return isset( $statuses[ $status ] ) ? $statuses[ $status ] : -1;
    }

    /**
     * Get formatted sale price.
     *
     * @param WC_Product $product Product being exported.
     *
     * @return string
     */
    protected function get_column_value_sale_price( $product ) {
        return wc_format_localized_price( $product->get_sale_price( 'view' ) );
    }

    /**
     * Get formatted regular price.
     *
     * @param WC_Product $product Product being exported.
     *
     * @return string
     */
    protected function get_column_value_regular_price( $product ) {
        return wc_format_localized_price( $product->get_regular_price() );
    }

    /**
     * Get product_cat value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_category_ids( $product ) {
        $term_ids = $product->get_category_ids( 'edit' );
        return $this->format_term_ids( $term_ids, 'product_cat' );
    }

    /**
     * Get product_tag value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_tag_ids( $product ) {
        $term_ids = $product->get_tag_ids( 'edit' );
        return $this->format_term_ids( $term_ids, 'product_tag' );
    }

    /**
     * Get product_shipping_class value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_shipping_class_id( $product ) {
        $term_ids = $product->get_shipping_class_id( 'edit' );
        return $this->format_term_ids( $term_ids, 'product_shipping_class' );
    }

    /**
     * Get images value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_images( $product ) {
        $image_ids = array_merge( array( $product->get_image_id( 'edit' ) ), $product->get_gallery_image_ids( 'edit' ) );
        $images    = array();

        foreach ( $image_ids as $image_id ) {
            $image = wp_get_attachment_image_src( $image_id, 'full' );

            if ( $image ) {
                $images[] = $image[0];
            }
        }

        return $this->implode_values( $images );
    }

    /**
     * Prepare linked products for export.
     *
     * @param int[] $linked_products Array of linked product ids.
     *
     * @since  1.8.0
     * @return string
     */
    protected function prepare_linked_products_for_export( $linked_products ) {
        $product_list = array();

        foreach ( $linked_products as $linked_product_id ) {
            $linked_product = wc_get_product( $linked_product_id );

            if ( ! $linked_product ) {
                $this->log( "Linked product ID {$linked_product_id} not found", 'warning' );
                continue;
            }

            if ( $linked_product->get_sku() ) {
                $product_list[] = $linked_product->get_sku();
            } else {
                $product_list[] = 'id:' . $linked_product->get_id();
            }
        }

        return $this->implode_values( $product_list );
    }

    /**
     * Get cross_sell_ids value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_cross_sell_ids( $product ) {
        $cross_sell_ids = $product->get_cross_sell_ids( 'edit' );
        return $this->prepare_linked_products_for_export( array_filter( (array) $cross_sell_ids ) );
    }

    /**
     * Get upsell_ids value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_upsell_ids( $product ) {
        $upsell_ids = $product->get_upsell_ids( 'edit' );
        return $this->prepare_linked_products_for_export( array_filter( (array) $upsell_ids ) );
    }

    /**
     * Get parent_id value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_parent_id( $product ) {
        if ( $product->get_parent_id( 'edit' ) ) {
            $parent = wc_get_product( $product->get_parent_id( 'edit' ) );
            if ( ! $parent ) {
                return '';
            }

            return $parent->get_sku( 'edit' ) ? $parent->get_sku( 'edit' ) : 'id:' . $parent->get_id();
        }
        return '';
    }

    /**
     * Get grouped_products value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_grouped_products( $product ) {
        if ( 'grouped' !== $product->get_type() ) {
            return '';
        }

        $grouped_products = array();
        $child_ids        = $product->get_children( 'edit' );
        foreach ( $child_ids as $child_id ) {
            $child = wc_get_product( $child_id );
            if ( ! $child ) {
                continue;
            }

            $grouped_products[] = $child->get_sku( 'edit' ) ? $child->get_sku( 'edit' ) : 'id:' . $child_id;
        }
        return $this->implode_values( $grouped_products );
    }

    /**
     * Get download_limit value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_download_limit( $product ) {
        return $product->is_downloadable() && $product->get_download_limit( 'edit' ) ? $product->get_download_limit( 'edit' ) : '';
    }

    /**
     * Get download_expiry value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_download_expiry( $product ) {
        return $product->is_downloadable() && $product->get_download_expiry( 'edit' ) ? $product->get_download_expiry( 'edit' ) : '';
    }

    /**
     * Get stock value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_stock( $product ) {
        $manage_stock   = $product->get_manage_stock( 'edit' );
        $stock_quantity = $product->get_stock_quantity( 'edit' );

        if ( $product->is_type( 'variation' ) && 'parent' === $manage_stock ) {
            return 'parent';
        } elseif ( $manage_stock ) {
            return $stock_quantity;
        } else {
            return '';
        }
    }

    /**
     * Get stock status value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_stock_status( $product ) {
        $status = $product->get_stock_status( 'edit' );

        if ( 'onbackorder' === $status ) {
            return 'backorder';
        }

        return 'instock' === $status ? 1 : 0;
    }

    /**
     * Get backorders.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_backorders( $product ) {
        $backorders = $product->get_backorders( 'edit' );

        switch ( $backorders ) {
            case 'notify':
                return 'notify';
            default:
                return wc_string_to_bool( $backorders ) ? 1 : 0;
        }
    }

    /**
     * Get low stock amount value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  3.5.0
     * @return int|string Empty string if value not set
     */
    protected function get_column_value_low_stock_amount( $product ) {
        return $product->managing_stock() && $product->get_low_stock_amount( 'edit' ) ? $product->get_low_stock_amount( 'edit' ) : '';
    }

    /**
     * Get type value.
     *
     * @param WC_Product $product Product being exported.
     *
     * @since  1.8.0
     * @return string
     */
    protected function get_column_value_type( $product ) {
        $types   = array();
        $types[] = $product->get_type();

        if ( $product->is_downloadable() ) {
            $types[] = 'downloadable';
        }

        if ( $product->is_virtual() ) {
            $types[] = 'virtual';
        }

        return $this->implode_values( $types );
    }

    /**
     * Filter description field for export.
     * Convert newlines to '\n'.
     *
     * @param string $description Product description text to filter.
     *
     * @since  3.5.4
     * @return string
     */
    protected function filter_description_field( $description ) {
        $description = str_replace( '\n', "\\\\n", $description );
        $description = str_replace( "\n", '\n', $description );
        return $description;
    }
    /**
     * Export downloads.
     *
     * @param WC_Product $product Product being exported.
     * @param array      $row     Row being exported.
     *
     * @since 1.8.0
     */
    protected function prepare_downloads_for_export( $product, &$row ) {
        if ( $product->is_downloadable() && $this->is_column_exporting( 'downloads' ) ) {
            $downloads = $product->get_downloads( 'edit' );

            if ( $downloads ) {
                $this->log( 'Processing ' . count( $downloads ) . " downloads for product ID {$product->get_id()}", 'info' );

                $i = 1;
                foreach ( $downloads as $download ) {
                    try {
                        /* translators: %s: download number */
                        $this->column_names[ 'downloads:name' . $i ] = sprintf( __( 'Download %d name', 'wcvendors-pro' ), $i );
                        /* translators: %s: download number */
                        $this->column_names[ 'downloads:url' . $i ] = sprintf( __( 'Download %d URL', 'wcvendors-pro' ), $i );
                        $row[ 'downloads:name' . $i ]               = $download->get_name();
                        $row[ 'downloads:url' . $i ]                = $download->get_file();
                        ++$i;
                    } catch ( Exception $e ) {
                        $this->log( "Error processing download {$i} for product ID {$product->get_id()}: " . $e->getMessage(), 'error' );
                    }
                }
            }
        }
    }

    /**
     * Export attributes data.
     *
     * @param WC_Product $product Product being exported.
     * @param array      $row     Row being exported.
     *
     * @since 1.8.0
     */
    protected function prepare_attributes_for_export( $product, &$row ) {
        if ( $this->is_column_exporting( 'attributes' ) ) {
            $attributes         = $product->get_attributes();
            $default_attributes = $product->get_default_attributes();

            if ( count( $attributes ) ) {
                $this->log( 'Processing ' . count( $attributes ) . " attributes for product ID {$product->get_id()}", 'info' );

                $i = 1;
                foreach ( $attributes as $attribute_name => $attribute ) {
                    try {
                        /* translators: %s: attribute number */
                        $this->column_names[ 'attributes:name' . $i ] = sprintf( __( 'Attribute %d name', 'wcvendors-pro' ), $i );
                        /* translators: %s: attribute number */
                        $this->column_names[ 'attributes:value' . $i ] = sprintf( __( 'Attribute %d value(s)', 'wcvendors-pro' ), $i );
                        /* translators: %s: attribute number */
                        $this->column_names[ 'attributes:visible' . $i ] = sprintf( __( 'Attribute %d visible', 'wcvendors-pro' ), $i );
                        /* translators: %s: attribute number */
                        $this->column_names[ 'attributes:taxonomy' . $i ] = sprintf( __( 'Attribute %d global', 'wcvendors-pro' ), $i );

                        if ( is_a( $attribute, 'WC_Product_Attribute' ) ) {
                            $row[ 'attributes:name' . $i ] = wc_attribute_label( $attribute->get_name(), $product );

                            if ( $attribute->is_taxonomy() ) {
                                $terms  = $attribute->get_terms();
                                $values = array();

                                foreach ( $terms as $term ) {
                                    $values[] = $term->name;
                                }

                                $row[ 'attributes:value' . $i ]    = $this->implode_values( $values );
                                $row[ 'attributes:taxonomy' . $i ] = 1;
                            } else {
                                $row[ 'attributes:value' . $i ]    = $this->implode_values( $attribute->get_options() );
                                $row[ 'attributes:taxonomy' . $i ] = 0;
                            }

                            $row[ 'attributes:visible' . $i ] = $attribute->get_visible();
                        } else {
                            $row[ 'attributes:name' . $i ] = wc_attribute_label( $attribute_name, $product );

                            if ( 0 === strpos( $attribute_name, 'pa_' ) ) {
                                $option_term = get_term_by( 'slug', $attribute, $attribute_name ); // @codingStandardsIgnoreLine.
                                $row[ 'attributes:value' . $i ]    = $option_term && ! is_wp_error( $option_term ) ? str_replace( ',', '\\,', $option_term->name ) : str_replace( ',', '\\,', $attribute );
                                $row[ 'attributes:taxonomy' . $i ] = 1;
                            } else {
                                $row[ 'attributes:value' . $i ]    = str_replace( ',', '\\,', $attribute );
                                $row[ 'attributes:taxonomy' . $i ] = 0;
                            }

                            $row[ 'attributes:visible' . $i ] = '';
                        }

                        if ( $product->is_type( 'variable' ) && isset( $default_attributes[ sanitize_title( $attribute_name ) ] ) ) {
                            /* translators: %s: attribute number */
                            $this->column_names[ 'attributes:default' . $i ] = sprintf( __( 'Attribute %d default', 'wcvendors-pro' ), $i );
                            $default_value                                   = $default_attributes[ sanitize_title( $attribute_name ) ];

                            if ( 0 === strpos( $attribute_name, 'pa_' ) ) {
                                $option_term = get_term_by( 'slug', $default_value, $attribute_name ); // @codingStandardsIgnoreLine.
                                $row[ 'attributes:default' . $i ] = $option_term && ! is_wp_error( $option_term ) ? $option_term->name : $default_value;
                            } else {
                                $row[ 'attributes:default' . $i ] = $default_value;
                            }
                        }
                        ++$i;
                    } catch ( Exception $e ) {
                        $this->log( "Error processing attribute {$attribute_name} for product ID {$product->get_id()}: " . $e->getMessage(), 'error' );
                    }
                }
            }
        }
    }

    /**
     * Export meta data.
     *
     * @param WC_Product $product Product being exported.
     * @param array      $row Row data.
     *
     * @since 1.8.0
     */
    protected function prepare_meta_for_export( $product, &$row ) {
        if ( $this->enable_meta_export ) {
            $meta_data = $product->get_meta_data();

            if ( count( $meta_data ) ) {
                $this->log( 'Processing ' . count( $meta_data ) . " meta fields for product ID {$product->get_id()}", 'info' );

                $meta_keys_to_skip = apply_filters( 'woocommerce_product_export_skip_meta_keys', array(), $product );

                $i = 1;
                foreach ( $meta_data as $meta ) {
                    try {
                        if ( in_array( $meta->key, $meta_keys_to_skip, true ) ) {
                            continue;
                        }

                        // Allow 3rd parties to process the meta, e.g. to transform non-scalar values to scalar.
                        $meta_value = apply_filters( 'woocommerce_product_export_meta_value', $meta->value, $meta, $product, $row );

                        if ( ! is_scalar( $meta_value ) ) {
                            $this->log( "Skipping non-scalar meta value for key '{$meta->key}' in product ID {$product->get_id()}", 'warning' );
                            continue;
                        }

                        $column_key = 'meta:' . esc_attr( $meta->key );
                        /* translators: %s: meta data name */
                        $this->column_names[ $column_key ] = sprintf( __( 'Meta: %s', 'wcvendors-pro' ), $meta->key );
                        $row[ $column_key ]                = $meta_value;
                        ++$i;
                    } catch ( Exception $e ) {
                        $this->log( "Error processing meta field '{$meta->key}' for product ID {$product->get_id()}: " . $e->getMessage(), 'error' );
                    }
                }
            }
        }
    }
}
