<?php
/**
 * Admin Order Controller functionality
 *
 * This class handles adding vendor columns to the WooCommerce Orders admin page.
 *
 * @package    WCVendors_Pro
 * @since      1.9.9
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
    die;
}

/**
 * Class for admin order controller functionality
 *
 * @version 1.9.9
 * @since   1.9.9
 */
class WCVendors_Pro_Admin_Order_Controller {

    /**
     * The ID of this plugin.
     *
     * @since  1.9.9
     * @access private
     * @var    string $wcvendors_pro The ID of this plugin.
     */
    private $wcvendors_pro;

    /**
     * The version of this plugin.
     *
     * @since  1.9.9
     * @access private
     * @var    string $version The current version of this plugin.
     */
    private $version;

    /**
     * Is the plugin in debug mode
     *
     * @since  1.9.9
     * @access private
     * @var    bool $debug plugin is in debug mode
     */
    private $debug;

    /**
     * Initialize the class and set its properties.
     *
     * @since 1.9.9
     * @param string $wcvendors_pro The name of this plugin.
     * @param string $version       The version of this plugin.
     * @param bool   $debug         Is the plugin in debug mode.
     */
    public function __construct( $wcvendors_pro, $version, $debug ) {
        $this->wcvendors_pro = $wcvendors_pro;
        $this->version       = $version;
        $this->debug         = $debug;
    }

    /**
     * Add a custom column to the orders list in the admin.
     *
     * @param array $columns Array of columns.
     * @return array Modified array of columns.
     * @since 1.9.9
     */
    public function add_vendor_column_to_orders_list( $columns ) {
        // Add a new column for the vendor (after the 'order actions' column).
        $columns['vendor_order'] = __( 'Vendors', 'wcvendors-pro' );
        return $columns;
    }

    /**
     * Display the vendor in the custom column in the orders list in the admin.
     *
     * @param string $column   Column name.
     * @param int    $order_id Order ID.
     * @since 1.9.9
     */
    public function display_vendor_in_orders_list( $column, $order_id ) {
        if ( 'vendor_order' === $column ) {
            // Get the vendor for this order, depending on how your vendors are managed.
            $order = wc_get_order( $order_id );

            if ( ! $order ) {
                return;
            }

            $vendor_ids = WCV_Vendors::get_vendor_ids_from_order( $order );

            if ( empty( $vendor_ids ) ) {
                echo esc_html__( 'No vendors', 'wcvendors-pro' );
                return;
            }

            $vendor_links = array();
            foreach ( $vendor_ids as $vendor_id ) {
                $link = $this->get_vendor_shop_link( $vendor_id );
                if ( ! empty( $link ) ) {
                    $vendor_links[] = $link;
                }
            }

            if ( ! empty( $vendor_links ) ) {
                echo wp_kses_post( implode( '<br/>', $vendor_links ) );
            } else {
                echo esc_html__( 'No vendors', 'wcvendors-pro' );
            }
        }
    }

    /**
     * Get the vendor shop link
     *
     * @param int $vendor_id The vendor ID.
     * @return string
     */
    private function get_vendor_shop_link( $vendor_id ) {
        if ( empty( $vendor_id ) || ! is_numeric( $vendor_id ) ) {
            return '';
        }

        $vendor_id = absint( $vendor_id );

        $shop_name = WCV_Vendors::get_vendor_shop_name( $vendor_id );
        $shop_url  = WCV_Vendors::get_vendor_shop_page( $vendor_id );

        if ( empty( $shop_url ) || empty( $shop_name ) ) {
            $user_edit_url = admin_url( 'user-edit.php?user_id=' . absint( $vendor_id ) );
            $display_name  = ! empty( $shop_name ) ? $shop_name : get_userdata( $vendor_id )->display_name ?? __( 'Unknown Vendor', 'wcvendors-pro' );

            return sprintf(
                '<a href="%s">%s</a>',
                esc_url( $user_edit_url ),
                esc_html( $display_name )
            );
        }

        return sprintf(
            '<a href="%s" target="_blank">%s</a>',
            esc_url( $shop_url ),
            esc_html( $shop_name )
        );
    }
}
