<?php

/**
 * WC Vendors Pro Admin functions
 *
 * Functions for the admin interface
 *
 * @package WCVendors Pro/Functions
 * @version 1.5.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Admin checkbox field
 *
 * @param array   $value The field attributes.
 * @param boolean $required Whether the field is required.
 * @return void
 */
function wcv_admin_checkbox( $value, $required = false ) {

    if ( ! isset( $value['type'] ) ) {
        return;
    }
    if ( ! isset( $value['id'] ) ) {
        $value['id'] = '';
    }
    if ( ! isset( $value['title'] ) ) {
        $value['title'] = isset( $value['name'] ) ? $value['name'] : '';
    }
    if ( ! isset( $value['class'] ) ) {
        $value['class'] = '';
    }
    if ( ! isset( $value['css'] ) ) {
        $value['css'] = '';
    }
    if ( ! isset( $value['default'] ) ) {
        $value['default'] = '';
    }
    if ( ! isset( $value['desc'] ) ) {
        $value['desc'] = '';
    }
    if ( ! isset( $value['desc_tip'] ) ) {
        $value['desc_tip'] = false;
    }
    if ( ! isset( $value['placeholder'] ) ) {
        $value['placeholder'] = '';
    }
    if ( ! isset( $value['suffix'] ) ) {
        $value['suffix'] = '';
    }

    if ( $required && ! isset( $value ['required_id'] ) ) {
        return;
    }

    if ( $required && isset( $value['required_id'] ) && ! empty( $value['required_id'] ) ) {
        $value['id'] = $value['required_id'];
    }

    // Custom attribute handling.
    $custom_attributes = array();

    if ( ! empty( $value['custom_attributes'] ) && is_array( $value['custom_attributes'] ) ) {
        foreach ( $value['custom_attributes'] as $attribute => $attribute_value ) {
            $custom_attributes[] = esc_attr( $attribute ) . '="' . esc_attr( $attribute_value ) . '"';
        }
    }

    $option_value      = WCVendors_Admin_Settings::get_option( $value['id'], $value['default'] );
    $visibility_class  = array();
    $field_description = WCVendors_Admin_Settings::get_field_description( $value );
    extract( $field_description ); // phpcs:ignore

    if ( ! empty( $value['title'] ) ) { ?>
        <legend class="screen-reader-text"><span><?php echo esc_html( $value['title'] ); ?></span></legend>

        <?php
    }
    ?>
    <input
            name="<?php echo esc_attr( $value['id'] ); ?>"
            id="<?php echo esc_attr( $value['id'] ); ?>"
            type="checkbox"
            class="<?php echo esc_attr( isset( $value['class'] ) ? $value['class'] : '' ); ?> wcv_admin_checkbox"
            value="1"
        <?php checked( $option_value, 'yes' ); ?>
        <?php echo esc_attr( implode( ' ', $custom_attributes ) ); ?>
    />

    <?php
}

if ( ! function_exists( 'wcv_vendor_select_field' ) ) {
    /**
     * Vendor select field
     *
     * @param array $args The arguments used to generate the field.
     * @param bool  $output_type The type of output. echo|return, default = return.
     * @return string
     * @version 2.5.0
     * @since   2.5.0
     */
    function wcv_vendor_select_field( $args, $output_type = 'return' ) {
        $args = wp_parse_args(
            $args,
            array(
                'field_class'   => '',
                'id'            => 'post_author',
                'placeholder'   => sprintf( '__ %s __', __( 'No change', 'wc-vendors' ) ),
                'selected'      => '',
                'label'         => '',
                'wrapper_class' => 'form-field post_author_field',
                'label_class'   => '',
                'user_args'     => array(),
                'wrapper'       => 'p',
            )
        );

        $is_select2 = isset( $args['is_select2'] ) ? $args['is_select2'] : true;

        /**
         * Filter the arguments used to render the select field.
         *
         * @param array $args The arguments to be filtered.
         */
        $args = apply_filters( 'wcvendors_vendor_select_field_args', $args );

        $field_class   = $args['field_class'];
        $id            = $args['id'];
        $placeholder   = $args['placeholder'];
        $selected      = $args['selected'];
        $label         = $args['label'];
        $wrapper_class = $args['wrapper_class'];
        $label_class   = $args['label_class'];

        $user_args = wp_parse_args(
            $args['user_args'],
            array(
                'fields'   => array( 'ID', 'display_name' ),
                'role__in' => array( 'vendor', 'administrator' ),
                'number'   => 100,
            )
        );

        if ( $selected ) {
            $user_args['include'] = array( $selected );
        }

        /**
         * Filter the arguments used to search for vendors.
         *
         * @param array $user_args The arguments to be filtered.
         */
        $user_args = apply_filters( 'wcvendors_vendor_select_user_query_args', $user_args );
        $users     = get_users( $user_args );
        $_class    = $is_select2 ? implode( ' ', array( 'wcv-vendor-select', $field_class ) ) : $field_class;
        ob_start();

        ?>
        <select
            style="width:200px;"
            name="<?php echo esc_attr( $id ); ?>"
            id="<?php echo esc_attr( $id ); ?>"
            class="<?php echo esc_attr( $_class ); ?>">
            <option value=""><?php echo esc_attr( $placeholder ); ?></option>
            <?php
            foreach ( (array) $users as $user ) {
                ?>
                <option
                    value="<?php echo esc_attr( $user->ID ); ?>" <?php selected( $user->ID, $selected, true ); ?>>
                    <?php echo esc_attr( $user->display_name ); ?>
                </option>
                <?php
            }
            ?>
        </select>
        <?php

        $output = ob_get_clean();

        $output = apply_filters( 'wcvendors_vendor_select_field_output', $output, $args, $user_args );

        if ( 'echo' === $output_type ) {
            echo wp_kses( $output, wcv_allowed_html_tags() );
        }

        return $output;
    }
}

if ( ! function_exists( 'wcv_vendor_selectbox' ) ) {
    /**
     * Create a select box to display vendor & administrator roles
     *
     * @version 2.4.3
     * @since   2.4.3
     * @param array $args  Arguments used to render user dropdown box.
     * @param bool  $media Whether to display assign media checkbox.
     * @param bool  $is_select2 Whether to use select2.
     *
     * @return string
     */
    function wcv_vendor_selectbox( $args, $media = true, $is_select2 = true ) {
        $args = wp_parse_args(
            $args,
            array(
                'class'       => '',
                'id'          => 'post_author',
                'placeholder' => sprintf( '__ %s __', __( 'No change', 'wc-vendors' ) ),
                'selected'    => '',
            )
        );

        /**
         * Filter the arguments used to render the selectbox.
         *
         * @param array $args The arguments to be filtered.
         */
        $args = apply_filters_deprecated( 'wcv_vendor_selectbox_args', array( $args ), '2.3.0', 'wcvendors_vendor_selectbox_args' );
        $args = apply_filters( 'wcvendors_vendor_selectbox_args', $args );

        extract( $args ); // phpcs:ignore

        $user_args = array(
            'fields'   => array( 'ID', 'display_name' ),
            'role__in' => array( 'vendor', 'administrator' ),
            'number'   => 100,
        );

        if ( $selected && '' !== $selected ) {
            $user_args['include'] = array( $selected );
        } elseif ( isset( $args['authors'] ) ) {
            $user_args['include'] = $args['authors'];
        }

        $args['is_select2']  = $is_select2;
        $args['field_class'] = $class;

        /**
         * Filter the arguments used to search for vendors.
         *
         * @param array $user_args The arguments to be filtered.
         */
        $user_args = apply_filters_deprecated( 'wcv_vendor_selectbox_user_args', array( $user_args ), '2.3.0', 'wcvendors_vendor_selectbox_user_args' );
        $user_args = apply_filters( 'wcvendors_vendor_selectbox_user_args', $user_args );

        $field_args              = $args;
        $field_args['user_args'] = $user_args;

        $output = wcv_vendor_select_field( $field_args );

        if ( $media ) {
            $output .= '<p><label class="product_media_author_override">';
            $output .= '<input name="product_media_author_override" type="checkbox" /> ';
            $output .= sprintf(
                // translators: %s is the name used to refer to a vendor.
                __( 'Assign media to %s', 'wc-vendors' ),
                wcv_get_vendor_name()
            );
            $output .= '</label></p>';
        }

        $output = apply_filters_deprecated( 'wcv_vendor_selectbox', array( $output, $user_args, $media ), '2.3.0', 'wcvendors_vendor_selectbox' );
        return apply_filters( 'wcvendors_vendor_selectbox', $output, $user_args, $media );
    }
}
