/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps } from '@wordpress/block-editor';
import { SelectControl, TextControl, CheckboxControl } from '@wordpress/components';


/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( { attributes, setAttributes } ) {
	const { title, orderBy, showAsDropdown, showCount, hideEmpty, showChildrenOnly, showOnStore, showHierarchical } = attributes;
	return (
		<div { ...useBlockProps() }>
			<TextControl
				label={ __( 'Title', 'wcvendors-pro' ) }
				value={ title }
				onChange={ ( title ) => setAttributes( { title } ) }
			/>
			<SelectControl
				label={ __( 'Order by', 'wcvendors-pro' ) }
				value={ orderBy }
				onChange={ ( orderBy ) => setAttributes( { orderBy } ) }
				options={ [
					{ label: __( 'Name', 'wcvendors-pro' ), value: 'name' },
					{ label: __( 'Category order', 'wcvendors-pro' ), value: 'order' },
				] }
			/>
			<CheckboxControl
				label={ __( 'Show as dropdown', 'wcvendors-pro' ) }
				checked={ showAsDropdown }
				onChange={ ( showAsDropdown ) => setAttributes( { showAsDropdown } ) }
			/>
			<CheckboxControl
				label={ __( 'Show count', 'wcvendors-pro' ) }
				checked={ showCount }
				onChange={ ( showCount ) => setAttributes( { showCount } ) }
			/>
			<CheckboxControl
				label={ __( 'Hide empty', 'wcvendors-pro' ) }
				checked={ hideEmpty }
				onChange={ ( hideEmpty ) => setAttributes( { hideEmpty } ) }
			/>
			<CheckboxControl
				label={ __( 'Show children only', 'wcvendors-pro' ) }
				checked={ showChildrenOnly }
				onChange={ ( showChildrenOnly ) => setAttributes( { showChildrenOnly } ) }
			/>
			<CheckboxControl
				label={ __( 'Show on store', 'wcvendors-pro' ) }
				checked={ showOnStore }
				onChange={ ( showOnStore ) => setAttributes( { showOnStore } ) }
			/>
			<CheckboxControl
				label={ __( 'Show hierarchical', 'wcvendors-pro' ) }
				checked={ showHierarchical }
				onChange={ ( showHierarchical ) => setAttributes( { showHierarchical } ) }
			/>
		</div>
	);
}
