/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps } from '@wordpress/block-editor';
import { TextControl, CheckboxControl, SelectControl } from '@wordpress/components';


/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( { attributes, setAttributes } ) {
	const {
		title,
		showPhone,
		showShopName,
		showShopUrl,
		showEmailAddress,
		showContactText,
		showQuickContactForm,
		showOpeningHours,
		showOpeningHoursAfter,
		contactText,
		ccAdmin,
		requiredQuickContactFormSubject,
		requiredQuickContactFormEmailAddress,
		requiredQuickContactFormMessage,
	} = attributes;
	return (
		<div { ...useBlockProps() }>
			<TextControl
				label={ __( 'Title' ) }
				value={ title }
				onChange={ ( value ) => setAttributes( { title: value } ) }
			/>

			<CheckboxControl
				label={ __( 'Show contact us text' ) }
				checked={ showContactText }
				onChange={ ( value ) => setAttributes( { showContactText: value } ) }
			/>

			{ showContactText && (
					<TextControl
					label={ __( 'Contact us text' ) }
					value={ contactText }
					onChange={ ( value ) => setAttributes( { contactText: value } ) }
				/>
			) }

			<CheckboxControl
				label={ __( 'Show shop name' ) }
				checked={ showShopName }
				onChange={ ( value ) => setAttributes( { showShopName: value } ) }
			/>

			<CheckboxControl
				label={ __( 'Show shop URL' ) }
				checked={ showShopUrl }
				onChange={ ( value ) => setAttributes( { showShopUrl: value } ) }
			/>

			<CheckboxControl
				label={ __( 'Show phone number' ) }
				checked={ showPhone }
				onChange={ ( value ) => setAttributes( { showPhone: value } ) }
			/>

			<CheckboxControl
				label={ __( 'Show email address' ) }
				checked={ showEmailAddress }
				onChange={ ( value ) => setAttributes( { showEmailAddress: value } ) }
			/>

			<CheckboxControl
				label={ __( 'Show quick contact form' ) }
				checked={ showQuickContactForm }
				onChange={ ( value ) => setAttributes( { showQuickContactForm: value } ) }
			/>

			<CheckboxControl
				label={ __( 'Show opening hours' ) }
				checked={ showOpeningHours }
				onChange={ ( value ) => setAttributes( { showOpeningHours: value } ) }
			/>
			{ showOpeningHours && (
				<SelectControl
					label={ __( 'Show hours after' ) }
					value={ showOpeningHoursAfter }
					onChange={ ( value ) => setAttributes( { showOpeningHoursAfter: value } ) }
					options={ [
						{ label: __( 'Contact us text' ), value: 'text' },
						{ label: __( 'Contact info' ), value: 'info' },
						{ label: __( 'Quick contact form' ), value: 'form'}
					] }
				/>
			) }

			{ showQuickContactForm && (
				<>
					<CheckboxControl
						label={ __( 'Send copy of quick contact form to admin' ) }
						checked={ ccAdmin }
						onChange={ ( value ) => setAttributes( { ccAdmin: value } ) }
						disabled={ ! showQuickContactForm }
					/>

					<CheckboxControl
						label={ __( 'Required quick contact form subject' ) }
						checked={ requiredQuickContactFormSubject }
						onChange={ ( value ) => setAttributes( { requiredQuickContactFormSubject: value } ) }
						disabled={ ! showQuickContactForm }
					/>

					<CheckboxControl
						label={ __( 'Required quick contact form email address' ) }
						checked={ requiredQuickContactFormEmailAddress }
						onChange={ ( value ) => setAttributes( { requiredQuickContactFormEmailAddress: value } ) }
						disabled={ ! showQuickContactForm }
					/>

					<CheckboxControl
						label={ __( 'Required quick contact form message' ) }
						checked={ requiredQuickContactFormMessage }
						onChange={ ( value ) => setAttributes( { requiredQuickContactFormMessage: value } ) }
						disabled={ ! showQuickContactForm }
					/>
				</>
			) }


		</div>
	);
}
