<?php

use WC_Vendors\Classes\Front\WCV_Vendor_Controller;

/**
 * The WCVendors Pro Commission Controller class
 *
 * This is the Commission controller class for all front end Commission management
 *
 * @package    WCVendors_Pro
 * @subpackage WCVendors_Pro/public
 * @author     Ketul Sheth <support@wcvendors.com>
 */
class WCVendors_Pro_Commission_Download_Controller { //phpcs:ignore

    /**
     * The ID of this plugin.
     *
     * @since    1.8.4
     * @access   private
     * @var      string $wcvendors_pro The ID of this plugin.
     */
    private $wcvendors_pro;

    /**
     * The version of this plugin.
     *
     * @since    1.8.4
     * @access   private
     * @var      string $version The current version of this plugin.
     */
    private $version;

    /**
     * Is the plugin in debug mode
     *
     * @since    1.8.4
     * @access   private
     * @var      bool $debug plugin is in debug mode
     */
    private $debug;

        /**
         * Max number of pages for pagination
         *
         * @since    1.9.3
         * @access   public
         * @var      int $max_num_pages integer for max number of pages for the query
         */
    public $max_num_pages;

    /**
     * Total number of orders for the query
     *
     * @since    1.9.3
     * @access   public
     * @var      int $total_order_count integer for total number of orders for the query
     */
    public $total_order_count;

    /**
     * Is the plugin base directory
     *
     * @since    1.8.4
     * @access   private
     * @var      string $base_dir string path for the plugin directory
     */
    private $base_dir;

    /**
     * The tables header rows
     *
     * @since    1.8.4
     * @access   private
     * @var      array $columns The table columns
     */
    private $columns;

    /**
     * The table rows
     *
     * @since    1.8.4
     * @access   private
     * @var      array $rows The table rows
     */
    private $rows;

    /**
     * The ID of this plugin.
     *
     * @since    1.8.4
     * @access   private
     * @var      string $wcvendors_pro The ID of this plugin.
     */
    private $controller_type;

    /**
     * Default start date.
     *
     * @var string
     * @version 1.8.8
     * @since   1.8.8
     */
    private $default_start;

    /**
     * Initialize the class and set its properties.
     *
     * @since 1.8.4
     *
     * @param string  $wcvendors_pro The name of the plugin.
     * @param string  $version       The version of this plugin.
     * @param boolean $debug         Whether debug is turned on or not.
     */
    public function __construct( $wcvendors_pro, $version, $debug ) {

        $this->wcvendors_pro    = $wcvendors_pro;
        $this->version          = $version;
        $this->debug            = $debug;
        $this->base_dir         = plugin_dir_path( __DIR__ );
        $this->controller_type  = 'commission';
        $this->default_start    = '';
        $commission_sales_range = get_option( 'wcvendors_commission_sales_range', 'monthly' );

        switch ( $commission_sales_range ) {
            case 'annually':
                $this->default_start = '-1 year';
                break;
            case 'quarterly':
                $this->default_start = '-3 month';
                break;
            case 'monthly':
                $this->default_start = '-1 month';
                break;
            case 'weekly':
                $this->default_start = '-1 week';
                break;
            case 'daily':
                $this->default_start = '-1 day';
                break;
            case 'custom':
                $this->default_start = '-1 year';
                break;
            default:
                $this->default_start = '-1 month';
                break;
        }

        $this->default_start = apply_filters( 'wcv_default_commission_start_date', $this->default_start );
    }

    /**
     * Display the custom Commission table
     *
     * @since    1.8.4
     */
    public function display() {
        // Use the internal table generator to create object list.
        $commission_table = new WCVendors_Pro_Table_Helper( $this->wcvendors_pro, $this->version, $this->controller_type, null, get_current_user_id() );
        $commission_table->set_columns( $this->table_columns() );
        $commission_table->set_rows( $this->table_rows() );
        // display the table.
        $commission_table->display();
    }

    /**
     *  Process the form submission from the front end.
     *
     * @since   1.8.4
     */
    public function process_submit() {

        if ( isset( $_GET['wcv_export_commission'] ) ) {

            $vendor_id = get_current_user_id();
            $this->export_csv();
        }

        if ( isset( $_POST['wcv_commission_id'] ) && isset( $_POST['wcv_add_note'] ) ) {

            if ( ! wp_verify_nonce( $_POST['wcv_add_note'], 'wcv-add-note' ) ) {
                return false;
            }

            $commission_id = (int) $_POST['wcv_commission_id'];
            $comment       = $_POST['wcv_comment_text'];

            if ( ! WCVendors_Pro_Dashboard::check_object_permission( 'commission', absint( $commission_id ) ) ) {
                return false;
            }

            if ( empty( $comment ) ) {
                wc_add_notice( __( 'You need type something in the note field', 'wcvendors-pro' ), 'error' );

                return false;
            }

            self::add_commission_note( $commission_id, $comment );
        }

        // Process the date updates for the form.
        if ( isset( $_POST['wcv_commission_date_update'] ) ) {

            if ( ! wp_verify_nonce( $_POST['wcv_commission_date_update'], 'wcv-commission-date-update' ) ) {
                return;
            }

            // Start Date.
            if ( isset( $_POST['_wcv_commission_start_date_input'] ) || '' === $_POST['_wcv_commission_start_date_input'] ) {
                WC()->session->set( 'wcv_commission_start_date', strtotime( $_POST['_wcv_commission_start_date_input'] ) );
            }

            // End Date.
            if ( isset( $_POST['_wcv_commission_end_date_input'] ) || '' === $_POST['_wcv_commission_end_date_input'] ) {
                WC()->session->set( 'wcv_commission_end_date', strtotime( $_POST['_wcv_commission_end_date_input'] ) );
            }
        }
    }

    /**
     *  Update Table Headers for display.
     *
     * @since 1.8.4
     *
     * @return array $headers array passed via filter.
     */
    public function table_columns() {

        $columns = apply_filters(
            'wcv_commission_table_columns',
            array(
                'ID'           => array(
                    'label'         => __( 'ID', 'wcvendors-pro' ),
                    'mobile_header' => false,
                ),
                'order_number' => array(
                    'label'         => __( 'Order', 'wcvendors-pro' ),
                    'mobile_header' => false,
                    'full_span'     => true,
                ),
                'products'     => array(
                    'label'         => __( 'Products', 'wcvendors-pro' ),
                    'mobile_header' => true,
                ),
                'commission'   => array(
                    'label'         => __( 'Commission', 'wcvendors-pro' ),
                    'mobile_header' => true,
                ),
                'status'       => array(
                    'label'         => __( 'Shipped', 'wcvendors-pro' ),
                    'mobile_header' => true,
                ),
                'order_date'   => array(
                    'label'         => __( 'Order Date', 'wcvendors-pro' ),
                    'mobile_header' => true,
                ),
            )
        );

        return $columns;
    }

    /**
     *  Create the table data
     *
     * @since    1.8.4
     * @version  1.8.4
     * @return   array  $new_rows   array of stdClass objects passed back to the filter
     */
    public function table_rows() {
        global $wpdb;
        $date_range       = array(
            'endate'    => date_i18n( 'Y-m-d', $this->get_end_date() ),
            'startdate' => date_i18n( 'Y-m-d', $this->get_start_date() ),
        );
        $allowed_statuses = array( 'wc-completed', 'wc-processing' );
        $vendor_id        = get_current_user_id();
        /**
         * Changing this to false will show refunded commissions no matter which payment gateway is used.
         *
         * @since 1.8.4
         */
        $show_refuned_order = wcv_is_show_reversed_order();

        if ( $show_refuned_order ) {
            $allowed_statuses[] = 'wc-refunded';
        }

        $limit  = apply_filters( 'wcv_commission_table_limit', get_option( 'wcvendors_commission_per_page', 20 ) );
        $paged  = ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;
        $offset = ( $paged - 1 ) * $limit;

        $vendor_orders = wc_get_orders(
            array(
                'limit'        => $limit,
                'offset'       => $offset,
                'status'       => $allowed_statuses,
                'type'         => 'shop_order_vendor',
                'vendor_id'    => $vendor_id,
                'date_created' => $date_range['startdate'] . '...' . $date_range['endate'],
            )
        );

        if ( empty( $vendor_orders ) ) {
            return array();
        }

        $count_total_orders_in_range = wc_get_orders(
            array(
                'limit'        => -1,
                'return'       => 'ids',
                'status'       => $allowed_statuses,
                'type'         => 'shop_order_vendor',
                'vendor_id'    => $vendor_id,
                'date_created' => $date_range['startdate'] . '...' . $date_range['endate'],
            )
        );

        $parent_ids = array_map(
            function ( $order ) {
                return $order->get_parent_id();
            },
            $vendor_orders
        );

        $vendor_ids_placeholder = implode( ',', array_fill( 0, count( $parent_ids ), '%d' ) );

        $commission_rows = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}pv_commission WHERE order_id IN ({$vendor_ids_placeholder}) AND vendor_id = %d", //phpcs:ignore
                array_merge( $parent_ids, array( $vendor_id ) )
            )
        );

        // use array reduce to group the commissions by order id.
        $commission_grouped = array_reduce(
            $commission_rows,
            function ( $carry, $item ) {
                $carry[ $item->order_id ][] = $item;

                return $carry;
            },
            array()
        );

        $this->total_order_count = count( $count_total_orders_in_range );
        $this->max_num_pages     = ceil( $this->total_order_count / $limit );

        $all_commission = array();

        foreach ( $vendor_orders as $order ) {
            $parent_id            = $order->get_parent_id();
            $parent_order         = wc_get_order( $parent_id );
            $_order               = new stdClass();
            $_order->order        = $order;
            $_order->total_due    = 0;
            $order_items          = $order->get_items();
            $_order->order_items  = $order_items;
            $commission_row_by_id = isset( $commission_grouped[ $parent_id ] ) ? $commission_grouped[ $parent_id ] : array();
            $shippers             = $parent_order->get_meta( 'wc_pv_shipped' );
            foreach ( $commission_row_by_id as $commission_row ) {
                $_order->total_due += $commission_row->total_due;
            }
            $shipped = false;
            if ( $shippers ) {
                $shipped = array_map( 'intval', $shippers );
                $shipped = in_array( $vendor_id, $shippers, true ) ? true : false;
            }

            $_order->shipped = $shipped;

            $all_commission[] = $_order;
        }

        $rows = array();

        if ( ! empty( $all_commission ) ) {

            foreach ( $all_commission as $_order ) {

                $order          = $_order->order;
                $products_html  = '';
                $needs_shipping = false;

                if ( ! empty( $_order->order_items ) ) {
                    $order_total = 0;

                    foreach ( $_order->order_items as $item ) {
                        $product_id     = $item->get_product_id();
                        $_product       = $item->get_product();
                        $needs_shipping = $product_id ? $_product->needs_shipping() : false;
                        $order_total   += $order->get_item_total( $item, false, true );
                        $item_name      = sprintf( '%s x %s ', $item->get_quantity(), $item->get_name() );
                        $products_html .= '<strong>' . $item_name . '</strong><br />';

                    }
                }

                $has_shipped    = $_order->shipped ? __( 'Yes', 'wcvendors-pro' ) : __( 'No', 'wcvendors-pro' );
                $shipped        = ( $needs_shipping ) ? $has_shipped : __( 'NA', 'wcvendors-pro' );
                $order_currency = $order->get_currency();
                $commission_due = sprintf( get_woocommerce_price_format(), get_woocommerce_currency_symbol( $order_currency ), number_format( $_order->total_due, 2 ) );
                $new_row        = new stdClass();
                $order_date     = $order->get_date_created();
                $order_details  = sprintf(
                    '<div class="order_id">#%s</div>',
                    $order->get_parent_id()
                );

                $new_row->ID           = $order->get_order_number();
                $new_row->order_number = $order_details;
                $new_row->products     = $products_html;
                $new_row->commission   = $commission_due;
                $new_row->status       = $shipped;
                $new_row->order_date   = date_i18n( get_option( 'date_format', 'F j, Y' ), ( $order_date->getOffsetTimestamp() ) );
                $rows[]                = $new_row;

            }
        } // check for commissions

        return apply_filters( 'wcv_commission_table_rows', $rows );
    }

    /**
     *  Change the column that actions are displayed in
     *
     * @since    1.8.4
     *
     * @param string $column column passed from filter.
     *
     * @return   string $new_column    new column passed back to filter
     */
    public function table_action_column( $column ) {
        return $column;
    }

    /**
     *  Add actions before and after the table
     *
     * @since    1.8.4
     */
    public function table_actions() {

        $search = isset( $_POST['wcv-search'] ) ? esc_attr( $_POST['wcv-search'] ) : ''; //phpcs:ignore
        $pagination_wrapper = apply_filters(
            'wcv_commission_paginate_wrapper',
            array(
                'wrapper_start' => '<nav class="woocommerce-pagination">',
                'wrapper_end'   => '</nav>',
            )
        );

        include apply_filters( 'wcvendors_pro_table_actions_path', 'partials/order/wcvendors-pro-commission-table-actions.php' );
    }

    /**
     * Change the column that actions are displayed in
     *
     * @param string $notice The notice to display.
     *
     * @since    1.8.4
     */
    public function table_no_data_notice( $notice ) {

        $notice = apply_filters( 'wcv_commission_table_no_data_notice', __( 'No Commission found.', 'wcvendors-pro' ) );

        return $notice;
    }

    /**
     * Commission Details Template
     *
     * @param object $order The order object.
     *
     * @since 1.8.4
     */
    public function commission_details_template( $order ) {

        $form = '';

        // Get line items.
        $order           = $order->order;
        $line_items      = $order->order_items;
        $is_order_refund = false;
        $total_refund    = 0;
        $order_currency  = $order->get_currency();
        foreach ( $line_items as $item_id => $item ) {

            $order_item_detail = array();
            $line_item_taxes   = array();
            // Check if this is a variation and get the parent id, this ensures that the correct vendor id is retrieved.
            $product_id                        = ( $item->get_variation_id() ) ? $item->get_product_id() : $item->get_product_id();
            $_product                          = $item->get_product();
            $item_qty                          = $item->get_quantity();
            $product_commission                = ( $item_qty > 1 ) ? $order->product_commissions[ $product_id ] / $item_qty : $order->product_commissions[ $product_id ];
            $sku                               = ( $_product && $_product->get_sku() ) ? esc_html( $_product->get_sku() ) . ' &ndash; ' : '';
            $_item_id                          = $item->get_id();
            $order_item_detail['product_name'] = ( $sku ) ? $sku . esc_html( $item->get_name() ) : esc_html( $item->get_name() );
            $order_item_detail['commission']   = wc_price( $product_commission, array( 'currency' => $order_currency ) );
            $order_item_detail['qty']          = ( $item->get_quantity() ) ? esc_html( $item->get_quantity() ) : '';
            $order_item_detail['cost']         = wc_price( $order->get_item_total( $item, false, true ), array( 'currency' => $order->get_currency() ) );
            $order_item_detail['total']        = wc_price( $item->get_total(), array( 'currency' => $order->get_currency() ) );
            $order_item_detail['item']         = $item;

        }

        ob_start();
        $form = ob_get_contents();
        ob_end_clean();
        return $form;
    }

    /**
     * Filter order item.
     *
     * @param array $formatted_meta The formatted meta.
     * @return array $formatted_data The formatted data.
     * @version 1.8.4
     * @since   1.8.4
     */
    public function filter_order_item_get_formatted_meta_data( $formatted_meta ) {

        $dashboard_page_id = get_option( 'wcvendors_vendor_dashboard_page_id' );

        if ( ! isset( $dashboard_page_id ) || ! is_page( $dashboard_page_id ) ) {
            return $formatted_meta;
        }

        $hide_meta = apply_filters(
            'wcvendors_hide_order_meta_data',
            array(
                'Sold By',
                __( get_option( 'wcvendors_label_sold_by', 'Sold By' ), 'wcvendors-pro' ), // phpcs:ignore
            )
        );

        // Filter any meta not to show.
        foreach ( $formatted_meta as $key => $meta ) {
            if ( in_array( $meta->key, $hide_meta, true ) ) {
                unset( $formatted_meta[ $key ] );
            }
        }

        return $formatted_meta;
    }


    /**
     *  Trigger the csv export
     *
     * @version 1.8.4
     * @since   1.8.4
     */
    public function export_csv() {

        include_once 'class-wcvendors-pro-commisson-export-helper.php';

        $date_range = array(
            'before' => gmdate( 'Y-m-d', $this->get_end_date() ),
            'after'  => gmdate( 'Y-m-d', $this->get_start_date() ),
        );

        $csv_output  = new WCVendors_Pro_Export_Helper_Commisson( $this->wcvendors_pro, $this->version, $this->debug );
        $csv_headers = $csv_output->get_export_headers();
        $orders      = WCV_Vendor_Controller::get_orders2( get_current_user_id(), $date_range, true );
        $rows        = $csv_output->format_orders_export( $orders );

        // Remove the ID column as its not required.
        unset( $csv_headers['ID'] );
        $csv_headers  = apply_filters( 'wcv_order_export_csv_headers', $csv_headers );
        $csv_rows     = apply_filters( 'wcv_order_export_csv_rows', $rows, $orders, $date_range );
        $csv_filename = apply_filters( 'wcv_order_export_csv_filename', 'orders' );

        $csv_output->download_csv( $csv_headers, $csv_rows, $csv_filename );
    } // download_csv

    /**
     * Get order start date.
     *
     * @return array|string
     */
    public function get_start_date() {
        return WC()->session->get( 'wcv_commission_start_date', strtotime( apply_filters( 'wcv_commission_start_date', $this->default_start ) ) );
    }

    /**
     * Get order end date.
     *
     * @return array|string
     */
    public function get_end_date() {
        return WC()->session->get( 'wcv_commission_end_date', strtotime( apply_filters( 'wcv_commission_end_date', 'now' ) ) );
    }

    /**
     * Get default start date.
     *
     * @return string
     */
    public function get_default_start_date() {
        return strtotime( apply_filters( 'wcv_commission_start_date', $this->default_start ) );
    }
}
