<?php
/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    WCVendors_Pro
 * @subpackage WCVendors_Pro/public
 * @author     Jamie Madden <support@wcvendors.com>
 */
class WCVendors_Pro_Public { //phpcs:ignore

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $wcvendors_pro The ID of this plugin.
     */
    private $wcvendors_pro;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $version The current version of this plugin.
     */
    private $version;

    /**
     * Is the plugin in debug mode
     *
     * @since    1.0.0
     * @access   private
     * @var      bool $debug plugin is in debug mode
     */
    private $debug;

    /**
     * Script suffix for debugging
     *
     * @since    1.0.0
     * @access   private
     * @var      string $suffix script suffix for including minified file versions
     */
    private $suffix;

    /**
     * Base dir
     *
     * @var string $base_dir Plugin base directory
     */
    private $base_dir;

    /**
     * Whether this is the dashboard page or not.
     *
     * @var boolean
     */
    private $is_dashboard_page = false;

    /**
     * The current page ID.
     *
     * @var integer
     */
    private $current_page_id = 0;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     *
     * @param      string $wcvendors_pro The name of the plugin.
     * @param      string $version       The version of this plugin.
     * @param      bool   $debug         Plugin is in debug mode.
     */
    public function __construct( $wcvendors_pro, $version, $debug ) {
        $this->current_page_id = get_queried_object_id();

        $this->wcvendors_pro = $wcvendors_pro;
        $this->version       = $version;
        $this->debug         = $debug;
        $this->base_dir      = plugin_dir_url( __FILE__ );
        $this->suffix        = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) || $this->debug ? '' : '.min';

        $this->is_dashboard_page = wcv_is_dashboard_page( $this->current_page_id );

        // Add limit text length for store description.
        add_filter( 'wcv_vendor_store_description', array( $this, 'add_store_description_limit' ) );
        add_filter( 'the_editor', array( $this, 'add_store_description_limit_editor' ), 20 );
        add_filter( 'update_user_metadata', array( $this, 'add_store_description_limit_update_user_metadata' ), 10, 3 );

        add_action( 'wcvendors_store_settings_saved', array( $this, 'redirect_to_settings_page' ), 99 );
    }

    /**
     * Redirect to settings page after saving store settings.
     *
     * @since 1.9.4
     * @version 1.9.4
     */
    public function redirect_to_settings_page() {
        wp_safe_redirect( WCV_Vendor_Dashboard::get_dashboard_page_url( 'settings' ) );
        exit;
    }

    /**
     * Add limit text length for store description on update_user_metadata.
     *
     * @param mixed  $value The value of the metadata.
     * @param int    $user_id The user ID.
     * @param string $meta_key The meta key.
     * @return mixed
     */
    public function add_store_description_limit_update_user_metadata( $value, $user_id, $meta_key ) {
        if ( 'pv_shop_description' === $meta_key && ! is_null( $value ) ) {
            $limit = get_option( 'wcvendors_store_description_limit', 250 );
            if ( function_exists( 'mb_substr' ) ) {
                $value = mb_substr( $value, 0, $limit, 'UTF-8' );
            } else {
                $value = substr( $value, 0, $limit );
            }
        }
        return $value;
    }

    /**
     * Register the stylesheets for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {

        $feedback_page_id = get_option( 'wcvendors_feedback_page_id', null );
        $current_page_id  = get_queried_object_id();
        $is_feedback_page = apply_filters( 'wcv_view_feedback', (int) $current_page_id === (int) $feedback_page_id ? true : false );

        // selectWoo style.
        wp_enqueue_style( 'select2' );

        // Store Style.
        if ( is_shop() || is_product() ) {
            wp_enqueue_style(
                'wcv-pro-store-style',
                apply_filters( 'wcv_pro_store_style', $this->base_dir . 'assets/css/store' . $this->suffix . '.css' ),
                array(),
                $this->version
            );

        }

        // Dashicons.
        wp_enqueue_style( 'dashicons' );

        // Dashboard styles.
        if ( wcv_is_dashboard_page( $current_page_id ) || $is_feedback_page ) {

            if ( is_user_logged_in() ) {
                // Dashboard Style.
                wp_enqueue_style(
                    'wcv-pro-dashboard',
                    apply_filters( 'wcv_pro_dashboard_style', $this->base_dir . 'assets/css/dashboard' . $this->suffix . '.css' ),
                    array(),
                    WCV_PRO_VERSION
                );
            }
        }

        // SVG Icon Styles.
        wp_enqueue_style(
            'wcv-pro-icons',
            $this->base_dir . 'assets/css/wcv-icons' . $this->suffix . '.css',
            array(),
            $this->version,
            'all'
        );
    }

    /**
     * Add custom wcvendors pro css classes
     *
     * @version  1.7.10
     * @since    1.0.0
     * @access   public
     *
     * @param array $classes - body css classes.
     *
     * @return array $classes - body css classes.
     */
    public function body_class( $classes ) {
        global $post;

        if ( ! $post ) {
            return $classes;
        }
        $dashboard_page_id = get_option( 'wcvendors_vendor_dashboard_page_id' );
        $feedback_page_id  = get_option( 'wcvendors_feedback_page_id' );

        // If the page is a 404 don't load anything.
        if ( is_404() ) {
            return $classes;
        }

        $pages = wcv_get_pages();

        $general_class = 'wcvendors wcvendors-pro wcvendors-page';
        if ( is_object( $post ) && $post->ID === $dashboard_page_id ) {
            $classes[] = $general_class;
        }

        if ( is_page( $dashboard_page_id ) ) {
            $classes[] = 'wcvendors-pro-dashboard wcvendors-is-single';
        }

        if ( is_page( $feedback_page_id ) ) {
            $classes[] = 'wcv-ratings-page';
        }

        foreach ( $pages as $slug => $_page_id ) {
            if ( is_page( $_page_id ) ) {
                $classes[] = "wcvendors-{$slug}-page";
            }
        }

        if ( is_object( $post ) && WCV_Vendors::is_vendor( $post->post_author ) && 'product' === $post->post_type ) {
            $classes[] = 'wcvendors-single-product wcvendors-product-page';
        }

        if ( WCV_Vendors::is_vendor_page() && is_archive() ) {
            $classes[] = 'wcvendors-store';
        }

        $page = get_query_var( 'page' );
        if ( $page ) {
            $classes[] = "wcvendors-{$page}-page";
        }

        $object = get_query_var( 'object' );
        if ( $object ) {
            if ( 'shop_coupon' === $object ) {
                $object = 'coupon';
            }
            $classes[] = "wcvendors-dashboard-{$object}-page";
        }

        if ( is_object( $post ) && has_shortcode( $post->post_content, 'wcv_pro_vendorslist' ) ) {
            $classes[] = 'wcvendors-vendorlist';
        }

        if ( is_object( $post ) && has_shortcode( $post->post_content, 'wcv_feedback_form' ) ) {
            $classes[] = 'wcvendors-feedback-form';
        }

        return $classes;
    }

    /**
     * Register the stylesheets for the public-facing side of the site.
     *
     * @since 1.8.7 - Extracted flatpickr and datepicker localization strings to separate methods.
     * @since   1.0.0
     * @version 1.8.0
     */
    public function enqueue_scripts() {
        global $post;

        $current_page_id         = get_the_ID();
        $file_display            = get_option( 'wcvendors_file_display', '' );
        $tag_separator           = get_option( 'wcvendors_tag_separator', '' );
        $category_limit          = get_option( 'wcvendors_category_limit', '' );
        $tag_limit               = get_option( 'wcvendors_tag_limit', '' );
        $feedback_page_id        = (int) get_option( 'wcvendors_feedback_page_id', null );
        $shipping_settings       = get_option( 'woocommerce_wcv_pro_vendor_shipping_settings', wcv_get_default_vendor_shipping() );
        $store_shipping_type     = get_user_meta( get_current_user_id(), '_wcv_shipping_type', true );
        $vendor_select           = wc_string_to_bool( get_option( 'wcvendors_vendor_select_shipping', 'no' ) );
        $shipping_type           = ( '' !== $store_shipping_type ) ? $store_shipping_type : $shipping_settings['shipping_system'];
        $shipping_methods        = WC()->shipping() ? WC()->shipping->load_shipping_methods() : array();
        $shipping_method_enabled = wcv_is_vendors_shipping_enabled();

        // Product reports scripts.
        if ( is_product() && is_user_logged_in() ) {
            wp_register_script(
                'wcv-product-reports-script',
                $this->base_dir . 'assets/js/product-reports' . $this->suffix . '.js',
                array( 'jquery' ),
                WCV_PRO_VERSION,
                true
            );
            wp_register_style(
                'wcv-product-reports-style',
                $this->base_dir . 'assets/css/product-reports' . $this->suffix . '.css',
                array(),
                WCV_PRO_VERSION
            );

            wp_enqueue_script( 'wcv-product-reports-script' );
            wp_enqueue_style( 'wcv-product-reports-style' );
        }

        $is_dashboard = $this->is_dashboard_page || ( is_a( $post, 'WP_Post' ) && has_shortcode( $post->post_content, 'wcv_pro_dashboard_nav' ) ) || ( is_a( $post, 'WP_Post' ) && has_shortcode( $post->post_content, 'wcv_pro_dashboard' ) ) || ( is_a( $post, 'WP_Post' ) && has_shortcode( $post->post_content, 'wcv_vendor_dashboard' ) );

        if ( $current_page_id === $feedback_page_id ) {
            wp_register_script( 'feedback', $this->base_dir . 'assets/js/feedback' . $this->suffix . '.js', array( 'jquery' ), WCV_PRO_VERSION, true );
            wp_enqueue_script( 'feedback' );

            $wcv_feedback_args = apply_filters(
                'wcv_frontend_feedback_strings',
                array(
                    'select_stars_message' => __( 'Please select your star rating.', 'wcvendors-pro' ),
                )
            );
            wp_localize_script( 'feedback', 'wcv_frontend_feedback', $wcv_feedback_args );
        }

        if ( ! $is_dashboard || ! is_user_logged_in() ) {
            return;
        }

        // Accounting.
        wp_register_script( 'accounting', $this->base_dir . 'assets/lib/accounting/accounting' . $this->suffix . '.js', array( 'jquery' ), '0.4.2', true );
        wp_localize_script( 'accounting', 'accounting_params', array( 'mon_decimal_point' => wc_get_price_decimal_separator() ) );

        // Order.
        wp_register_script( 'wcv-frontend-order', $this->base_dir . 'assets/js/order' . $this->suffix . '.js', array( 'jquery' ), WCV_PRO_VERSION, true );
        wp_enqueue_script( 'wcv-frontend-order' );

        wp_localize_script(
            'wcv-frontend-order',
            'wcv_pro_order',
            array(
                'confirm_shipped' => __( 'Are you sure the item was shipped?', 'wcvendors-pro' ),
            )
        );

        $decimal_separator  = wc_get_price_decimal_separator();
        $thousand_separator = wc_get_price_thousand_separator();

        if ( ! $thousand_separator ) {
            if ( '.' !== $decimal_separator ) {
                $thousand_separator = '.';
            } else {
                $thousand_separator = ',';
            }
        }

        // Country select.
        $country_select_args = array(
            'countries'                 => wp_json_encode( array_merge( WC()->countries->get_allowed_country_states(), WC()->countries->get_shipping_country_states() ) ),
            'i18n_select_state_text'    => esc_attr__( 'Select an option&hellip;', 'wcvendors-pro' ),
            'i18n_matches_1'            => _x( 'One country is available, press enter to select it.', 'enhanced select', 'wcvendors-pro' ),
            'i18n_matches_n'            => _x( '%qty% countries are available, use up and down arrow keys to navigate.', 'enhanced select', 'wcvendors-pro' ),
            'i18n_no_matches'           => _x( 'No matches found', 'enhanced select', 'wcvendors-pro' ),
            'i18n_ajax_error'           => _x( 'Loading failed', 'enhanced select', 'wcvendors-pro' ),
            'i18n_input_too_short_1'    => _x( 'Please enter 1 or more characters', 'enhanced select', 'wcvendors-pro' ),
            'i18n_input_too_short_n'    => _x( 'Please enter %qty% or more characters', 'enhanced select', 'wcvendors-pro' ),
            'i18n_input_too_long_1'     => _x( 'Please delete 1 character', 'enhanced select', 'wcvendors-pro' ),
            'i18n_input_too_long_n'     => _x( 'Please delete %qty% characters', 'enhanced select', 'wcvendors-pro' ),
            'i18n_selection_too_long_1' => _x( 'You can only select 1 country', 'enhanced select', 'wcvendors-pro' ),
            'i18n_selection_too_long_n' => _x( 'You can only select %qty% countries', 'enhanced select', 'wcvendors-pro' ),
            'i18n_load_more'            => _x( 'Loading more results&hellip;', 'enhanced select', 'wcvendors-pro' ),
            'i18n_searching'            => _x( 'Searching&hellip;', 'enhanced select', 'wcvendors-pro' ),
        );

        wp_register_script( 'wcv-country-select', $this->base_dir . '../includes/assets/js/country-select' . $this->suffix . '.js', array( 'jquery', 'select2' ), WCV_PRO_VERSION, true );
        wp_localize_script( 'wcv-country-select', 'wcv_country_select_params', $country_select_args );
        wp_enqueue_script( 'wcv-country-select' );

        $current_page = get_query_var( 'object' );

        $google_map_api_key = apply_filters( 'wcvendors_pro_google_maps_api_key', get_option( 'wcvendors_pro_google_maps_api_key', '' ) );
        $map_zoom_level     = get_option( 'wcvendors_pro_google_maps_zoom_level', '' );
        $key_exists         = empty( $google_map_api_key ) ? false : true;

        if ( $key_exists && ! empty( $map_zoom_level ) && ( 'settings' === $current_page || ( isset( $_GET['terms'] ) && 1 === $_GET['terms'] ) ) ) { // phpcs:ignore

            wp_enqueue_script(
                'google-maps-api',
                add_query_arg(
                    array(
                        'sensor'    => false,
                        'key'       => esc_attr( $google_map_api_key ),
                        'libraries' => 'places',
                    ),
                    '//maps.googleapis.com/maps/api/js'
                ),
                array(),
                WCV_PRO_VERSION,
                false
            );

            wp_enqueue_script(
                'wcv-maps-address-picker',
                $this->base_dir . 'assets/js/maps-address-picker' . $this->suffix . '.js',
                array( 'jquery', 'select2' ),
                WCV_PRO_VERSION,
                true
            );
        }

        // Vendor Product Exporter.
        wp_register_script(
            'wcv-product-export',
            $this->base_dir . 'assets/js/product-export' . $this->suffix . '.js',
            array( 'jquery' ),
            WCV_PRO_VERSION,
            true
        );

        wp_localize_script(
            'wcv-product-export',
            'wcv_product_export_params',
            array(
                'export_nonce' => wp_create_nonce( 'wcv-product-export' ),
                'ajax_url'     => admin_url( 'admin-ajax.php' ),
            )
        );

        // Vendor Product Importer.
        wp_register_script(
            'wc-product-import',
            $this->base_dir . 'assets/js/product-import' . $this->suffix . '.js',
            array( 'jquery' ),
            WCV_PRO_VERSION,
            true
        );
        $wcv_general_args = array(
            'require_featured_image' => wc_string_to_bool( get_option( 'wcvendors_required_product_media_featured', 'no' ) ),
            'require_gallery_image'  => wc_string_to_bool( get_option( 'wcvendors_required_product_media_gallery', 'no' ) ),
        );
        wp_register_script( 'wcv-general', $this->base_dir . 'assets/js/general' . $this->suffix . '.js', array( 'jquery' ), WCV_PRO_VERSION, true );
        wp_localize_script( 'wcv-general', 'wcv_general', $wcv_general_args );
        wp_enqueue_script( 'wcv-general' );
    }

    /**
     * Get flatpickr localization strings
     *
     * @return array
     * @version 1.8.7
     * @since   1.8.7 - Refactored from WCVendors_Pro_Dashboard::enqueue_scripts()
     */
    public static function get_flatpickr_locale() {
        return apply_filters(
            'wcvendors_pro_flatpickr_locale',
            array(
                'weekdays'         => array(
                    'shorthand' => array(
                        __( 'Sun', 'wcvendors-pro' ),
                        __( 'Mon', 'wcvendors-pro' ),
                        __( 'Tue', 'wcvendors-pro' ),
                        __( 'Wed', 'wcvendors-pro' ),
                        __( 'Thu', 'wcvendors-pro' ),
                        __( 'Fri', 'wcvendors-pro' ),
                        __( 'Sat', 'wcvendors-pro' ),
                    ),
                    'longhand'  => array(
                        __( 'Sunday', 'wcvendors-pro' ),
                        __( 'Monday', 'wcvendors-pro' ),
                        __( 'Tuesday', 'wcvendors-pro' ),
                        __( 'Wednesday', 'wcvendors-pro' ),
                        __( 'Thursday', 'wcvendors-pro' ),
                        __( 'Friday', 'wcvendors-pro' ),
                        __( 'Saturday', 'wcvendors-pro' ),
                    ),
                ),
                'months'           => array(
                    'shorthand' => array(
                        __( 'Jan', 'wcvendors-pro' ),
                        __( 'Feb', 'wcvendors-pro' ),
                        __( 'Mar', 'wcvendors-pro' ),
                        __( 'Apr', 'wcvendors-pro' ),
                        __( 'May', 'wcvendors-pro' ),
                        __( 'Jun', 'wcvendors-pro' ),
                        __( 'Jul', 'wcvendors-pro' ),
                        __( 'Aug', 'wcvendors-pro' ),
                        __( 'Sep', 'wcvendors-pro' ),
                        __( 'Oct', 'wcvendors-pro' ),
                        __( 'Nov', 'wcvendors-pro' ),
                        __( 'Dec', 'wcvendors-pro' ),
                    ),
                    'longhand'  => array(
                        __( 'January', 'wcvendors-pro' ),
                        __( 'February', 'wcvendors-pro' ),
                        __( 'March', 'wcvendors-pro' ),
                        __( 'April', 'wcvendors-pro' ),
                        __( 'May', 'wcvendors-pro' ),
                        __( 'June', 'wcvendors-pro' ),
                        __( 'July', 'wcvendors-pro' ),
                        __( 'August', 'wcvendors-pro' ),
                        __( 'September', 'wcvendors-pro' ),
                        __( 'October', 'wcvendors-pro' ),
                        __( 'November', 'wcvendors-pro' ),
                        __( 'December', 'wcvendors-pro' ),
                    ),
                ),
                'daysInMonth'      => array( 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 ),
                'firstDayOfWeek'   => 0,
                'rangeSeparator'   => __( ' to ', 'wcvendors-pro' ),
                'weekAbbreviation' => __( 'Wk', 'wcvendors-pro' ),
                'scrollTitle'      => __( 'Scroll to increment', 'wcvendors-pro' ),
                'toggleTitle'      => __( 'Click to toggle', 'wcvendors-pro' ),
                'amPM'             => array(
                    __( 'AM', 'wcvendors-pro' ),
                    __( 'PM', 'wcvendors-pro' ),
                ),
                'yearAriaLabel'    => __( 'Year', 'wcvendors-pro' ),
                'monthAriaLabel'   => __( 'Month', 'wcvendors-pro' ),
                'hourAriaLabel'    => __( 'Hour', 'wcvendors-pro' ),
                'minuteAriaLabel'  => __( 'Minute', 'wcvendors-pro' ),
                'time_24hr'        => apply_filters( 'wcv_time_picker_use_twenty_four_hour_clock', false ),
            )
        );
    }

    /**
     * Get datepicker localization strings
     *
     * @return array
     * @version 1.8.7
     * @since   1.8.7 - Extracted from WCVendors_Pro_Dashboard::enqueue_scripts()
     */
    public static function get_datepicker_localization() {
        $datepicker_localization = array(
            'locale'      => self::get_flatpickr_locale(),
            'date_format' => apply_filters( 'wcvendors_datepicker_date_format', 'Y-m-d' ),
            'time_format' => apply_filters( 'wcvendors_datepicker_time_format', 'H:i' ),

        );

        return apply_filters( 'wcvendors_datepicker_localization', $datepicker_localization );
    }

    /**
     * Select 2 seperator options for tag search
     *
     * @since  1.3.6
     * @access public
     * @return array separator types
     * @param  string $option sprarator option.
     */
    public function select2_separator( $option ) {
        $separator = array( ',', ' ' );
        switch ( $option ) {
            case 'space':
                $separator = array( ' ' );
                break;
            case 'comma':
                $separator = array( ',' );
                break;
            default:
                return apply_filters( 'wcv_tag_separator_defaults', array( ',', ' ' ) );
        }
        return apply_filters( 'wcv_tag_separator_defaults', $separator );
    }

    /**
     * Change the post title to the specified SEO Title for the product
     *
     * @param string $title   title of the product.
     * @param int    $post_id product id.
     *
     * @return string $title
     * @since 1.5.8
     */
    public function seo_title( $title, $post_id ) {
        global $post;

        $hide_seo = wc_string_to_bool( get_option( 'wcvendors_hide_product_seo', 'no' ) );

        if ( ! $hide_seo ) {
            $title = get_post_meta( $post_id, 'wcv_product_seo_title', true );

            if ( false === $title ) {
                $product = wc_get_product( $post );
                $title   = $product->get_name();
            }

            return $title;
        }

        return $title;
    }

    /**
     * Output SEO & OpenGraph meta tags
     *
     * @return    void
     * @since      1.5.8
     * @version    1.5.9
     */
    public function product_seo_meta() {

        global $post;

        if ( is_archive() ) {
            return;
        }

        $hide_seo = wc_string_to_bool( get_option( 'wcvendors_hide_product_seo', 'no' ) );

        if ( $hide_seo ) {
            return;
        }

        $product = wc_get_product( $post );

        if ( ! is_a( $product, 'WC_Product' ) ) {
            return;
        }

        $product_id = $product->get_id();

        $seo_title       = get_post_meta( $product_id, 'wcv_product_seo_title', true );
        $seo_description = get_post_meta( $product_id, 'wcv_product_seo_description', true );
        $seo_keywords    = get_post_meta( $product_id, 'wcv_product_seo_keywords', true );
        $seo_image_url   = get_the_post_thumbnail_url( $post, 'large' );

        $seo_opengraph    = get_post_meta( $product_id, 'wcv_product_seo_opengraph', true );
        $seo_twitter_card = get_post_meta( $product_id, 'wcv_product_seo_twitter_card', true );

        $seo_store_name     = get_user_meta( $post->post_author, 'pv_shop_name', true );
        $seo_store_url      = WCV_Vendors::is_vendor( $post->post_author ) ? WCV_Vendors::get_vendor_shop_page( $post->post_author ) : '';
        $seo_twitter_author = get_user_meta( $post->post_author, '_wcv_twitter_username', true );

        // use categories for keywords if none are defined.
        if ( '' === $seo_keywords ) {
            $categories   = get_the_term_list( $product_id, 'product_cat', '', ',', '' );
            $seo_keywords = wcv_strip_html( $categories );
        }

        $seo_title       = ! empty( $seo_title ) ? $seo_title : $product->get_name();
        $seo_description = ! empty( $seo_description ) ? $seo_description : substr( $product->get_description(), 0, apply_filters( 'wcv_seo_description_length', 155 ) );

        $seo_product_amount  = $product->get_price();
        $seo_currency_code   = get_woocommerce_currency();
        $seo_currency_symbol = get_woocommerce_currency_symbol( $seo_currency_code );

        wc_get_template(
            'product-seo-meta.php',
            array(
                'product_id'          => $product_id,
                'seo_title'           => $seo_title,
                'seo_description'     => $seo_description,
                'seo_keywords'        => $seo_keywords,
                'seo_image_url'       => $seo_image_url,
                'seo_product_amount'  => $seo_product_amount,
                'seo_currency_code'   => $seo_currency_code,
                'seo_currency_symbol' => $seo_currency_symbol,
                'seo_store_name'      => $seo_store_name,
                'seo_twitter_author'  => $seo_twitter_author,
                'seo_opengraph'       => $seo_opengraph,
                'seo_twitter_card'    => $seo_twitter_card,
                'seo_store_url'       => $seo_store_url,
            ),
            'wc-vendors/product/',
            WCV_PRO_ABSPATH_TEMPLATES . 'product/'
        );
    }

    /**
     * Preview for file_uploader
     */
    public static function file_uploader_preview() {

        $file_url = $_POST['file_url']; // phpcs:ignore WordPress.Security.NonceVerification
        global $wp_embed;
        echo( do_shortcode( $wp_embed->run_shortcode( '[embed]' . esc_url( $file_url ) . '[/embed]' ) ) );
        wp_die();
    }


    /**
     * Load theme support automatically
     *
     * @since 1.6.0
     * @version 1.7.9
     */
    public function load_theme_support() {

        $theme_support = get_option( 'wcvendors_load_theme_support', 'no' );

        if ( ! wc_string_to_bool( $theme_support ) ) {
            return;
        }

        $theme = wp_get_theme();

        switch ( $theme->template ) {
            case 'Divi':
                include_once 'theme-support/class-divi.php';
                break;
            case 'storefront':
                include_once 'theme-support/class-storefront.php';
                break;
            case 'my-listing':
                include_once 'theme-support/class-mylisting.php';
                break;
            case 'astra':
                include_once 'theme-support/class-astra.php';
                break;
            case 'generatepress':
                include_once 'theme-support/class-generatepress.php';
                break;
            default:
                // Allow hook into the theme support.
                // @todo fix this filter, it isn't accesible from functions.php - jamie.
                $theme_support_file = apply_filters( 'wcv_load_theme_support', '' );
                if ( $theme_support_file ) {
                    include $theme_support_file;
                }
                break;
        }
    }

    /**
     * Load SEO plugin support
     *
     * @since 1.8.7
     * @version 1.8.7
     */
    public function load_seo_plugins_compatibility() {
        $is_enabled = wc_string_to_bool( get_option( 'wcvendors_load_seo_compatibility' ) );

        if ( ! $is_enabled ) {
            return;
        }

        include 'integrations/seo/abstract-class-seo-compatibility.php';
        include 'integrations/seo/class-seo-framework.php';
        include 'integrations/seo/class-yoast.php';
        include 'integrations/seo/class-aios.php';
        include 'integrations/seo/class-rank-math.php';
    }

    /**
     * Policy tab
     *
     * @param array $tabs Tabs.
     *
     * @return mixed
     */
    public function product_policy_tab( $tabs ) {

        global $post;
        $vendor_id = $post->post_author;

        if ( ! WCV_vendors::is_vendor( $vendor_id ) ) {
            return $tabs;
        }

        if ( wc_string_to_bool( get_option( 'wcvendors_hide_settings_tab_policies', 'no' ) ) ) {
            return $tabs;
        }

        if (
            wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_policy_privacy', 'no' ) )
            && wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_policy_terms', 'no' ) )
            && wc_string_to_bool( get_option( 'wcvendors_hide_settings_shipping_shipping_policy', 'no' ) )
            && wc_string_to_bool( get_option( 'wcvendors_hide_settings_shipping_return_policy', 'no' ) )
        ) {
            return $tabs;
        }

        $allow_override = wc_string_to_bool( get_option( 'wcvendors_allow_policy_override', 'yes' ) );
        $privacy        = '';
        $terms          = '';
        $shipping       = '';
        $return         = '';

        if ( $allow_override ) {
            // When override is enabled, check vendor-specific policies first, then fall back to global.
            $privacy = trim( get_user_meta( $vendor_id, 'wcv_policy_privacy', true ) );
            $terms   = trim( get_user_meta( $vendor_id, 'wcv_policy_terms', true ) );

            $shipping_detail = get_user_meta( $vendor_id, '_wcv_shipping', true );
            if ( isset( $shipping_detail['shipping_policy'] ) && $shipping_detail['shipping_policy'] ) {
                $shipping = trim( $shipping_detail['shipping_policy'] );
            }
            if ( isset( $shipping_detail['return_policy'] ) && $shipping_detail['return_policy'] ) {
                $return = trim( $shipping_detail['return_policy'] );
            }

            // Fall back to global policies if vendor hasn't set their own.
            if ( ! $privacy ) {
                $global_privacy = trim( get_option( 'wcvendors_global_policy_privacy', '' ) );
                if ( $global_privacy ) {
                    $privacy = $global_privacy;
                }
            }

            if ( ! $terms ) {
                $global_terms = trim( get_option( 'wcvendors_global_policy_terms', '' ) );
                if ( $global_terms ) {
                    $terms = $global_terms;
                }
            }

            if ( ! $shipping ) {
                $global_shipping = trim( get_option( 'wcvendors_global_policy_shipping', '' ) );
                if ( $global_shipping ) {
                    $shipping = $global_shipping;
                }
            }

            if ( ! $return ) {
                $global_return = trim( get_option( 'wcvendors_global_policy_return', '' ) );
                if ( $global_return ) {
                    $return = $global_return;
                }
            }
        } else {
            // When override is disabled, only use global policies.
            $privacy  = trim( get_option( 'wcvendors_global_policy_privacy', '' ) );
            $terms    = trim( get_option( 'wcvendors_global_policy_terms', '' ) );
            $shipping = trim( get_option( 'wcvendors_global_policy_shipping', '' ) );
            $return   = trim( get_option( 'wcvendors_global_policy_return', '' ) );
        }

        if ( ! ( $privacy || $terms || $shipping || $return ) ) {
            return $tabs;
        }

        $tabs['policies'] = apply_filters(
            'wcv_single_product_policies_tab',
            array(
                'title'    => sprintf(
                    // translators: %s is the name used to refer to a vendor.
                    __( '%s Policies', 'wcvendors-pro' ),
                    wcv_get_vendor_name()
                ),
                'priority' => 50,
                'callback' => array( $this, 'product_policy_tab_content' ),
            )
        );

        return $tabs;
    }

    /**
     * Policy tab content
     */
    public function product_policy_tab_content() {
        global $post;
        $vendor_id      = $post->post_author;
        $policies       = array();
        $allow_override = wc_string_to_bool( get_option( 'wcvendors_allow_policy_override', 'yes' ) );
        $privacy        = '';
        $terms          = '';
        $shipping       = '';
        $return         = '';

        if ( $allow_override ) {
            // When override is enabled, check vendor-specific policies first, then fall back to global.
            $privacy = trim( get_user_meta( $vendor_id, 'wcv_policy_privacy', true ) );
            $terms   = trim( get_user_meta( $vendor_id, 'wcv_policy_terms', true ) );

            $shipping_detail = get_user_meta( $vendor_id, '_wcv_shipping', true );
            if ( isset( $shipping_detail['shipping_policy'] ) && $shipping_detail['shipping_policy'] ) {
                $shipping = $shipping_detail['shipping_policy'];
            }
            if ( isset( $shipping_detail['return_policy'] ) && $shipping_detail['return_policy'] ) {
                $return = $shipping_detail['return_policy'];
            }

            // Fall back to global policies if vendor hasn't set their own.
            if ( ! $privacy ) {
                $global_privacy = trim( get_option( 'wcvendors_global_policy_privacy', '' ) );
                if ( $global_privacy ) {
                    $privacy = $global_privacy;
                }
            }

            if ( ! $terms ) {
                $global_terms = trim( get_option( 'wcvendors_global_policy_terms', '' ) );
                if ( $global_terms ) {
                    $terms = $global_terms;
                }
            }

            if ( ! $shipping ) {
                $global_shipping = trim( get_option( 'wcvendors_global_policy_shipping', '' ) );
                if ( $global_shipping ) {
                    $shipping = $global_shipping;
                }
            }

            if ( ! $return ) {
                $global_return = trim( get_option( 'wcvendors_global_policy_return', '' ) );
                if ( $global_return ) {
                    $return = $global_return;
                }
            }
        } else {
            // When override is disabled, only use global policies.
            $privacy  = trim( get_option( 'wcvendors_global_policy_privacy', '' ) );
            $terms    = trim( get_option( 'wcvendors_global_policy_terms', '' ) );
            $shipping = trim( get_option( 'wcvendors_global_policy_shipping', '' ) );
            $return   = trim( get_option( 'wcvendors_global_policy_return', '' ) );
        }

        $privacy = wpautop( $privacy );

        if ( ! wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_policy_privacy', 'no' ) ) && $privacy ) {
            $policies['privacy'] = array(
                'title'   => __( 'Privacy Policy', 'wcvendors-pro' ),
                'content' => $privacy,
            );
        }

        $terms = wpautop( $terms );

        if ( ! wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_policy_terms', 'no' ) ) && $terms ) {
            $policies['terms'] = array(
                'title'   => __( 'Terms and Conditions', 'wcvendors-pro' ),
                'content' => $terms,
            );
        }

        $shipping = wpautop( $shipping );
        $return   = wpautop( $return );

        if (
            ! wc_string_to_bool( get_option( 'wcvendors_hide_settings_shipping_shipping_policy', 'no' ) )
            && $shipping
        ) {
            $policies['shipping'] = array(
                'title'   => __( 'Shipping Policy', 'wcvendors-pro' ),
                'content' => $shipping,
            );
        }

        if (
            ! wc_string_to_bool( get_option( 'wcvendors_hide_settings_shipping_return_policy', 'no' ) )
            && $return
        ) {
            $policies['return'] = array(
                'title'   => __( 'Return Policy', 'wcvendors-pro' ),
                'content' => $return,
            );
        }

        $policies = apply_filters( 'wcv_product_policy_tab_content_args', $policies, $vendor_id );

        foreach ( $policies as $id => $policy ) { ?>
            <div class="policy <?php echo esc_attr( $id ); ?>" id="policy-<?php echo esc_attr( $id ); ?>">
                <h3><?php echo esc_html( $policy['title'] ); ?></h3>
                <div class="policy-content">
                    <?php echo wp_kses_post( $policy['content'] ); ?>
                </div>
            </div>
            <?php
        }
    }

    /**
     * Filter the category list output.
     *
     * Change the url of each category in the list by replacing the shortened url with a full url.
     *
     * @param string $output The categories list html output to be changed.
     * @param array  $args   The arguments used to search for the categories.
     * @return string
     * @version 1.7.3
     * @since   1.7.3
     */
    public function filter_categories_list_output( $output, $args ) {
        $vendor_id = wcv_get_vendor_id();

        if ( ! $vendor_id ) {
            return $output;
        }

        $vendor_shop_url = WCV_Vendors::get_vendor_shop_page( $vendor_id );

        if ( $vendor_shop_url ) {
            return str_replace(
                'href="?',
                apply_filters(
                    'wcvendors_categories_vendor_shop_url',
                    'href="' . esc_url_raw( $vendor_shop_url ) . '?',
                    $args
                ),
                $output
            );
        }

        return $output;
    }

    /**
     * Output the vendor google analytics code if they have added their tracking ID to their settings page .
     */
    public function add_vendor_ga_code() {
        global $post;

        $vendor_id = 0;

        // Not on vendor store page or vendor single product bail out .
        if ( WCV_Vendors::is_vendor_page() ) {
            $vendor_shop = urldecode( get_query_var( 'vendor_shop' ) );
            $vendor_id   = WCV_Vendors::get_vendor_id( $vendor_shop );
        } elseif ( is_singular( 'product' ) && WCV_Vendors::is_vendor_product_page( $post->post_author ) ) {
            $vendor_id = $post->post_author;
        }

        $vendor_ga_code = wcv_output_vendor_ga_code( $vendor_id );
        echo $vendor_ga_code; // phpcs:ignore
    }


    /**
     * Add dimension fields after tax field.
     *
     * @param string $object_id .
     * @version 1.8.0
     * @since   1.8.0
     */
    public function show_dimension_when_shipping_disabled( $object_id ) {

        $wcv_product_panel       = get_option( 'wcvendors_capability_product_data_tabs', array() );
        $shipping_methods        = WC()->shipping() ? WC()->shipping->load_shipping_methods() : array();
        $shipping_method_enabled = ( array_key_exists( 'wcv_pro_vendor_shipping', $shipping_methods ) && 'yes' === $shipping_methods['wcv_pro_vendor_shipping']->enabled ) ? true : false;

        if ( ! $shipping_method_enabled && ! array_key_exists( 'shipping', $wcv_product_panel ) ) {
            WCVendors_Pro_Product_Form::weight( $object_id );
            WCVendors_Pro_Product_Form::dimensions( $object_id );
        }
    }

    /**
     * Get Pro dashboard args
     *
     * @param array $dashboard_args Dashboard args.
     */
    public function get_pro_dashboard_args( $dashboard_args ) {
        $shipping_settings   = get_option( 'woocommerce_wcv_pro_vendor_shipping_settings', wcv_get_default_vendor_shipping() );
        $store_shipping_type = get_user_meta( get_current_user_id(), '_wcv_shipping_type', true );
        $vendor_select       = wc_string_to_bool( get_option( 'wcvendors_vendor_select_shipping', 'no' ) );
        $shipping_type       = ( '' !== $store_shipping_type ) ? $store_shipping_type : $shipping_settings['shipping_system'];
        $dashboard_args      = array(
            'shipping_settings'   => $shipping_settings,
            'store_shipping_type' => $store_shipping_type,
            'vendor_select'       => $vendor_select,
            'shipping_type'       => $shipping_type,
        );
        return $dashboard_args;
    }


    /**
     * Add more allowed tags
     *
     * @param array $allowedtags The allowed tags.
     *
     * @since 1.9.4
     */
    public function allowed_html_tags( $allowedtags ) {
        $more_allowed_tags = array(
            'svg' => array(
                'class'       => array(),
                'xmlns'       => array(),
                'width'       => array(),
                'height'      => array(),
                'viewBox'     => array(),
                'fill'        => array(),
                'aria-hidden' => array(),
                'role'        => array(),
                'focusable'   => array(),
            ),
            'use' => array(
                'xlink:href' => array(),
            ),
        );

        $allowedtags = wp_parse_args( $allowedtags, $more_allowed_tags );

        return $allowedtags;
    }

    /**
     * Add limit text length for store description
     *
     * @param array $args Field arguments.
     * @return array
     *
     * @since 1.9.4
     * @version 1.9.4 - Add limit text length for store description.
     */
    public function add_store_description_limit( $args ) {
        $limit       = get_option( 'wcvendors_store_description_limit', 250 );
        $limit_array = array(
            // translators: %s: maximum number of characters allowed.
            'data-parsley-maxlength-message' => sprintf( __( 'This value must be less than %s characters', 'wcvendors-pro' ), $limit ),
            'data-parsley-maxlength'         => $limit,
        );
        // Parsley custom attribute.
        $args['custom_attributes'] = array_merge( $args['custom_attributes'], $limit_array );
        return $args;
    }

    /**
     * Add limit text length for store description in the editor
     *
     * @param string $markup Editor markup.
     * @return string
     *
     * @since 1.9.4
     * @version 1.9.4 - Add limit text length for store description.
     */
    public function add_store_description_limit_editor( $markup ) {
        // check if have pv_shop_description id in the markup.
        if ( strpos( $markup, 'pv_shop_description' ) !== false ) {
            $limit  = get_option( 'wcvendors_store_description_limit', 250 );
            $markup = str_replace( '<textarea', '<textarea data-parsley-maxlength="' . $limit . '" data-parsley-maxlength-message="' . sprintf( /* translators: %s: maximum number of characters allowed. */ __( 'This value must be less than %s characters', 'wcvendors-pro' ), $limit ) . '" data-parsley-errors-container="#wp-pv_shop_description-wrap"', $markup );
            $markup = str_replace( 'data-parsley-error-message', 'data-parsley-required-message', $markup );
        }
        return $markup;
    }
}
