<?php

/**
 * The WCVendors Pro Coupon Form class
 *
 * This is the order form class
 *
 * @package    WCVendors_Pro
 * @subpackage WCVendors_Pro/public/forms
 * @author     Jamie Madden <support@wcvendors.com>
 */
class WCVendors_Pro_Coupon_Form {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $wcvendors_pro The ID of this plugin.
     */
    private $wcvendors_pro;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $version The current version of this plugin.
     */
    private $version;

    /**
     * Is the plugin in debug mode
     *
     * @since    1.0.0
     * @access   private
     * @var      bool $debug plugin is in debug mode
     */
    private $debug;

    /**
     * Is the plugin base directory
     *
     * @since    1.0.0
     * @access   private
     * @var      string $base_dir string path for the plugin directory
     */
    private $base_dir;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     *
     * @param      string $wcvendors_pro The name of the plugin.
     * @param      string $version       The version of this plugin.
     * @param      bool   $debug         Debug mode.
     */
    public function __construct( $wcvendors_pro, $version, $debug ) {

        $this->wcvendors_pro = $wcvendors_pro;
        $this->version       = $version;
        $this->debug         = $debug;
        $this->base_dir      = plugin_dir_path( __DIR__ );
    }

    /**
     *  Output required form data
     *
     * @since    1.0.0
     *
     * @param     string $button_text button text.
     * @param     int    $coupon_id coupon id.
     */
    public static function form_data( $button_text, $coupon_id ) {

        wp_nonce_field( 'wcv-save-coupon', 'wcv_save_coupon' );

        // Coupon ID if it already exists.
        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_coupon_post_id',
                array(
                    'post_id' => $coupon_id,
                    'type'    => 'hidden',
                    'id'      => '_wcv_coupon_post_id',
                    'value'   => $coupon_id,
                )
            )
        );

        self::save_button( $button_text );
    }

    /**
     *  Output coupon title
     *
     * @since    1.0.0
     * @version 1.7.7
     *
     * @param string $coupon_code coupon code.
     */
    public static function coupon_code( $coupon_code ) {

        if ( isset( $_POST['wcv_save_coupon'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
            $coupon_code = array_key_exists( '_wcv_coupon_post_title', $_POST ) ? $_POST['_wcv_coupon_post_title'] : ''; // phpcs:ignore
        }

        // Coupon Code.
        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_coupon_post_title',
                array(
                    'id'                => '_wcv_coupon_post_title',
                    'label'             => __( 'Coupon code', 'wcvendors-pro' ),
                    'placeholder'       => __( 'Coupon code', 'wcvendors-pro' ),
                    'type'              => 'text',
                    'value'             => $coupon_code,
                    'custom_attributes' => array(
                        'required'                   => '',
                        'data-parsley-error-message' => __( 'A coupon code is required.', 'wcvendors-pro' ),
                    ),
                )
            )
        );
    }

    /**
     *  Output coupon description
     *
     * @since    1.0.0
     * @version 1.7.7
     *
     * @param string $coupon_description coupon description.
     */
    public static function coupon_description( $coupon_description ) {

        // Coupon Description.
        WCVendors_Pro_Form_Helper::textarea(
            apply_filters(
                'wcv_coupon_post_content',
                array(
                    'id'          => '_wcv_coupon_post_excerpt',
                    'label'       => __( 'Coupon description', 'wcvendors-pro' ),
                    'placeholder' => __( 'Coupon description', 'wcvendors-pro' ),
                    'type'        => 'text',
                    'value'       => $coupon_description,
                )
            )
        );
    }

    /**
     *  Output discount type drop down
     *
     * @since    1.0.0
     * @version 1.7.7
     *
     * @param string $discount_type discount type.
     */
    public static function discount_type( $discount_type ) {

        // Discount Type.
        WCVendors_Pro_Form_Helper::select(
            apply_filters(
                'wcv_coupon_discount_type',
                array(
                    'id'          => '_wcv_coupon_post_meta_discount_type',
                    'label'       => __( 'Discount type', 'wcvendors-pro' ),
                    'value'       => $discount_type,
                    'placeholder' => __( 'Discount type', 'wcvendors-pro' ),
                    'options'     => array(
                        'fixed_product' => __( 'Fixed discount', 'wcvendors-pro' ),
                        'percent'       => __( 'Percentage discount', 'wcvendors-pro' ),
                    ),
                )
            )
        );
    }

    /**
     *  Output appl to all products checkbox
     *
     * @since    1.0.0
     *
     * @param bool $all_products all products.
     */
    public static function apply_to_all_products( $all_products ) {

        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_coupon_apply_to_all_products',
                array(
                    'id'          => '_wcv_coupon_post_meta_apply_to_all_products',
                    'label'       => __( 'Apply to all my products', 'wcvendors-pro' ),
                    'type'        => 'checkbox',
                    'value'       => $all_products,
                    'desc_tip'    => true,
                    'description' => __( 'Check this box if the coupon applies to all your products.', 'wcvendors-pro' ),
                )
            )
        );
    }

    /**
     *  Output coupon amount
     *
     * @since    1.0.0
     * @version  1.7.7
     *
     * @todo When discount type change, the amount format is not changed until the coupon is saved.
     *
     * @param int    $coupon_amount coupon amount.
     * @param string $discount_type discount type.
     */
    public static function coupon_amount( $coupon_amount, $discount_type ) {

        $custom_attributes = array(
            'required' => '',
        );

        // Coupon Amount.
        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_coupon_amount',
                array(
                    'id'                => '_wcv_coupon_post_meta_coupon_amount',
                    'label'             => __( 'Coupon amount', 'wcvendors-pro' ),
                    'placeholder'       => __( 'Coupon amount', 'wcvendors-pro' ),
                    'data_type'         => 'decimal',
                    'value'             => 'percent' === $discount_type ? $coupon_amount : wc_format_localized_price( $coupon_amount ),
                    'custom_attributes' => $custom_attributes,
                )
            )
        );
    }

    /**
     *  Output Allow free shipping checkbox
     *
     * @param bool $free_shipping free shipping.
     *
     * @since   1.0.0
     * @version 1.7.7
     * @version 1.8.3 - Addressed issue with WooCommerce shipping detection
     */
    public static function free_shipping( $free_shipping ) {

        if ( wcv_is_wc_shipping_enabled() && wcv_is_vendors_shipping_enabled() ) {
            WCVendors_Pro_Form_Helper::input(
                apply_filters(
                    'wcv_coupon_free_shipping',
                    array(
                        'id'          => '_wcv_coupon_post_meta_free_shipping',
                        'label'       => __( 'Allow free shipping', 'wcvendors-pro' ),
                        'type'        => 'checkbox',
                        'value'       => $free_shipping,
                        'desc_tip'    => true,
                        'description' => __( 'Check this box if the coupon grants free shipping.', 'wcvendors-pro' ),
                    )
                )
            );
        }
    }

    /**
     *  Output coupon expiry date
     *
     * @param string $expiry_date expiry date.
     *
     * @since    1.0.0
     * @version 1.7.7
     */
    public static function expiry_date( $expiry_date ) {

        $date_format  = get_option( 'wcvendors_dashboard_date_format', 'Y-m-d' );
        $default_date = gmdate( $date_format, strtotime( 'now' ) );

        if ( isset( $_POST['wcv_save_coupon'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
            $expiry_date = array_key_exists( '_wcv_coupon_post_meta_expiry_date', $_POST ) ? gmdate( $date_format, strtotime( $_POST['_wcv_coupon_post_meta_expiry_date'] ) ) : ''; // phpcs:ignore
        }

        // Coupon sale date.
        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_product_sale_price_date_from',
                array(
                    'id'                  => '_wcv_coupon_post_meta_expiry_date',
                    'label'               => __( 'Coupon expiry date', 'wcvendors-pro' ),
                    'class'               => 'wcv-datepicker wcv-init-picker',
                    'value'               => $expiry_date,
                    'placeholder'         => 'YYYY-MM-DD',
                    'append_before'       => '<span class="wcv-flex" title="toggle" data-toggle>' . wcv_get_icon( 'wcv-icon wcv-icon-dashboard-icon', 'wcv-icon-calendar' ) . '</span>',
                    'input_wrapper_class' => 'wcv-datepicker-wrapper wcv-flex',
                    'custom_attributes'   => array(
                        'maxlenth'     => '10',
                        'pattern'      => '[0-9]{4}-(0[1-9]|1[012])-(0[1-9]|1[0-9]|2[0-9]|3[01])',
                        'data-input'   => '',
                        'data-default' => $default_date,
                    ),
                )
            )
        );
    }

    /**
     *  Output min spend
     *
     * @since    1.0.0
     * @version  1.7.7
     *
     * @param int $minimum_amount minimum amount.
     */
    public static function minimum_amount( $minimum_amount ) {

        $custom_attributes = array(
            'data-parsley-price' => '',
        );

        // Minimum Spend.
        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_coupon_minimum_amount',
                array(
                    'id'                => '_wcv_coupon_post_meta_minimum_amount',
                    'label'             => __( 'Minimum spend', 'wcvendors-pro' ),
                    'placeholder'       => __( 'No minimum', 'wcvendors-pro' ),
                    'data_type'         => 'price',
                    'value'             => wc_format_localized_price( $minimum_amount ),
                    'custom_attributes' => $custom_attributes,
                )
            )
        );
    }

    /**
     *  Output max spend
     *
     * @since    1.0.0
     * @version  1.7.7
     *
     * @param int $maximum_amount maximum amount.
     */
    public static function maximum_amount( $maximum_amount ) {

        $custom_attributes = array(
            'data-parsley-price' => '',
        );

        // Minimum Spend.
        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_coupon_maximum_amount',
                array(
                    'id'                => '_wcv_coupon_post_meta_maximum_amount',
                    'label'             => __( 'Maximum spend', 'wcvendors-pro' ),
                    'placeholder'       => __( 'No maximum', 'wcvendors-pro' ),
                    'data_type'         => 'price',
                    'value'             => wc_format_localized_price( $maximum_amount ),
                    'custom_attributes' => $custom_attributes,
                )
            )
        );
    }

    /**
     *  Output Individual use only
     *
     * @since    1.0.0
     * @version 1.7.7
     *
     * @param bool $individual_use individual use.
     */
    public static function individual_use( $individual_use ) {

        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_coupon_individual_use',
                array(
                    'id'            => '_wcv_coupon_post_meta_individual_use',
                    'label'         => __( 'Individual use only', 'wcvendors-pro' ),
                    'type'          => 'checkbox',
                    'value'         => wc_bool_to_string( $individual_use ),
                    'wrapper_start' => '<div class="all-50">',
                    'wrapper_end'   => '</div>',
                    'desc_tip'      => true,
                    'description'   => __( 'Check this box if the coupon cannot be used in conjunction with other coupons.', 'wcvendors-pro' ),
                )
            )
        );
    }

    /**
     *  Output exclude sale items
     *
     * @param bool $exclude_sale_items exclude sale items.
     *
     * @since    1.0.0
     */
    public static function exclude_sale_items( $exclude_sale_items ) {

        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_coupon_exclude_sale_items',
                array(
                    'id'            => '_wcv_coupon_post_meta_exclude_sale_items',
                    'label'         => __( 'Exclude sale items', 'wcvendors-pro' ),
                    'type'          => 'checkbox',
                    'value'         => wc_bool_to_string( $exclude_sale_items ),
                    'wrapper_start' => '<div class="all-50">',
                    'wrapper_end'   => '</div>',
                    'desc_tip'      => true,
                    'description'   => __( 'Check this box if the coupon should not apply to items on sale. Per-item coupons will only work if the item is not on sale. Per-cart coupons will only work if there are no sale items in the cart.', 'wcvendors-pro' ),
                )
            )
        );
    }

    /**
     *  Output included products
     *
     * @since    1.0.0
     * @version 1.7.7
     *
     * @param array $product_ids product ids.
     */
    public static function products( $product_ids ) {

        $return_products_ids = array();

        if ( is_array( $product_ids ) ) {

            if ( ! empty( $product_ids ) ) {
                foreach ( $product_ids as $product_id ) {
                    $product = wc_get_product( $product_id );
                    if ( is_object( $product ) ) {
                        $return_products_ids[ $product_id ] = wp_kses_post( html_entity_decode( $product->get_formatted_name() ) );
                    }
                }
            }
        }

        // Output product searcher.
        WCVendors_Pro_Form_Helper::select(
            apply_filters(
                'wcv_coupon_product_ids',
                array(
                    'id'                => '_wcv_coupon_post_meta_product_ids',
                    'label'             => __( 'Products', 'wcvendors-pro' ),
                    'value'             => implode( ',', array_keys( $return_products_ids ) ),
                    'style'             => 'width: 100%;',
                    'class'             => 'wc-product-search',
                    'show_label'        => true,
                    'multiple'          => true,
                    'options'           => $return_products_ids,
                    'custom_attributes' => array(
                        'data-placeholder' => __( 'Search for a product &hellip;', 'wcvendors-pro' ),
                        'data-action'      => 'wcv_json_search_products',
                    ),
                )
            )
        );
    }

    /**
     *  Output the exclude products field
     *
     * @since    1.0.0
     * @version 1.7.7
     *
     * @param array $exclude_products exclude products.
     */
    public static function exclude_products( $exclude_products ) {

        // Find what the meta key this is stored in.
        $return_excluded_product_ids = array();

        if ( is_array( $exclude_products ) ) {

            if ( ! empty( $exclude_products ) ) {
                foreach ( $exclude_products as $product_id ) {

                    $product = wc_get_product( $product_id );

                    if ( is_object( $product ) ) {
                        $return_excluded_product_ids[ $product_id ] = wp_kses_post( html_entity_decode( $product->get_formatted_name() ) );
                    }
                }
            }
        }

        // Output product searcher.
        WCVendors_Pro_Form_Helper::select(
            apply_filters(
                'wcv_coupon_exlude_product_ids',
                array(
                    'id'                => '_wcv_coupon_post_meta_exclude_product_ids',
                    'label'             => __( 'Exclude products', 'wcvendors-pro' ),
                    'value'             => implode( ',', array_keys( $return_excluded_product_ids ) ),
                    'style'             => 'width: 100%;',
                    'class'             => 'wc-product-search',
                    'show_label'        => true,
                    'multiple'          => true,
                    'options'           => $return_excluded_product_ids,
                    'custom_attributes' => array(
                        'data-placeholder' => __( 'Search for a product &hellip;', 'wcvendors-pro' ),
                        'data-action'      => 'wcv_json_search_products',
                    ),
                )
            )
        );
    }

    /**
     *  Output email addresses
     *
     * @since    1.0.0
     *
     * @param array $email_addresses email addresses.
     */
    public static function email_addresses( $email_addresses ) {

        // email addresses.
        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_coupon_email_addresses',
                array(
                    'id'          => '_wcv_coupon_post_meta_email_addresses',
                    'label'       => __( 'Email Restrictions', 'wcvendors-pro' ),
                    'placeholder' => __( 'No restrictions', 'wcvendors-pro' ),
                    'type'        => 'text',
                    'value'       => implode( ', ', (array) $email_addresses ),
                    'desc_tip'    => true,
                    'description' => __( 'List of allowed emails to check against the customer&#039;s billing email when an order is placed. Separate email addresses with commas.', 'wcvendors-pro' ),
                )
            )
        );
    }

    /**
     *  Output usage_limit
     *
     * @since    1.0.0
     *
     * @param int $usage_limit usage limit.
     */
    public static function usage_limit( $usage_limit ) {

        $custom_attributes = array(
            'min'  => 0,
            'step' => 1,
        );

        // usage_limit.
        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_coupon_usage_limit',
                array(
                    'id'                => '_wcv_coupon_post_meta_usage_limit',
                    'label'             => __( 'Usage limit per coupon', 'wcvendors-pro' ),
                    'placeholder'       => __( 'Unlimited usage', 'wcvendors-pro' ),
                    'type'              => 'number',
                    'value'             => $usage_limit ? $usage_limit : '',
                    'custom_attributes' => $custom_attributes,
                )
            )
        );
    }

    /**
     *  Output usage_limit_per_user
     *
     * @param int $limit_usage_to_x_items limit usage to x items.
     *
     * @since    1.0.0
     */
    public static function limit_usage_to_x_items( $limit_usage_to_x_items ) {

        $custom_attributes = array(
            'min'  => 0,
            'step' => 1,
        );

        // usage_limit.
        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_coupon_limit_usage_to_x_items',
                array(
                    'id'                => '_wcv_coupon_post_meta_limit_usage_to_x_items',
                    'label'             => __( 'Usage limit to x items', 'wcvendors-pro' ),
                    'placeholder'       => __( 'Unlimited usage', 'wcvendors-pro' ),
                    'type'              => 'number',
                    'value'             => $limit_usage_to_x_items ? $limit_usage_to_x_items : '',
                    'custom_attributes' => $custom_attributes,
                )
            )
        );
    }

    /**
     *  Output usage_limit_per_user
     *
     * @param int $usage_limit_per_user usage limit per user.
     *
     * @since    1.0.0
     */
    public static function usage_limit_per_user( $usage_limit_per_user ) {

        $custom_attributes = array(
            'min'  => 0,
            'step' => 1,
        );

        // usage_limit_per_user.
        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcv_coupon_usage_limit_per_user',
                array(
                    'id'                => '_wcv_coupon_post_meta_usage_limit_per_user',
                    'label'             => __( 'Usage limit per user', 'wcvendors-pro' ),
                    'placeholder'       => __( 'Unlimited usage', 'wcvendors-pro' ),
                    'type'              => 'number',
                    'value'             => $usage_limit_per_user ? $usage_limit_per_user : '',
                    'custom_attributes' => $custom_attributes,
                )
            )
        );
    }

    /**
     *  Output save button
     *
     * @since    1.0.0
     *
     * @param     string $button_text button text.
     */
    public static function save_button( $button_text ) {

        WCVendors_Pro_Form_helper::submit(
            apply_filters(
                'wcv_coupon_save_button',
                array(
                    'id'    => 'coupon_save_button',
                    'value' => $button_text,
                    'class' => 'wcv-button wcv-button-primary',
                )
            )
        );
    }
}
