<?php
use WC_Vendors\Classes\Front\Forms\WCV_Store_Form;
use WC_Vendors\Classes\Front\WCV_Form_Helper;
/**
 * The WCVendors Pro Order Form Class
 *
 * This is the order form class
 *
 * @package    WCVendors_Pro
 * @subpackage WCVendors_Pro/public/forms
 * @author     Jamie Madden <support@wcvendors.com>
 */
class WCVendors_Pro_Store_Form extends WCV_Store_Form { // phpcs:ignore

    /**
     * The base directory of the plugin
     *
     * @since    1.2.0
     * @access   public
     * @var      string $base_dir The base directory of the plugin
     */
    public $base_dir;

    /**
     * The name of the plugin
     *
     * @since    1.2.0
     * @access   public
     * @var      string $wcvendors_pro The name of the plugin
     */
    public $wcvendors_pro;

    /**
     * The version of the plugin
     *
     * @since    1.2.0
     * @access   public
     * @var      string $version The version of the plugin
     */
    public $version;

    /**
     * Is the plugin in debug mode
     *
     * @since    1.2.0
     * @access   public
     * @var      bool $debug Is the plugin in debug mode
     */
    public $debug;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     *
     * @param      string  $wcvendors_pro The name of the plugin.
     * @param      string  $version       The version of this plugin.
     * @param boolean $debug         Is the plugin in debug mode.
     */
    public function __construct( $wcvendors_pro, $version, $debug ) {

        $this->wcvendors_pro = $wcvendors_pro;
        $this->version       = $version;
        $this->debug         = $debug;
        $this->base_dir      = plugin_dir_path( __DIR__ );
    }

    /**
     *  Output store banner uploader
     *
     * @since    1.2.0
     */
    public static function store_banner() {

        $branding             = ( 'signup' === self::$form_type ) ? get_option( 'wcvendors_hide_signup_branding_store_banner', 'no' ) : get_option( 'wcvendors_hide_settings_branding_store_banner', 'no' );
        $require_store_banner = apply_filters(
            'wcv_require_store_banner',
            wc_string_to_bool( get_option( 'wcvendors_required_settings_branding_store_banner', 'no' ) )
        );
        if ( 'yes' !== $branding ) {

            $value = get_user_meta( get_current_user_id(), '_wcv_store_banner_id', true );
            echo '<div class="control-group">';
            echo $require_store_banner ? '<h3>' . esc_html__( 'Store Banner', 'wcvendors-pro' ) . '<span class="required">*</span></h3>' : '<h3>' . esc_html__( 'Store Banner', 'wcvendors-pro' ) . '</h3>';

            if ( 'signup' === self::$form_type ) {
                // translators: %s is the name used to refer to a vendor.
                echo '<p>' . esc_html( sprintf( __( 'Once you become a %s you can upload your banner here.', 'wcvendors-pro' ), wcv_get_vendor_name( true, false ) ) ) . '</p>';

            } else {

                $store_banner_args = array(
                    'id'             => '_wcv_store_banner_id',
                    'header_text'    => __( 'Store Banner', 'wcvendors-pro' ),
                    'add_text'       => __( 'Add Store Banner', 'wcvendors-pro' ),
                    'remove_text'    => __( 'Remove Store Banner', 'wcvendors-pro' ),
                    'image_meta_key' => '_wcv_store_banner_id',
                    'save_button'    => __( 'Add Store Banner', 'wcvendors-pro' ),
                    'window_title'   => __( 'Select an Image', 'wcvendors-pro' ),
                    'value'          => $value,
                );

                if ( $require_store_banner ) {
                    $store_banner_args['required'] = true;
                }

                // Store Banner Image.
                WCVendors_Pro_Form_Helper::pro_file_uploader(
                    apply_filters(
                        'wcv_vendor_store_banner',
                        $store_banner_args
                    )
                );

            }

            echo '</div>';
        }
    }

    /**
     *  Output store icon uploader
     *
     * @since    1.2.0
     * @todo     dimension limits
     */
    public static function store_icon() {

        $branding           = ( 'signup' === self::$form_type ) ? get_option( 'wcvendors_hide_signup_branding_store_icon', 'no' ) : get_option( 'wcvendors_hide_settings_branding_store_icon', 'no' );
        $require_store_icon = apply_filters(
            'wcv_require_store_banner',
            wc_string_to_bool( get_option( 'wcvendors_required_settings_branding_store_icon', 'no' ) )
        );

        if ( 'yes' !== $branding ) {

            $value = get_user_meta( get_current_user_id(), '_wcv_store_icon_id', true );
            echo '<div class="control-group">';
            echo $require_store_icon ? '<h3>' . esc_html__( 'Store Icon', 'wcvendors-pro' ) . '<span class="required">*</span></h3>' : '<h3>' . esc_html__( 'Store Icon', 'wcvendors-pro' ) . '</h3>';

            if ( 'signup' === self::$form_type ) {

                echo '<p>' . esc_html(
                    sprintf(
                    // translators: %s is the name used to refer to a vendor.
                        __( 'Once you become a %s you can upload your store icon here.', 'wcvendors-pro' ),
                        wcv_get_vendor_name( true, false )
                    )
                ) . '</p>';

            } else {

                $store_icon_args = array(
                    'id'             => '_wcv_store_icon_id',
                    'header_text'    => __( 'Store Icon', 'wcvendors-pro' ),
                    'add_text'       => __( 'Add Store Icon', 'wcvendors-pro' ),
                    'remove_text'    => __( 'Remove Store Icon', 'wcvendors-pro' ),
                    'image_meta_key' => '_wcv_store_icon_id',
                    'save_button'    => __( 'Add Store Icon', 'wcvendors-pro' ),
                    'window_title'   => __( 'Select an Image', 'wcvendors-pro' ),
                    'value'          => $value,
                    'size'           => 'thumbnail',
                    'class'          => 'wcv-store-icon',
                );

                if ( $require_store_icon ) {
                    $store_icon_args['required'] = true;
                }

                // Store Icon.
                WCVendors_Pro_Form_Helper::pro_file_uploader(
                    apply_filters(
                        'wcv_vendor_store_icon',
                        $store_icon_args
                    )
                );
            }
            echo '</div>';
        }
    }

    /**
     * Output the show/hide map button
     *
     * @return    void
     * @since      1.5.9
     * @version    1.5.9
     */
    public static function show_hide_map() {

        $hide_store_address_chooser = ( 'signup' === self::$form_type )
            ? wc_string_to_bool( get_option( 'wcvendors_hide_signup_store_address_chooser', 'no' ) )
            : wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_address_chooser', 'no' ) );
        $hide_store_address1        = ( 'signup' === self::$form_type )
            ? wc_string_to_bool( get_option( 'wcvendors_hide_signup_store_address', 'no' ) )
            : wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_address', 'no' ) );

        $api_key        = get_option( 'wcvendors_pro_google_maps_api_key', '' );
        $map_zoom_level = get_option( 'wcvendors_pro_google_maps_zoom_level', '' );
        $key_exists     = empty( $api_key ) ? false : true;

        $required      = ( 'signup' === self::$form_type ) ? wc_string_to_bool( get_option( 'wcvendors_required_signup_store_address_chooser', 'no' ) ) : wc_string_to_bool( get_option( 'wcvendors_required_settings_store_address_chooser', 'no' ) );
        $required_attr = $required ? array( 'required' => '' ) : array();

        if ( $key_exists && ! empty( $map_zoom_level ) && ! $hide_store_address_chooser && ! $hide_store_address1 ) {
            echo '<div class="wcv-cols-group wcv-horizontal-gutters wcv-cols-group-medium">';
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_vendor_store_address1',
                    array(
                        'id'                => '_wcv_store_search_address',
                        'label'             => __( 'Search address', 'wcvendors-pro' ),
                        'placeholder'       => __( 'Search address', 'wcvendors-pro' ),
                        'type'              => 'text',
                        'value'             => get_user_meta( get_current_user_id(), '_wcv_store_search_address', true ),
                        'custom_attributes' => $required_attr,
                        'wrapper_start'     => '<div class="all-60 small-100">',
                        'wrapper_end'       => '</div>',
                        'no_margin'         => true,
                    )
                )
            );
            echo '<div class="all-40 small-100"><div class="wcv-button-group small" style="margin-top: 24px;">';

            WCV_Form_Helper::html_element(
                apply_filters(
                    'wcv_show_location_picker',
                    array(
                        'id'         => 'use_current_position',
                        'element'    => 'a',
                        'content'    => __( 'Your location', 'wcvendors-pro' ),
                        'attributes' => array(
                            'class' => 'wcv-button',
                            'href'  => '#',
                        ),
                    )
                )
            );
            WCV_Form_Helper::html_element(
                apply_filters(
                    'wcv_show_location_picker',
                    array(
                        'id'         => 'show_location_picker',
                        'element'    => 'a',
                        'content'    => __( 'Show map', 'wcvendors-pro' ),
                        'attributes' => array(
                            'class' => 'wcv-button',
                            'href'  => '#',
                        ),
                    )
                )
            );
            echo '</div></div></div>';
        }
    }

    /**
     * Output map for location picker and its latitude and longitude fields
     *
     * @return void
     * @ver
     */
    public static function location_picker() {

        $hide_store_address_chooser = ( 'signup' === self::$form_type )
            ? wc_string_to_bool( get_option( 'wcvendors_hide_signup_store_address_chooser', 'no' ) )
            : wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_address_chooser', 'no' ) );
        $hide_store_address1        = ( 'signup' === self::$form_type )
            ? wc_string_to_bool( get_option( 'wcvendors_hide_signup_store_address', 'no' ) )
            : wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_address', 'no' ) );

        $api_key        = get_option( 'wcvendors_pro_google_maps_api_key', '' );
        $map_zoom_level = get_option( 'wcvendors_pro_google_maps_zoom_level', '' );
        $key_exists     = empty( $api_key ) ? false : true;

        $required = ( 'signup' === self::$form_type )
            ? wc_string_to_bool( get_option( 'wcvendors_required_signup_store_address_chooser', 'no' ) )
            : wc_string_to_bool( get_option( 'wcvendors_required_settings_store_address_chooser', 'no' ) );

        if ( $required ) {
            $required_attr = array( 'required' => '' );
        }

        if ( $key_exists && ! empty( $map_zoom_level ) && ! $hide_store_address_chooser && ! $hide_store_address1 ) {

            $visibility = apply_filters( 'wcvendors_pro_location_picker_default_visibility', get_option( 'wcvendors_pro_location_picker_default_visibility', 'no' ) );
            $visibility = 'visible' === $visibility ? 'block' : 'none';

            WCV_Form_Helper::html_element(
                apply_filters(
                    'wcv_location_picker',
                    array(
                        'id'            => 'wcv_location_picker',
                        'element'       => 'div',
                        'attributes'    => array(
                            'style' => 'width: 100%; height: 400px; display:' . $visibility,
                        ),
                        'wrapper_start' => '<div class="control-group"><div class="all-100 small-100">',
                        'wrapper_end'   => '</div></div>',
                    )
                )
            );
        }
    }

    /**
     * Output the latitude and longitude fields
     *
     * @return    void
     * @since      1.5.9
     * @version    1.5.9
     */
    public static function coordinates() {

        $hide_store_address_chooser = ( 'signup' === self::$form_type )
            ? wc_string_to_bool( get_option( 'wcvendors_hide_signup_store_address_chooser', 'no' ) )
            : wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_address_chooser', 'no' ) );
        $hide_store_address1        = ( 'signup' === self::$form_type )
            ? wc_string_to_bool( get_option( 'wcvendors_hide_signup_store_address', 'no' ) )
            : wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_address', 'no' ) );

        $api_key        = get_option( 'wcvendors_pro_google_maps_api_key', '' );
        $map_zoom_level = get_option( 'wcvendors_pro_google_maps_zoom_level', '' );
        $key_exists     = empty( $api_key ) ? false : true;

        if ( $key_exists && ! empty( $map_zoom_level ) && ! $hide_store_address_chooser && ! $hide_store_address1 ) {
            $latitude  = get_user_meta( get_current_user_id(), 'wcv_address_latitude', true );
            $longitude = get_user_meta( get_current_user_id(), 'wcv_address_longitude', true );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_address_latitude',
                    array(
                        'id'            => 'wcv_address_latitude',
                        'placeholder'   => __( 'Latitude', 'wcvendors-pro' ),
                        'label'         => __( 'Latitude', 'wcvendors-pro' ),
                        'type'          => 'text',
                        'value'         => $latitude,
                        'desc'          => __( 'This value will be set automatically after choosing your address.', 'wcvendors-pro' ),
                        'attributes'    => array( 'readonly' => 'readonly' ),
                        'wrapper_start' => '<div class="wcv-cols-group wcv-horizontal-gutters"><div class="all-50 small-100">',
                        'wrapper_end'   => '</div>',
                    )
                )
            );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_address_longitude',
                    array(
                        'id'            => 'wcv_address_longitude',
                        'placeholder'   => __( 'Longitude', 'wcvendors-pro' ),
                        'label'         => __( 'Longitude', 'wcvendors-pro' ),
                        'type'          => 'text',
                        'value'         => $longitude,
                        'desc'          => __( 'This value will be set automatically after choosing your address.', 'wcvendors-pro' ),
                        'attributes'    => array( 'readonly' => 'readonly' ),
                        'wrapper_start' => '<div class="all-50 small-100">',
                        'wrapper_end'   => '</div></div>',
                    )
                )
            );
        }
    }

    /**
     * Output a formatted store address postcode
     *
     * @since      1.2.0
     */
    public static function store_address_postcode() {

        $hide_store_postcode = ( 'signup' === self::$form_type )
            ? wc_string_to_bool( get_option( 'wcvendors_hide_signup_store_address', 'no' ) )
            : wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_address', 'no' ) );

        if ( ! $hide_store_postcode ) {

            $postcode = get_user_meta( get_current_user_id(), '_wcv_store_postcode', true );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_vendor_store_postcode',
                    array(
                        'id'            => '_wcv_store_postcode',
                        'label'         => __( 'Postcode / Zip', 'wcvendors-pro' ),
                        'placeholder'   => __( 'Postcode / Zip', 'wcvendors-pro' ),
                        'value'         => $postcode,
                        'wrapper_start' => '<div class="all-50 small-100">',
                        'wrapper_end'   => '</div></div>',
                    )
                )
            );

        }
    }

    /**
     * Output all social media fields.
     *
     * @see wcv_get_social_media_settings();
     * @version 1.7.5
     */
    public static function render_social_media_settings() {
        $settings  = wcv_get_social_media_settings();
        $vendor_id = get_current_user_id();
        foreach ( $settings as $key => $setting ) {

            $disabled = ( 'signup' === self::$form_type )
                ? get_option( 'wcvendors_hide_signup_social_' . $key, 'no' )
                : get_option( 'wcvendors_hide_settings_social_' . $key, 'no' );

            if ( 'yes' === $disabled ) {
                continue;
            }

            $value = get_user_meta( $vendor_id, $setting['id'], true );
            $args  = apply_filters( 'wcv_social_field_' . $key, array_merge( $setting, array( 'value' => $value ) ) );
            WCV_Form_Helper::input( $args );
        }
    }

    /**
     *
     *    Shipping Information
     */

    /**
     *  Output default national shipping fee field
     *
     * @since    1.2.0
     *
     * @param    array $shipping_details The shipping details.
     */
    public static function shipping_fee_national( $shipping_details ) {
        $default_shipping_fee = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_national_default_shipping_fee', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_national_default_shipping_fee', 'no' );
        if ( 'yes' !== $default_shipping_fee ) {
            $value = ( is_array( $shipping_details ) && array_key_exists( 'national', $shipping_details ) ) ? $shipping_details['national'] : '';

            $shipping_fee_national_require = apply_filters(
                'wcv_require_settings_shipping_fee_national',
                wc_string_to_bool( get_option( 'wcvendors_required_settings_shipping_national_default_shipping_fee', 'no' ) )
            );

            // Shipping Fee.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_national_fee',
                    array(
                        'id'                => '_wcv_shipping_fee_national',
                        'label'             => __( 'Default Shipping Fee', 'wcvendors-pro' ),
                        'placeholder'       => __( '0', 'wcvendors-pro' ),
                        'desc_tip'          => 'number',
                        'description'       => __( 'The default shipping fee within your country, this can be overridden on a per product basis.', 'wcvendors-pro' ),
                        'data_type'         => 'price',
                        'class'             => 'wcv-disable-national-input',
                        'value'             => $value,
                        'custom_attributes' => $shipping_fee_national_require ? array( 'required' => '' ) : array(),
                    )
                )
            );
        }

        $min_charge = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_national_min_charge', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_national_min_charge', 'no' );
        if ( 'yes' !== $min_charge ) {
            $shipping_national_min_charge_require = apply_filters(
                'wcv_require_settings_shipping_national_min_charge',
                wc_string_to_bool( get_option( 'wcvendors_required_settings_shipping_national_min_charge', 'no' ) )
            );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_national_min_charge',
                    array(
                        'id'                => '_wcv_shipping_national_min_charge',
                        'label'             => __( 'Minimum Charged Per Order', 'wcvendors-pro' ),
                        'description'       => __( 'The minimum national shipping fee charged for an order.', 'wcvendors-pro' ),
                        'placeholder'       => __( '0', 'wcvendors-pro' ),
                        'desc_tip'          => 'number',
                        'data_type'         => 'price',
                        'class'             => 'wcv-disable-national-input',
                        'value'             => $shipping_details['national_min_charge'],
                        'custom_attributes' => $shipping_national_min_charge_require ? array( 'required' => '' ) : array(),
                    )
                )
            );
        }

        $max_charge = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_national_max_charge', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_national_max_charge', 'no' );
        if ( 'yes' !== $max_charge ) {

            $shipping_national_max_charge_require = apply_filters(
                'wcv_require_settings_shipping_national_max_charge',
                wc_string_to_bool( get_option( 'wcvendors_required_settings_shipping_national_max_charge', 'no' ) )
            );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_national_max_charge',
                    array(
                        'id'                => '_wcv_shipping_national_max_charge',
                        'label'             => __( 'Maximum Shipping Charged Per Order', 'wcvendors-pro' ),
                        'description'       => __( 'The maximum national shipping fee charged for an order.', 'wcvendors-pro' ),
                        'placeholder'       => __( '0', 'wcvendors-pro' ),
                        'desc_tip'          => 'number',
                        'data_type'         => 'price',
                        'class'             => 'wcv-disable-national-input',
                        'value'             => $shipping_details['national_max_charge'],
                        'custom_attributes' => $shipping_national_max_charge_require ? array( 'required' => '' ) : array(),
                    )
                )
            );
        }

        $shipping_order = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_national_free_shipping_order', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_national_free_shipping_order', 'no' );
        if ( 'yes' !== $shipping_order ) {

            $shipping_national_free_shipping_order_require = apply_filters(
                'wcv_require_settings_shipping_national_free_shipping_order',
                wc_string_to_bool( get_option( 'wcvendors_required_settings_shipping_national_free_shipping_order', 'no' ) )
            );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_national_free_shipping_order',
                    array(
                        'id'                => '_wcv_shipping_national_free_shipping_order',
                        'label'             => __( 'Free Shipping Order', 'wcvendors-pro' ),
                        'placeholder'       => __( '0', 'wcvendors-pro' ),
                        'desc_tip'          => 'number',
                        'description'       => __( 'Free national shipping for order spends over this amount. This will override the max shipping charge above.', 'wcvendors-pro' ),
                        'data_type'         => 'price',
                        'class'             => 'wcv-disable-national-input',
                        'value'             => $shipping_details['national_free_shipping_order'],
                        'custom_attributes' => $shipping_national_free_shipping_order_require ? array( 'required' => '' ) : array(),
                        'custom_margin'     => 40,
                    )
                )
            );
        }
    }

    /**
     *  Output default national shipping qty override field
     *
     * @since    1.2.0
     *
     * @param array $shipping_details The shipping details.
     */
    public static function shipping_fee_national_qty( $shipping_details ) {
        $shipping_fee_national_qty = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_national_charge_once_per_product', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_national_charge_once_per_product', 'no' );
        if ( 'yes' !== $shipping_fee_national_qty ) {
            $qty_value = ( is_array( $shipping_details ) && array_key_exists( 'national_qty_override', $shipping_details ) ) ? $shipping_details['national_qty_override'] : 0;

            // QTY Override.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_national_qty',
                    array(
                        'id'            => '_wcv_shipping_fee_national_qty',
                        'label'         => __( 'Charge Once Per Product for National Shipping, Even if More than One Is Purchased', 'wcvendors-pro' ),
                        'type'          => 'checkbox',
                        'class'         => 'wcv-disable-national-input',
                        'value'         => $qty_value,
                        'desc_tip'      => true,
                        'description'   => __( 'Kindly be advised that this will not supersede the minimum/maximum/free shipping order thresholds.', 'wcvendors-pro' ),
                        'custom_margin' => 30,
                    )
                )
            );
        }
    }

    /**
     *  Output default national shipping qty override field
     *
     * @since    1.0.0
     *
     * @param array $shipping_details The shipping details.
     */
    public static function shipping_fee_national_free( $shipping_details ) {
        $shipping_fee_national = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_national_free_national_shipping', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_national_free_national_shipping', 'no' );
        if ( 'yes' !== $shipping_fee_national ) {
            $free = ( is_array( $shipping_details ) && array_key_exists( 'national_free', $shipping_details ) ) ? $shipping_details['national_free'] : 0;

            // QTY Override.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_national_free',
                    array(
                        'id'            => '_wcv_shipping_fee_national_free',
                        'label'         => __( 'Free National Shipping', 'wcvendors-pro' ),
                        'type'          => 'checkbox',
                        'class'         => 'wcv-disable-national-input',
                        'value'         => $free,
                        'custom_margin' => 38,
                    )
                )
            );
        }
    }

    /**
     *  Output default national shipping qty override field
     *
     * @since    1.0.0
     *
     * @param array $shipping_details The shipping details.
     */
    public static function shipping_fee_national_disable( $shipping_details ) {
        $disable_shipping = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_national_disable_shipping', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_national_disable_shipping', 'no' );
        if ( 'yes' !== $disable_shipping ) {
            $disable = ( is_array( $shipping_details ) && array_key_exists( 'national_disable', $shipping_details ) ) ? $shipping_details['national_disable'] : 0;

            // QTY Override.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_national_disable',
                    array(
                        'id'    => '_wcv_shipping_fee_national_disable',
                        'label' => __( 'Disable National Shipping', 'wcvendors-pro' ),
                        'type'  => 'checkbox',
                        'value' => $disable,
                    )
                )
            );
        }
    }

    /**
     *  Output default international shipping fee field
     *
     * @since    1.0.0
     *
     * @param array $shipping_details The shipping details.
     */
    public static function shipping_fee_international( $shipping_details ) {
        $default_shipping_fee = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_international_default_shipping_fee', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_international_default_shipping_fee', 'no' );
        if ( 'yes' !== $default_shipping_fee ) {
            $value = ( is_array( $shipping_details ) && array_key_exists( 'international', $shipping_details ) ) ? $shipping_details['international'] : '';

            $default_shipping_international_fee_require = apply_filters(
                'wcv_require_settings_shipping_international_default_shipping_fee',
                wc_string_to_bool( get_option( 'wcvendors_required_settings_shipping_international_default_shipping_fee', 'no' ) )
            );

            // Shipping Fee.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_international_fee',
                    array(
                        'id'                => '_wcv_shipping_fee_international',
                        'label'             => __( 'Default Shipping Fee', 'wcvendors-pro' ),
                        'placeholder'       => __( '0', 'wcvendors-pro' ),
                        'desc_tip'          => 'true',
                        'description'       => __( 'The default shipping fee outside your country, this can be overridden on a per product basis. ', 'wcvendors-pro' ),
                        'data_type'         => 'price',
                        'class'             => 'wcv-disable-international-input',
                        'value'             => $value,
                        'custom_attributes' => $default_shipping_international_fee_require ? array(
                            'required' => '',
                        ) : array(),
                    )
                )
            );
        }

        $min_charge = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_international_min_charge', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_international_min_charge', 'no' );
        if ( 'yes' !== $min_charge ) {

            $shipping_international_min_charge_require = apply_filters(
                'wcv_require_settings_shipping_international_min_charge',
                wc_string_to_bool( get_option( 'wcvendors_required_settings_shipping_international_min_charge', 'no' ) )
            );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_international_min_charge',
                    array(
                        'id'                => '_wcv_shipping_international_min_charge',
                        'label'             => __( 'Minimum Charged Per Order', 'wcvendors-pro' ),
                        'description'       => __( 'The minimum international shipping fee charged for an order.', 'wcvendors-pro' ),
                        'placeholder'       => __( '0', 'wcvendors-pro' ),
                        'desc_tip'          => 'number',
                        'data_type'         => 'price',
                        'class'             => 'wcv-disable-international-input',
                        'value'             => $shipping_details['international_min_charge'],
                        'custom_attributes' => $shipping_international_min_charge_require ? array(
                            'required' => '',
                        ) : array(),
                    )
                )
            );
        }

        $max_charge = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_international_max_charge', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_international_max_charge', 'no' );
        if ( 'yes' !== $max_charge ) {

            $shipping_international_max_charge_require = apply_filters(
                'wcv_require_settings_shipping_international_max_charge',
                wc_string_to_bool( get_option( 'wcvendors_required_settings_shipping_international_max_charge', 'no' ) )
            );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_international_max_charge',
                    array(
                        'id'                => '_wcv_shipping_international_max_charge',
                        'label'             => __( 'Maximum Shipping Charged Per Order', 'wcvendors-pro' ),
                        'description'       => __( 'The maximum international shipping fee charged for an order.', 'wcvendors-pro' ),
                        'placeholder'       => __( '0', 'wcvendors-pro' ),
                        'desc_tip'          => 'number',
                        'data_type'         => 'price',
                        'class'             => 'wcv-disable-international-input',
                        'value'             => $shipping_details['international_max_charge'],
                        'custom_attributes' => $shipping_international_max_charge_require ? array(
                            'required' => '',
                        ) : array(),
                    )
                )
            );
        }

        $shipping_order = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_international_free_shipping_order', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_international_free_shipping_order', 'no' );
        if ( 'yes' !== $shipping_order ) {

            $shipping_international_free_shipping_order_require = apply_filters(
                'wcv_require_settings_shipping_international_free_shipping_order',
                wc_string_to_bool( get_option( 'wcvendors_required_settings_shipping_international_free_shipping_order', 'no' ) )
            );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_international_free_shipping_order',
                    array(
                        'id'                => '_wcv_shipping_international_free_shipping_order',
                        'label'             => __( 'Free Shipping Order', 'wcvendors-pro' ),
                        'placeholder'       => __( '0', 'wcvendors-pro' ),
                        'desc_tip'          => 'number',
                        'description'       => __( 'Free international shipping for order spends over this amount. This will override the max shipping charge above.', 'wcvendors-pro' ),
                        'data_type'         => 'price',
                        'class'             => 'wcv-disable-international-input',
                        'value'             => $shipping_details['international_free_shipping_order'],
                        'custom_attributes' => $shipping_international_free_shipping_order_require ? array(
                            'required' => '',
                        ) : array(),
                        'custom_margin'     => 40,
                    )
                )
            );
        }
    }

    /**
     *  Output default international shipping fee field
     *
     * @since    1.0.0
     *
     * @param array $shipping_details The shipping details.
     */
    public static function shipping_fee_international_qty( $shipping_details ) {
        $shipping_fee_national_qty = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_international_charge_once_per_product', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_international_charge_once_per_product', 'no' );
        if ( 'yes' !== $shipping_fee_national_qty ) {
            $qty_value = ( is_array( $shipping_details ) && array_key_exists( 'international_qty_override', $shipping_details ) ) ? $shipping_details['international_qty_override'] : 0;

            // QTY Override.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_international_qty',
                    array(
                        'id'            => '_wcv_shipping_fee_international_qty',
                        'label'         => __( 'Charge Once Per Product for International Shipping, Even if More than One Is Purchased', 'wcvendors-pro' ),
                        'type'          => 'checkbox',
                        'class'         => 'wcv-disable-international-input',
                        'value'         => $qty_value,
                        'desc_tip'      => true,
                        'description'   => __( 'Kindly be advised that this will not supersede the minimum/maximum/free shipping order thresholds.', 'wcvendors-pro' ),
                        'custom_margin' => 30,
                    )
                )
            );
        }
    }

    /**
     *  Output default international shipping free field
     *
     * @since    1.0.0
     *
     * @param array $shipping_details The shipping details.
     */
    public static function shipping_fee_international_free( $shipping_details ) {
        $shipping_fee_national = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_international_free_international_shipping', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_international_free_international_shipping', 'no' );
        if ( 'yes' !== $shipping_fee_national ) {
            $free = ( is_array( $shipping_details ) && array_key_exists( 'international_free', $shipping_details ) ) ? $shipping_details['international_free'] : 0;

            // QTY Override.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_international_free',
                    array(
                        'id'            => '_wcv_shipping_fee_international_free',
                        'label'         => __( 'Free International Shipping', 'wcvendors-pro' ),
                        'type'          => 'checkbox',
                        'class'         => 'wcv-disable-international-input',
                        'value'         => $free,
                        'custom_margin' => 38,
                    )
                )
            );
        }
    }

    /**
     * Vendor shipping type
     *
     * @return void
     * @version 1.0.0
     * @since   1.0.0
     */
    public static function vendor_shipping_type() {

        if ( ! wc_string_to_bool( get_option( 'wcvendors_vendor_select_shipping', 'no' ) ) ) {
            return;
        }

        $shipping_settings   = get_option( 'woocommerce_wcv_pro_vendor_shipping_settings', wcv_get_default_vendor_shipping() );
        $store_shipping_type = get_user_meta( get_current_user_id(), '_wcv_shipping_type', true );
        $shipping_type       = ( '' !== $store_shipping_type ) ? $store_shipping_type : $shipping_settings['shipping_system'];
        $required            = wc_string_to_bool( get_option( 'wcvendors_required_settings_store_shipping_type', 'no' ) );

        if ( 'signup' === self::$form_type ) {
            $required      = wc_string_to_bool( get_option( 'wcvendors_required_signup_store_shipping_type', 'no' ) );
            $shipping_type = '';
        }
        $required_attr = $required ? array( 'required' => '' ) : array();

        WCV_Form_Helper::select(
            apply_filters(
                'wcv_vendor_shipping_type',
                array(
                    'id'                => '_wcv_shipping_type',
                    'type'              => 'select',
                    'label'             => __( 'Shipping type', 'wcvendors-pro' ),
                    'class'             => 'wcv-shipping-type',
                    'options'           => array_merge( array( '' => '' ), WCVendors_Pro_Shipping_Controller::shipping_types() ),
                    'value'             => $shipping_type,
                    'custom_attributes' => $required_attr,
                )
            )
        );
    }

    /**
     *  Output default international shipping free field
     *
     * @since    1.0.0
     * @param array $shipping_details The shipping details.
     */
    public static function shipping_fee_international_disable( $shipping_details ) {
        $disable_shipping = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_international_disable_shipping', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_international_disable_shipping', 'no' );
        if ( 'yes' !== $disable_shipping ) {
            $disable = ( is_array( $shipping_details ) && array_key_exists( 'international_disable', $shipping_details ) ) ? $shipping_details['international_disable'] : 0;

            // QTY Override.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_international_disable',
                    array(
                        'id'    => '_wcv_shipping_fee_international_disable',
                        'label' => __( 'Disable International Shipping', 'wcvendors-pro' ),
                        'type'  => 'checkbox',
                        'value' => $disable,
                    )
                )
            );
        }
    }

    /**
     *  Output the shipping rate depending on the admin settings
     *
     * @since    1.0.0
     */
    public static function shipping_rates() {

        $shipping_settings     = get_option( 'woocommerce_wcv_pro_vendor_shipping_settings', wcv_get_default_vendor_shipping() );
        $shipping_details      = get_user_meta( get_current_user_id(), '_wcv_shipping', true );
        $shipping_details      = wp_parse_args( $shipping_details, WCVendors_Pro_Shipping_Controller::get_shipping_defaults() );
        $national_disable      = isset( $shipping_settings['national_disable'] ) ? wc_string_to_bool( $shipping_settings['national_disable'] ) : false;
        $international_disable = isset( $shipping_settings['international_disable'] ) ? wc_string_to_bool( $shipping_settings['international_disable'] ) : false;

        echo '<div id="shipping-flat-rates" class="wcv-cols-group wcv-horizontal-gutters">';

        if ( ! $national_disable ) {
            if ( $international_disable ) {
                echo '<div class="all-100 small-100">';
            } else {
                echo '<div class="all-50 small-100">';
            }

            printf(
                '<label class="wcv-sub-heading" style="margin-bottom: 40px;">%s<span class="vertical-middle">%s</span></label>',
                wp_kses(
                    wcv_get_icon( 'wcv-icon wcv-icon-24 wcv-icon-middle wcv-icon-left', 'wcv-icon-national' ),
                    wcv_allowed_html_tags()
                ),
                esc_html__( 'National', 'wcvendors-pro' )
            );

            self::shipping_fee_national( $shipping_details );
            self::shipping_fee_national_free( $shipping_details );
            self::shipping_fee_national_qty( $shipping_details );
            self::shipping_fee_national_disable( $shipping_details );
            echo '</div>';

        }

        if ( ! $international_disable ) {
            if ( $national_disable ) {
                echo '<div class="all-100 small-100">';
            } else {
                echo '<div class="all-50 small-100">';
            }

            echo '<div class="wcv_mobile control-group">&nbsp;</div>';

            printf(
                '<label class="wcv-sub-heading" style="margin-bottom: 40px;">%s<span class="vertical-middle">%s</span></label>',
                wp_kses(
                    wcv_get_icon( 'wcv-icon wcv-icon-24 wcv-icon-middle wcv-icon-left', 'wcv-icon-international' ),
                    wcv_allowed_html_tags()
                ),
                esc_html__( 'International', 'wcvendors-pro' )
            );

            self::shipping_fee_international( $shipping_details );
            self::shipping_fee_international_free( $shipping_details );
            self::shipping_fee_international_qty( $shipping_details );
            self::shipping_fee_international_disable( $shipping_details );

            echo '</div>';

        }

        echo '</div>';

        self::shipping_rate_table();

        // Backwards compatibility
        // This has been moved into the store-settings template for 1.3.7 and above.
        if ( version_compare( WCV_PRO_VERSION, '1.3.7', '<' ) ) {

            self::product_handling_fee( $shipping_details );
            self::shipping_policy( $shipping_details );
            self::return_policy( $shipping_details );
            self::shipping_from( $shipping_details );
            self::shipping_address( $shipping_details );

        }
    }

    /**
     *  Output default product handling fee field
     *
     * @since    1.0.0
     *
     * @param array $shipping_details The shipping details.
     */
    public static function product_handling_fee( $shipping_details ) {

        $shipping = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_handling_fee', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_handling_fee', 'no' );

        if ( 'yes' !== $shipping ) {

            $value = ( is_array( $shipping_details ) && array_key_exists( 'product_handling_fee', $shipping_details ) ) ? $shipping_details['product_handling_fee'] : '';

            // Product handling Fee.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_product_fee',
                    array(
                        'id'          => '_wcv_shipping_product_handling_fee',
                        'label'       => __( 'Product handling fee', 'wcvendors-pro' ),
                        'placeholder' => __( 'Leave empty to disable', 'wcvendors-pro' ),
                        'desc_tip'    => 'true',
                        'description' => __( 'The product handling fee, this can be overridden on a per product basis. Amount (5.00) or Percentage (5%).', 'wcvendors-pro' ),
                        'type'        => 'text',
                        'value'       => $value,
                    )
                )
            );

        }
    }

    /**
     *  Output the max order spend for shipping
     *
     * @since    1.0.0
     *
     * @param array $shipping_details The shipping details.
     */
    public static function order_min_charge( $shipping_details ) {

        $value = ( is_array( $shipping_details ) && array_key_exists( 'min_charge', $shipping_details ) ) ? $shipping_details['min_charge'] : '';

        // Order min Spend.
        WCV_Form_Helper::input(
            apply_filters(
                'wcv_shipping_min_charge',
                array(
                    'id'          => '_wcv_shipping_min_charge',
                    'label'       => __( 'Minimum shipping charged per order', 'wcvendors-pro' ),
                    'placeholder' => __( '0', 'wcvendors-pro' ),
                    'desc_tip'    => 'true',
                    'description' => __( 'The minimum shipping fee charged for an order.', 'wcvendors-pro' ),
                    'data_type'   => 'price',
                    'value'       => $value,
                )
            )
        );
    }

    /**
     * Output the max order spend for shipping
     *
     * @since    1.0.0
     *
     * @param array $shipping_details The shipping details.
     */
    public static function order_max_charge( $shipping_details ) {

        $value = ( is_array( $shipping_details ) && array_key_exists( 'max_charge', $shipping_details ) ) ? $shipping_details['max_charge'] : '';

        // Order min spend.
        WCV_Form_Helper::input(
            apply_filters(
                'wcv_shipping_max_charge',
                array(
                    'id'          => '_wcv_shipping_max_charge',
                    'label'       => __( 'Maximum shipping charged per order', 'wcvendors-pro' ),
                    'placeholder' => __( '0', 'wcvendors-pro' ),
                    'desc_tip'    => 'true',
                    'description' => __( 'The maximum shipping fee charged for an order.', 'wcvendors-pro' ),
                    'data_type'   => 'price',
                    'value'       => $value,
                )
            )
        );
    }

    /**
     *  Output the min and max order spend for shipping
     *
     * @since    1.0.0
     * @deprecated 1.7.4
     *
     * @param array $shipping_details The shipping details.
     */
    public static function order_shipping_charge( $shipping_details ) {
        $max = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_max_charge', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_max_charge', 'no' );
        $min = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_min_charge', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_min_charge', 'no' );
        $max = ! wc_string_to_bool( $max );
        $min = ! wc_string_to_bool( $min );

        if ( ! $max && ! $min ) {
            return;
        }

        printf(
            '<p><b>%s</b></p>',
            esc_html__( 'Notice: These settings will soon be removed and replaced with the new settings above.', 'wcvendors-pro' )
        );

        if ( ! $min ) {
            self::order_max_charge( $shipping_details );
            return;
        }

        if ( ! $max ) {
            self::order_min_charge( $shipping_details );
            return;
        }
        ?>

        <div class="wcv-cols-group wcv-horizontal-gutters">
            <div class="all-50 small-100">
                <?php self::order_min_charge( $shipping_details ); ?>
            </div>
            <div class="all-50 small-100">
                <?php self::order_max_charge( $shipping_details ); ?>
            </div>
        </div>
        <?php
    }

    /**
     *  Output the free shipping for order spend for shipping
     *
     * @since    1.0.0
     * @deprecated 1.7.4
     * @param array $shipping_details The shipping details.
     */
    public static function free_shipping_order( $shipping_details ) {

        $shipping = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_free_shipping_order', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_free_shipping_order', 'no' );

        if ( 'yes' !== $shipping ) {

            $value = ( is_array( $shipping_details ) && array_key_exists( 'free_shipping_order', $shipping_details ) ) ? $shipping_details['free_shipping_order'] : '';

            // Order min Spend.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_free_shipping_order',
                    array(
                        'id'          => '_wcv_shipping_free_shipping_order',
                        'label'       => __( 'Free shipping order', 'wcvendors-pro' ),
                        'placeholder' => __( '0', 'wcvendors-pro' ),
                        'desc_tip'    => 'true',
                        'description' => __( 'Free shipping for order spends over this amount. This will override the max shipping charge above.', 'wcvendors-pro' ),
                        'data_type'   => 'price',
                        'value'       => $value,
                    )
                )
            );

        }
    }

    /**
     *  Output the free shipping for product
     *
     * @since    1.0.0
     * @deprecated 1.7.4
     * @param array $shipping_details The shipping details.
     */
    public static function free_shipping_product( $shipping_details ) {

        $shipping = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_free_shipping_product', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_free_shipping_product', 'no' );

        if ( 'yes' !== $shipping ) {
            $value = ( is_array( $shipping_details ) && array_key_exists( 'free_shipping_product', $shipping_details ) ) ? $shipping_details['free_shipping_product'] : '';

            // Order min Spend.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_free_shipping_product',
                    array(
                        'id'          => '_wcv_shipping_free_shipping_product',
                        'label'       => __( 'Free shipping product', 'wcvendors-pro' ),
                        'placeholder' => __( '0', 'wcvendors-pro' ),
                        'desc_tip'    => 'true',
                        'description' => __( 'Free shipping if the spend per product is over this amount. This will override the max shipping charge above.', 'wcvendors-pro' ),
                        'data_type'   => 'price',
                        'value'       => $value,
                    )
                )
            );

        }
    }

    /**
     *  Output the max product spend for shipping
     *
     * @since    1.0.0
     * @deprecated 1.7.4
     * @param array $shipping_details The shipping details.
     */
    public static function product_max_charge( $shipping_details ) {

        $shipping = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_max_charge_product', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_max_charge_product', 'no' );

        if ( 'yes' !== $shipping ) {

            $value = ( is_array( $shipping_details ) && array_key_exists( 'max_charge_product', $shipping_details ) ) ? $shipping_details['max_charge_product'] : '';

            // Order min spend.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_shipping_max_charge_product',
                    array(
                        'id'          => '_wcv_shipping_max_charge_product',
                        'label'       => __( 'Maximum product charge', 'wcvendors-pro' ),
                        'placeholder' => __( '0', 'wcvendors-pro' ),
                        'desc_tip'    => 'true',
                        'description' => __( 'The maximum shipping charged per product no matter the quantity.', 'wcvendors-pro' ),
                        'data_type'   => 'price',
                        'value'       => $value,
                    )
                )
            );
        }
    }

    /**
     *  Output shipping policy field
     *
     * @since    1.0.0
     * @param array $shipping_details The shipping details.
     */
    public static function shipping_policy( $shipping_details ) {

        $hide = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_shipping_policy', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_shipping_policy', 'no' );

        if ( wc_string_to_bool( $hide ) ) {
            return;
        }

        $required = wc_string_to_bool( get_option( 'wcvendors_required_settings_shipping_return_policy', 'no' ) );
        if ( 'signup' === self::$form_type ) {
            $required = wc_string_to_bool( get_option( 'wcvendors_required_signup_shipping_return_policy', 'no' ) );
        }
        $required_attr = $required ? array( 'required' => '' ) : array();

        $enable_media        = wc_string_to_bool( get_option( 'wcvendors_allow_editor_media', 'no' ) );
        $store_policies_html = wc_string_to_bool( get_option( 'wcvendors_allow_settings_policy_html', 'no' ) );
        if ( 'signup' === self::$form_type ) {
            $store_policies_html = wc_string_to_bool( get_option( 'wcvendors_allow_signup_policy_html', 'no' ) );
        }

        $value = ( is_array( $shipping_details ) && array_key_exists( 'shipping_policy', $shipping_details ) ) ? $shipping_details['shipping_policy'] : '';
        $label = __( 'Shipping Policy', 'wcvendors-pro' );

        $allow_markup = 'yes' === get_option( 'wcvendors_allow_form_markup', 'no' ) ? true : false;

        // If html in info is allowed then display the tinyMCE otherwise just display a text box.
        if ( $store_policies_html && $allow_markup ) {

            if ( $required ) {
                add_filter( 'the_editor', array( __CLASS__, 'wp_editor_required' ) );
                add_filter( 'tiny_mce_before_init', array( __CLASS__, 'wp_tinymce_required' ) );
                add_filter( 'teeny_mce_before_init', array( __CLASS__, 'wp_tinymce_required' ) );
            }

            $required_class = $required ? 'wcv-required' : '';

            $settings = apply_filters(
                'wcv_vendor_store_policy_editor_settings',
                array(
                    'editor_height' => 200,
                    'media_buttons' => $enable_media,
                    'teeny'         => true,
                    'editor_class'  => $required_class,
                    'tinymce'       => array(
                        'setup' => 'function (editor) {
                            editor.on("change", function () {
                                var content = tinyMCE.activeEditor.getContent( {format : "raw"} )
                                    .replace( \'<p><br data-mce-bogus="1"></p>\', "" );

                                if ( content != undefined && content != "" ) {
                                    jQuery( "#" + editor.id ).html( content );
                                }
                            });
                        }',
                    ),
                )
            );

            echo '<label>' . esc_html( $label ) . '</label>';

            wp_editor( $value, '_wcv_shipping_policy', $settings );

        } else {

            WCV_Form_Helper::textarea(
                apply_filters(
                    'wcv_vendor_shipping_policy',
                    array(
                        'id'                => '_wcv_shipping_policy',
                        'label'             => $label,
                        'value'             => $value,
                        'custom_attributes' => $required_attr,
                    )
                )
            );
        }
    }

    /**
     *  Output shipping policy field
     *
     * @since    1.0.0
     * @param array $shipping_details The shipping details.
     */
    public static function return_policy( $shipping_details ) {

        $hide = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_shipping_return_policy', 'no' )
            : get_option( 'wcvendors_hide_settings_shipping_return_policy', 'no' );

        if ( wc_string_to_bool( $hide ) ) {
            return;
        }

        $required = wc_string_to_bool( get_option( 'wcvendors_required_settings_shipping_return_policy', 'no' ) );
        if ( 'signup' === self::$form_type ) {
            $required = wc_string_to_bool( get_option( 'wcvendors_required_signup_shipping_return_policy', 'no' ) );
        }
        $required_attr = $required ? array( 'required' => '' ) : array();

        $enable_media        = wc_string_to_bool( get_option( 'wcvendors_allow_editor_media', 'no' ) );
        $store_policies_html = wc_string_to_bool( get_option( 'wcvendors_allow_settings_policy_html', 'no' ) );
        if ( 'signup' === self::$form_type ) {
            $store_policies_html = wc_string_to_bool( get_option( 'wcvendors_allow_signup_policy_html', 'no' ) );
        }

        $value = ( is_array( $shipping_details ) && array_key_exists( 'return_policy', $shipping_details ) ) ? $shipping_details['return_policy'] : '';
        $label = __( 'Return Policy', 'wcvendors-pro' );

        $allow_markup = 'yes' === get_option( 'wcvendors_allow_form_markup', 'no' ) ? true : false;

        // If html in info is allowed then display the tinyMCE otherwise just display a text box.
        if ( $store_policies_html && $allow_markup ) {

            if ( $required ) {
                add_filter( 'the_editor', array( __CLASS__, 'wp_editor_required' ) );
                add_filter( 'tiny_mce_before_init', array( __CLASS__, 'wp_tinymce_required' ) );
                add_filter( 'teeny_mce_before_init', array( __CLASS__, 'wp_tinymce_required' ) );
            }

            $required_class = $required ? 'wcv-required' : '';

            $settings = apply_filters(
                'wcv_vendor_store_policy_editor_settings',
                array(
                    'editor_height' => 200,
                    'media_buttons' => $enable_media,
                    'teeny'         => true,
                    'editor_class'  => $required_class,
                    'tinymce'       => array(
                        'setup' => 'function (editor) {
                            editor.on("change", function () {
                                var content = tinyMCE.activeEditor.getContent( {format : "raw"} )
                                    .replace( \'<p><br data-mce-bogus="1"></p>\', "" );

                                if ( content != undefined && content != "" ) {
                                    jQuery( "#" + editor.id ).html( content );
                                }
                            });
                        }',
                    ),
                )
            );

            echo '<label>' . esc_html( $label ) . '</label>';

            wp_editor( $value, '_wcv_shipping_return_policy', $settings );

        } else {

            WCV_Form_Helper::textarea(
                apply_filters(
                    'wcv_shipping_return_policy',
                    array(
                        'id'                => '_wcv_shipping_return_policy',
                        'label'             => $label,
                        'value'             => $value,
                        'custom_attributes' => $required_attr,
                    )
                )
            );
        }
    }

    /**
     *  Output shipping type
     *
     * @since    1.0.0
     *
     * @param array $shipping_details The shipping details.
     */
    public static function shipping_from( $shipping_details ) {

        $value = ( is_array( $shipping_details ) && array_key_exists( 'shipping_from', $shipping_details ) ) ? $shipping_details['shipping_from'] : '';

        // shipping from.
        WCV_Form_Helper::select(
            apply_filters(
                'wcv_vendor_shipping_from',
                array(
                    'id'            => '_wcv_shipping_from',
                    'class'         => 'select2',
                    'label'         => __( 'Shipping from', 'wcvendors-pro' ),
                    'desc_tip'      => 'true',
                    'description'   => __( 'Where products will be shipped from.', 'wcvendors-pro' ),
                    'wrapper_start' => '<div class="all-100">',
                    'wrapper_end'   => '</div>',
                    'value'         => $value,
                    'options'       => array(
                        'store_address' => __( 'Store Address', 'wcvendors-pro' ),
                        'other'         => __( 'Other', 'wcvendors-pro' ),
                    ),
                )
            )
        );
    }

    /**
     * Output a formatted store address state
     *
     * @since      1.2.0
     */
    public static function store_address_state() {

        $hide_store_state = ( 'signup' === self::$form_type )
            ? wc_string_to_bool( get_option( 'wcvendors_hide_signup_store_address', 'no' ) )
            : wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_address', 'no' ) );
        $required         = ( 'signup' === self::$form_type )
            ? wc_string_to_bool( get_option( 'wcvendors_required_signup_store_address', 'no' ) )
            : wc_string_to_bool( get_option( 'wcvendors_required_settings_store_address', 'no' ) );
        if ( ! $hide_store_state ) {

            $required_attr = $required ? array( 'required' => '' ) : array();
            $state         = get_user_meta( get_current_user_id(), '_wcv_store_state', true );

            WCVendors_Pro_Form_Helper::input(
                apply_filters(
                    'wcv_vendor_store_state',
                    array(
                        'id'                => '_wcv_store_state',
                        'label'             => __( 'State / County', 'wcvendors-pro' ),
                        'placeholder'       => __( 'State / County', 'wcvendors-pro' ),
                        'value'             => $state,
                        'class'             => 'js_field-state',
                        'wrapper_start'     => '<div class="wcv-cols-group wcv-horizontal-gutters"><div class="all-50 small-100">',
                        'wrapper_end'       => '</div>',
                        'custom_attributes' => $required_attr,
                    )
                )
            );

        }
    }

    /**
     * Output a formatted store address
     *
     * @since      1.0.0
     *
     * @param      array $shipping_details the shipping details meta.
     */
    public static function shipping_address( $shipping_details ) {

        $value = ( is_array( $shipping_details ) && array_key_exists( 'shipping_address', $shipping_details ) ) ? $shipping_details['shipping_address'] : '';

        $address1 = ( is_array( $value ) && array_key_exists( 'address1', $value ) ) ? $value['address1'] : '';
        $address2 = ( is_array( $value ) && array_key_exists( 'address2', $value ) ) ? $value['address2'] : '';
        $city     = ( is_array( $value ) && array_key_exists( 'city', $value ) ) ? $value['city'] : '';
        $state    = ( is_array( $value ) && array_key_exists( 'state', $value ) ) ? $value['state'] : '';
        $country  = ( is_array( $value ) && array_key_exists( 'country', $value ) ) ? $value['country'] : '';
        $postcode = ( is_array( $value ) && array_key_exists( 'postcode', $value ) ) ? $value['postcode'] : '';

        include apply_filters( 'wcvendors_pro_store_form_shipping_address_path', 'wcvendors-pro-shipping-address.php' );
    }

    /**
     *  Output shipping rate table
     *
     * @since    1.0.0
     */
    public static function shipping_rate_table() {

        $helper_text = apply_filters( 'wcv_shipping_rate_table_msg', __( 'Countries must use the international standard for two letter country codes. eg. AU for Australia.', 'wcvendors-pro' ) );

        $shipping_rates = get_user_meta( get_current_user_id(), '_wcv_shipping_rates', true );

        include_once apply_filters( 'wcvendors_pro_store_form_shipping_rate_table_path', 'partials/wcvendors-pro-shipping-table.php' );
    }

    /**
     *  Output vacation mode
     *
     * @since    1.3.0
     */
    public static function vacation_mode() {

        $hide_vacation_mode = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_store_vacation_mode', 'no' )
            : get_option( 'wcvendors_hide_settings_store_vacation_mode', 'no' );

        if ( ! wc_string_to_bool( $hide_vacation_mode ) ) {

            $vacation_mode = get_user_meta( get_current_user_id(), '_wcv_vacation_mode', true );
            $vacation_msg  = get_user_meta( get_current_user_id(), '_wcv_vacation_mode_msg', true );

            $disable_cart = get_user_meta( get_current_user_id(), '_wcv_vacation_disable_cart', true );

            // Vacation Mode.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_vacation_mode',
                    array(
                        'id'            => '_wcv_vacation_mode',
                        'label'         => __( 'Enable vacation mode', 'wcvendors-pro' ),
                        'type'          => 'checkbox',
                        'class'         => 'wcv-vacaction-mode',
                        'wrapper_start' => '<div class="wcv-cols-group wcv-horizontal-gutters"><div class="all-100">',
                        'wrapper_end'   => '</div>',
                        'value'         => $vacation_mode,
                    )
                )
            );

            WCV_Form_Helper::textarea(
                apply_filters(
                    'wcv_vacation_mode_msg',
                    array(
                        'id'            => '_wcv_vacation_mode_msg',
                        'label'         => __( 'Vacation message', 'wcvendors-pro' ),
                        'class'         => 'wcv-vacaction-mode-msg',
                        'wrapper_start' => '<div class="all-100 wcv-vacation-mode-msg-wrapper">',
                        'wrapper_end'   => '</div>',
                        'value'         => $vacation_msg,
                    )
                )
            );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_vacation_disable_cart',
                    array(
                        'id'            => '_wcv_vacation_disable_cart',
                        'label'         => __( 'Disable add to cart for my products', 'wcvendors-pro' ),
                        'type'          => 'checkbox',
                        'class'         => 'wcv-vacation-disable-cart',
                        'wrapper_start' => '<div class="all-100 wcv-vacation-mode-msg-wrapper">',
                        'wrapper_end'   => '</div></div>',
                        'value'         => $disable_cart,
                    )
                )
            );
        }
    }

    /**
     *  Show product total sales
     *
     * @since    1.5.8
     * @version  1.5.8
     * @return   void
     */
    public static function product_total_sales() {

        if ( ! wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_product_total_sales', 'no' ) ) ) {

            $show_product_total_sales  = get_user_meta( get_current_user_id(), '_wcv_show_product_total_sales', true );
            $product_total_sales_label = WCVendors_Pro_Vendor_Controller::get_total_sales_label( get_current_user_id(), 'product' );
            $store_total_sales_label   = WCVendors_Pro_Vendor_Controller::get_total_sales_label( get_current_user_id(), 'store' );
            // Show product sold count.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_show_product_total_sales',
                    array(
                        'id'            => '_wcv_show_product_total_sales',
                        'label'         => __( 'Show product total sales', 'wcvendors-pro' ),
                        'type'          => 'checkbox',
                        'class'         => 'wcv-product-total-sales',
                        'wrapper_start' => '<div class="wcv-cols-group wcv-horizontal-gutters"><div class="all-100">',
                        'wrapper_end'   => '</div></div>',
                        'value'         => $show_product_total_sales,
                    )
                )
            );

            // Product sold label.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_product_total_sales_label',
                    array(
                        'id'            => '_wcv_product_total_sales_label',
                        'label'         => __( 'Product Total Sales Label', 'wcvendors-pro' ),
                        'type'          => 'text',
                        'wrapper_start' => '<div class="wcv-cols-group wcv-horizontal-gutters"><div class="all-100">',
                        'wrapper_end'   => '</div></div>',
                        'value'         => $product_total_sales_label,
                    )
                )
            );

            // Store total sales label.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_store_total_sales_label',
                    array(
                        'id'            => '_wcv_store_total_sales_label',
                        'label'         => __( 'Store total sales label', 'wcvendors-pro' ),
                        'type'          => 'text',
                        'wrapper_start' => '<div class="wcv-cols-group wcv-horizontal-gutters"><div class="all-100">',
                        'wrapper_end'   => '</div></div>',
                        'value'         => $store_total_sales_label,
                    )
                )
            );
        }
    }

    /**
     * Output enable store notice field
     *
     * @return    void
     * @since      1.5.9
     * @version    1.5.9
     */
    public static function enable_store_notice() {
        if ( ! wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_enable_notice', 'no' ) ) ) {
            $enable_store_notice = get_user_meta( get_current_user_id(), '_wcv_vendor_enable_store_notice', true );
            // Store total sales label.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_vendor_enable_store_notice',
                    array(
                        'id'            => '_wcv_vendor_enable_store_notice',
                        'label'         => __( 'Enable store notice', 'wcvendors-pro' ),
                        'type'          => 'checkbox',
                        'wrapper_start' => '<div class="wcv-cols-group wcv-horizontal-gutters"><div class="all-100">',
                        'wrapper_end'   => '</div></div>',
                        'value'         => $enable_store_notice,
                    )
                )
            );
        }
    }

    /**
     * Output vendor store notice editor
     *
     * @return    void
     * @since      1.5.9
     * @version    1.7.7
     */
    public static function vendor_store_notice() {

        if ( ! wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_vendor_store_notice', 'no' ) ) ) {

            $content = get_user_meta( get_current_user_id(), '_wcv_vendor_store_notice', true );

            if ( 'yes' === get_option( 'wcvendors_allow_settings_store_notice', 'no' ) ) {

                wp_editor(
                    wp_kses_post( $content ),
                    '_wcv_vendor_store_notice',
                    array(
                        'wpautop'       => true,
                        'media_buttons' => true,
                        'textarea_name' => '_wcv_vendor_store_notice',
                        'textarea_rows' => apply_filters( 'wcv_store_notice_edit_rows', 4 ),
                        'tinymce'       => array(
                            'setup' => 'function (editor) {
                                jQuery( "#" + editor.id ).attr( "name", editor.id );

                                editor.on("change", function () {
                                    var content = tinyMCE.activeEditor.getContent( {format : "raw"} )
                                        .replace( \'<p><br data-mce-bogus="1"></p>\', "" );

                                    if ( content != undefined && content != "" ) {
                                        jQuery( "#" + editor.id ).html( content );
                                    }
                                });
                            }',
                        ),
                    )
                );
            } else {
                WCV_Form_Helper::textarea(
                    apply_filters(
                        '_wcv_vendor_store_notice',
                        array(
                            'id'            => '_wcv_vendor_store_notice',
                            'value'         => wp_strip_all_tags( $content ),
                            'wrapper_start' => '<div id="wp-_wcv_vendor_store_notice-wrap">',
                            'wrapper_end'   => '</div>',
                        )
                    )
                );
            }
        }
    }

    /**
     * Output Vendor terms on the signup page
     *
     * @since 1.3.2
     */
    public static function vendor_terms() {

        $terms_page = get_option( 'wcvendors_vendor_terms_page_id', null );

        if ( ( $terms_page ) && ( ! isset( $_GET['terms'] ) ) ) { // phpcs:ignore

            // Vendor Terms checkbox.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_vendor_terms_args',
                    array(
                        'id'                => '_wcv_agree_to_terms',
                        'label'             => sprintf(
                            // translators: %s is the link to the terms and conditions page.
                            __( 'I have read and accepted the <a href="%s" target="_blank">terms and conditions</a>', 'wcvendors-pro' ),
                            get_permalink( $terms_page )
                        ),
                        'type'              => 'checkbox',
                        'class'             => '',
                        'wrapper_start'     => '<div class="wcv-cols-group wcv-horizontal-gutters"><div class="all-100">',
                        'wrapper_end'       => '</div>',
                        'value'             => 1,
                        'custom_attributes' => array(
                            'required'                   => '',
                            'data-parsley-error-message' => sprintf(
                                // translators: %s is the name of the vendor.
                                __( 'You must agree to the terms and conditions to apply to be a %s.', 'wcvendors-pro' ),
                                wcv_get_vendor_name( true, false )
                            ),
                        ),
                    )
                )
            );
        }
    }


    /**
     *
     *    Store SEO
     */

    /**
     * Store SEO
     *
     * @since 1.5.0
     */
    public static function seo_title() {
        $seo_option = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_seo_title', 'no' )
            : get_option( 'wcvendors_hide_settings_seo_title', 'no' );

        if ( 'yes' !== $seo_option ) {

            $value = get_user_meta( get_current_user_id(), 'wcv_seo_title', true );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_vendor_seo_title',
                    array(
                        'id'          => 'wcv_seo_title',
                        'label'       => __( 'SEO Title', 'wcvendors-pro' ),
                        'placeholder' => '',
                        'desc_tip'    => 'true',
                        'description' => __( 'The SEO title shown in search engines', 'wcvendors-pro' ),
                        'type'        => 'text',
                        'value'       => $value,
                    )
                )
            );
        }
    }

    /**
     * Store SEO
     *
     * @since 1.5.0
     */
    public static function seo_meta_description() {

        $seo_option = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_seo_meta_description', 'no' )
            : get_option( 'wcvendors_hide_settings_seo_meta_description', 'no' );

        if ( 'yes' !== $seo_option ) {

            $value = get_user_meta( get_current_user_id(), 'wcv_seo_meta_description', true );

            WCV_Form_Helper::textarea(
                apply_filters(
                    'wcv_vendor_seo_meta_description',
                    array(
                        'id'          => 'wcv_seo_meta_description',
                        'label'       => __( 'Meta description ', 'wcvendors-pro' ),
                        'desc_tip'    => 'true',
                        'description' => __( 'This should be a maximum of 300 characters', 'wcvendors-pro' ),
                        'value'       => $value,
                        'rows'        => 3,
                    )
                )
            );
        }
    }

    /**
     * Store SEO
     *
     * @since 1.5.0
     */
    public static function seo_meta_keywords() {
        $seo_option = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_seo_meta_keywords', 'no' )
            : get_option( 'wcvendors_hide_settings_seo_meta_keywords', 'no' );

        if ( 'yes' !== $seo_option ) {

            $value = get_user_meta( get_current_user_id(), 'wcv_seo_meta_keywords', true );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_vendor_seo_meta_keywords',
                    array(
                        'id'          => 'wcv_seo_meta_keywords',
                        'label'       => __( 'Meta keywords ', 'wcvendors-pro' ),
                        'placeholder' => '',
                        'desc_tip'    => 'true',
                        'description' => __( 'A comma separated list of keywords', 'wcvendors-pro' ),
                        'type'        => 'text',
                        'value'       => $value,
                    )
                )
            );
        }
    }

    /**
     * Store SEO Facebook Title
     *
     * @since 1.5.0
     */
    public static function seo_fb_title() {
        $seo_option = ( 'signup' === self::$form_type ) ? get_option( 'wcvendors_hide_signup_seo_fb_title', 'no' ) : get_option( 'wcvendors_hide_settings_seo_fb_title', 'no' );

        if ( 'yes' !== $seo_option ) {

            $value = get_user_meta( get_current_user_id(), 'wcv_seo_fb_title', true );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_vendor_seo_fb_title',
                    array(
                        'id'          => 'wcv_seo_fb_title',
                        'label'       => __( 'Title', 'wcvendors-pro' ),
                        'placeholder' => '',
                        'desc_tip'    => true,
                        'description' => __( 'Facebook title of the page you are sharing.', 'wcvendors-pro' ),
                        'type'        => 'text',
                        'value'       => $value,
                    )
                )
            );
        }
    }

    /**
     * Store SEO Facebook Description
     *
     * @since 1.5.0
     */
    public static function seo_fb_description() {
        $seo_option = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_seo_fb_description', 'no' )
            : get_option( 'wcvendors_hide_settings_seo_fb_description', 'no' );

        if ( 'yes' !== $seo_option ) {

            $value = get_user_meta( get_current_user_id(), 'wcv_seo_fb_description', true );

            WCV_Form_Helper::textarea(
                apply_filters(
                    'wcv_vendor_seo_fb_description',
                    array(
                        'id'           => 'wcv_seo_fb_description',
                        'label'        => __( 'Description ', 'wcvendors-pro' ),
                        'value'        => $value,
                        'rows'         => 3,
                        'description'  => __( 'Maximum of 200 characters.', 'wcvendors-pro' ),
                        'desc_tip'     => true,
                        'show_tooltip' => false,
                    )
                )
            );
        }
    }

    /**
     * SEO Facebook Image
     *
     * @since 1.5.0
     */
    public static function seo_fb_image() {

        $seo_option = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_seo_fb_image', 'no' )
            : get_option( 'wcvendors_hide_settings_seo_fb_image', 'no' );

        if ( 'yes' !== $seo_option ) {

            $value = get_user_meta( get_current_user_id(), 'wcv_seo_fb_image_id', true );
            echo '<div class="control-group">';
            WCVendors_Pro_Form_Helper::pro_file_uploader(
                apply_filters(
                    'wcv_vendor_seo_fb_image',
                    array(
                        'id'             => 'wcv_seo_fb_image_id',
                        'header_text'    => __( 'Facebook Image', 'wcvendors-pro' ),
                        'add_text'       => __( 'Add Facebook Image', 'wcvendors-pro' ),
                        'remove_text'    => __( 'Remove Facebook Image', 'wcvendors-pro' ),
                        'image_meta_key' => 'wcv_seo_fb_image_id',
                        'save_button'    => __( 'Add Facebook Image', 'wcvendors-pro' ),
                        'window_title'   => __( 'Select an Image', 'wcvendors-pro' ),
                        'value'          => $value,
                        'size'           => 'medium',
                    )
                )
            );
            echo '</div>';

        }
    }

    /**
     * Store SEO Facebook Title
     *
     * @since 1.5.0
     */
    public static function seo_twitter_title() {
        $seo_option = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_seo_twitter_title', 'no' )
            : get_option( 'wcvendors_hide_settings_seo_twitter_title', 'no' );

        if ( 'yes' !== $seo_option ) {

            $value = get_user_meta( get_current_user_id(), 'wcv_seo_twitter_title', true );

            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_vendor_seo_twitter_title',
                    array(
                        'id'          => 'wcv_seo_twitter_title',
                        'label'       => __( 'Title', 'wcvendors-pro' ),
                        'placeholder' => '',
                        'desc_tip'    => true,
                        'description' => __( 'X (Twitter) title of the page you are sharing.', 'wcvendors-pro' ),
                        'type'        => 'text',
                        'value'       => $value,
                    )
                )
            );
        }
    }

    /**
     * Store SEO Twitter Description
     *
     * @since 1.5.0
     */
    public static function seo_twitter_description() {
        $seo_option = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_seo_twitter_description', 'no' )
            : get_option( 'wcvendors_hide_settings_seo_twitter_description', 'no' );

        if ( 'yes' !== $seo_option ) {

            $value = get_user_meta( get_current_user_id(), 'wcv_seo_twitter_description', true );

            WCV_Form_Helper::textarea(
                apply_filters(
                    'wcv_vendor_seo_twitter_description',
                    array(
                        'id'           => 'wcv_seo_twitter_description',
                        'label'        => __( 'Description', 'wcvendors-pro' ),
                        'value'        => $value,
                        'rows'         => 3,
                        'description'  => __( 'Maximum of 200 characters.', 'wcvendors-pro' ),
                        'desc_tip'     => true,
                        'show_tooltip' => false,
                    )
                )
            );
        }
    }

    /**
     * SEO Twitter Image
     *
     * @since 1.5.0
     */
    public static function seo_twitter_image() {

        $seo_option = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_seo_twitter_image', 'no' )
            : get_option( 'wcvendors_hide_settings_seo_twitter_image', 'no' );

        if ( 'yes' !== $seo_option ) {

            $value = get_user_meta( get_current_user_id(), 'wcv_seo_twitter_image_id', true );
            echo '<div class="control-group">';
            WCVendors_Pro_Form_Helper::pro_file_uploader(
                apply_filters(
                    'wcv_vendor_seo_twitter_image',
                    array(
                        'id'             => 'wcv_seo_twitter_image_id',
                        'header_text'    => __( 'X (Twitter) Image', 'wcvendors-pro' ),
                        'add_text'       => __( 'Add X (Twitter) Image', 'wcvendors-pro' ),
                        'remove_text'    => __( 'Remove X (Twitter Image)', 'wcvendors-pro' ),
                        'image_meta_key' => 'wcv_seo_twitter_image_id',
                        'save_button'    => __( 'Add X (Twitter) Image', 'wcvendors-pro' ),
                        'window_title'   => __( 'Select an Image', 'wcvendors-pro' ),
                        'value'          => $value,
                        'size'           => 'medium',
                    )
                )
            );
            echo '</div>';

        }
    }

    /**
     * Show store opening hours form
     *
     * @return void
     * @since   1.5.8
     * @version 1.7.4
     */
    public static function store_opening_hours_form() {

        $hide = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_opening_hours', 'no' )
            : get_option( 'wcvendors_hide_settings_opening_hours', 'no' );

        if ( ! wc_string_to_bool( $hide ) ) {
            $required = ( 'signup' === self::$form_type ) ? get_option( 'wcvendors_required_signup_store_opening_hours', 'no' ) : get_option( 'wcvendors_required_settings_opening_hours', 'no' );
            $hours    = get_user_meta( get_current_user_id(), 'wcv_store_opening_hours', true );
            $hours    = apply_filters( 'wcv_store_opening_hours_' . get_current_user_id(), $hours );

            if ( empty( $hours ) ) {
                $hours = wcv_get_default_opening_hours();
            }

            $enable_opening_hours = get_user_meta( get_current_user_id(), '_wcv_enable_opening_hours', true );

            // Opening Hours.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_enable_opening_hours',
                    array(
                        'id'                => '_wcv_enable_opening_hours',
                        'label'             => __( 'Enable opening hours', 'wcvendors-pro' ),
                        'type'              => 'checkbox',
                        'class'             => 'wcv-enable-opening-hours',
                        'wrapper_start'     => '<div class="wcv-cols-group wcv-horizontal-gutters"><div class="all-100">',
                        'wrapper_end'       => '</div></div>',
                        'value'             => $enable_opening_hours,
                        'custom_attributes' => wc_string_to_bool( $required ) ? array(
                            'required'                   => '',
                            'data-parsley-error-message' => __( 'Opening hours is required' ),
                        ) : array(),
                    )
                )
            );

            include_once apply_filters( 'wcv_partial_path_opening_hours', 'partials/store-opening-hours.php' );
        }
    }

    /**
     *  Output store policy
     *
     * @since    1.6.1
     *
     * @param string $type  Policy type string.
     * @param string $label Field value.
     */
    public static function store_policy( $type, $label ) {

        $hide = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_hide_signup_store_policy_' . $type, 'no' )
            : get_option( 'wcvendors_hide_settings_store_policy_' . $type, 'no' );

        if ( wc_string_to_bool( $hide ) ) {
            return;
        }

        $required      = ( 'signup' === self::$form_type )
            ? get_option( 'wcvendors_required_signup_store_policy_' . $type, 'no' )
            : get_option( 'wcvendors_required_settings_store_policy_' . $type, 'no' );
        $required      = wc_string_to_bool( $required );
        $required_attr = $required ? array( 'required' => '' ) : array();

        $store_policies_html = ( 'signup' === self::$form_type )
            ? wc_string_to_bool( get_option( 'wcvendors_allow_signup_policy_html', 'no' ) )
            : wc_string_to_bool( get_option( 'wcvendors_allow_settings_policy_html', 'no' ) );
        $enable_media        = wc_string_to_bool( get_option( 'wcvendors_allow_editor_media', 'no' ) );
        $value               = get_user_meta( get_current_user_id(), 'wcv_policy_' . $type, true );

        $allow_markup = 'yes' === get_option( 'wcvendors_allow_form_markup', 'no' ) ? true : false;

        // If html in info is allowed then display the tinyMCE otherwise just display a text box.
        if ( $store_policies_html && $allow_markup ) {

            if ( $required ) {
                add_filter( 'the_editor', array( __CLASS__, 'wp_editor_required' ) );
                add_filter( 'tiny_mce_before_init', array( __CLASS__, 'wp_tinymce_required' ) );
                add_filter( 'teeny_mce_before_init', array( __CLASS__, 'wp_tinymce_required' ) );
            }

            $required_class = $required ? 'wcv-required' : '';

            $settings = apply_filters(
                'wcv_vendor_store_policy_editor_settings',
                array(
                    'editor_height' => 200,
                    'media_buttons' => $enable_media,
                    'teeny'         => true,
                    'editor_class'  => $required_class,
                    'tinymce'       => array(
                        'setup' => 'function (editor) {
                            editor.on("change", function () {
                                var content = tinyMCE.activeEditor.getContent( {format : "raw"} )
                                    .replace( \'<p><br data-mce-bogus="1"></p>\', "" );

                                if ( content != undefined && content != "" ) {
                                    jQuery( "#" + editor.id ).html( content );
                                }
                            });
                        }',
                    ),
                )
            );

            echo '<label>' . esc_html( $label ) . '</label>';

            wp_editor( $value, 'wcv_policy_' . $type, $settings );

        } else {

            WCV_Form_Helper::textarea(
                apply_filters(
                    'wcv_policy_' . $type,
                    array(
                        'id'                => 'wcv_policy_' . $type,
                        'label'             => $label,
                        'value'             => wp_strip_all_tags( $value ),
                        'custom_attributes' => $required_attr,
                    )
                )
            );
        }
    }

    /**
     * Output enable local pickup
     *
     * @return    void
     * @since      1.6.3
     */
    public static function enable_local_pickup() {

        if ( ! wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_enable_local_pickup', 'no' ) ) ) {

            $local_pickup_enabled = get_user_meta( get_current_user_id(), '_wcv_local_pickup_enabled', true );

            echo wp_kses_post( apply_filters( 'wcv_store_local_pickup_heading', '<hr /><h3>' . __( 'Local Delivery', 'wcvendors-pro' ) . '</h3>' ) );

            // Enable local delivery.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_vendor_enable_local_pickup',
                    array(
                        'id'            => '_wcv_local_pickup_enabled',
                        'label'         => __( 'Enable local pickup', 'wcvendors-pro' ),
                        'type'          => 'checkbox',
                        'wrapper_start' => '<div class="wcv-cols-group wcv-horizontal-gutters"><div class="all-100">',
                        'wrapper_end'   => '</div></div>',
                        'value'         => $local_pickup_enabled,
                    )
                )
            );
        }
    }

    /**
     * Add the Google Analytics Tracking ID field to the settings page for vendors
     */
    public static function store_ga_code() {

        if ( ! wc_string_to_bool( get_option( 'wcvendors_hide_settings_ga', 'no' ) ) ) {
            $value = get_user_meta( get_current_user_id(), '_wcv_settings_ga_tracking_id', true );

            $is_ga_enable = get_user_meta( get_current_user_id(), '_wcv_settings_enable_ga_code', true );

            WCV_Form_Helper::input(
                apply_filters(
                    'wc_vendor_enable_ga_code',
                    array(
                        'id'            => '_wcv_settings_enable_ga_code',
                        'type'          => 'checkbox',
                        'class'         => 'wcv-ga-enable',
                        'label'         => __( 'Enable Google Analytics', 'wcvendors-pro' ),
                        'wrapper_start' => '<div class="wcv-cols-group wcv-horizontal-gutters"><div class="all-100">',
                        'wrapper_end'   => '</div></div>',
                        'value'         => $is_ga_enable,
                    )
                )
            );

            // Output GA property field data.
            WCV_Form_Helper::input(
                apply_filters(
                    'wcv_vendor_ga_code',
                    array(
                        'id'            => '_wcv_settings_ga_tracking_id',
                        'class'         => 'wcv-ga-id-value',
                        'label'         => __( 'Google Analytics Tracking ID', 'wcvendors-pro' ),
                        'wrapper_start' => '<div class="all-100 wcv-ga-id-value-wrapper">',
                        'wrapper_end'   => '</div>',
                        'value'         => $value,
                    )
                )
            );
        }
    }

    /**
     * Extend store tabs from Free
     *
     * @since 1.9.2
     * @since 1.9.7 Hide shipping tab if vendor shipping is disabled.
     * @version 1.9.7
     *
     * @param array $tabs The tabs array.
     */
    public function store_tabs( $tabs ) {
        $pro_tabs = array(
            'branding' => array(
                'label'  => __( 'Branding', 'wcvendors-pro' ),
                'target' => 'branding',
                'class'  => array(),
            ),
            'shipping' => array(
                'label'  => __( 'Shipping', 'wcvendors-pro' ),
                'target' => 'shipping',
                'class'  => array(),
            ),
            'social'   => array(
                'label'  => __( 'Social', 'wcvendors-pro' ),
                'target' => 'social',
                'class'  => array(),
            ),
            'policies' => array(
                'label'  => __( 'Policies', 'wcvendors-pro' ),
                'target' => 'policies',
                'class'  => array(),
            ),
            'seo'      => array(
                'label'  => __( 'SEO', 'wcvendors-pro' ),
                'target' => 'seo',
                'class'  => array(),
            ),
        );

        $shipping_disabled = wc_string_to_bool( get_option( 'wcvendors_shipping_management_cap', 'no' ) );

        if ( $shipping_disabled ) {
            unset( $pro_tabs['shipping'] );
        }

        $vendor_shipping = get_option( 'woocommerce_wcv_pro_vendor_shipping_settings', wcv_get_default_vendor_shipping() );

        if ( isset( $vendor_shipping['enabled'] ) && 'no' === $vendor_shipping['enabled'] ) {
            unset( $pro_tabs['shipping'] );
        }

        $tabs = array_merge( $tabs, $pro_tabs );

        /**
         * Filter the pro dashboard store settings tabs
         *
         * @var array $tabs The store tabs.
         */
        $tabs = apply_filters(
            'wcvendors_store_tabs',
            $tabs
        );

        return $tabs;
    }

    /**
     * Extend store settings field
     *
     * @since 1.9.2
     * @version 1.9.2
     */
    public function pro_fields_after_payment() {
        $shipping_details = get_user_meta( get_current_user_id(), '_wcv_shipping', true );
        wc_get_template(
            'store-settings.php',
            array(
                'shipping_details' => $shipping_details,
            ),
            'wc-vendors/dashboard/',
            WCV_PRO_ABSPATH_TEMPLATES . 'dashboard/'
        );
    }


    /**
     * Pro fields before address
     *
     * @since 1.9.2
     * @version 1.9.2
     */
    public function pro_fields_before_address() {
        self::show_hide_map();
        self::location_picker();
        self::coordinates();
    }

    /**
     * Pro fields after address
     *
     * @since 1.9.2
     * @version 1.9.2
     */
    public function pro_fields_after_address() {
        // Store Opening Hours.
        self::store_opening_hours_form();

        // Store Vacation Mode.
        do_action( 'wcvendors_settings_before_vacation_mode' );
        self::vacation_mode();
        do_action( 'wcvendors_settings_after_vacation_mode' );

        // Vendor Store Notice.
        do_action( 'wcvendors_settings_before_vendor_store_notice' );
        self::enable_store_notice();
        self::vendor_store_notice();
        do_action( 'wcvendors_settings_after_vendor_store_notice' );

        // Store GA Code.
        do_action( 'wcvendors_settings_before_ga_code' );
        self::store_ga_code();
        do_action( 'wcvendors_settings_after_ga_code' );
    }

    /**
     * Hide tabs
     *
     * @since 1.9.2
     * @version 1.9.2
     *
     * @param array $hide_tabs_settings The tabs array.
     */
    public function hide_tabs_settings( $hide_tabs_settings ) {
        if ( 'yes' === get_option( 'wcvendors_hide_settings_tab_payment', 'no' ) ) {
            $hide_tabs_settings[] = 'payment';
        }
        if ( 'yes' === get_option( 'wcvendors_hide_settings_tab_branding', 'no' ) ) {
            $hide_tabs_settings[] = 'branding';
        }
        if ( 'yes' === get_option( 'wcvendors_hide_settings_tab_shipping', 'no' ) ) {
            $hide_tabs_settings[] = 'shipping';
        }
        if ( 'yes' === get_option( 'wcvendors_hide_settings_tab_social', 'no' ) ) {
            $hide_tabs_settings[] = 'social';
        }
        if (
            ! wc_string_to_bool( get_option( 'wcvendors_allow_policy_override', 'yes' ) ) ||
            wc_string_to_bool( get_option( 'wcvendors_hide_settings_tab_policies', 'no' ) )
            || (
                wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_policy_privacy', 'no' ) )
                && wc_string_to_bool( get_option( 'wcvendors_hide_settings_store_policy_terms', 'no' ) )
                && wc_string_to_bool( get_option( 'wcvendors_hide_settings_shipping_shipping_policy', 'no' ) )
                && wc_string_to_bool( get_option( 'wcvendors_hide_settings_shipping_return_policy', 'no' ) )
            )
        ) {
            $hide_tabs_settings[] = 'policies';
        }

        if ( 'yes' === get_option( 'wcvendors_hide_settings_tab_seo', 'no' ) ) {
            $hide_tabs_settings[] = 'seo';
        }

        return $hide_tabs_settings;
    }

    /**
     * Hide tabs signup
     *
     * @since 1.9.2
     * @version 1.9.2
     *
     * @param array $hide_tabs_signup The tabs array.
     */
    public function hide_tabs_signup( $hide_tabs_signup ) {
        if ( 'yes' === get_option( 'wcvendors_hide_signup_tab_payment', 'no' ) ) {
            $hide_tabs_signup[] = 'payment';
        }
        if ( 'yes' === get_option( 'wcvendors_hide_signup_tab_branding', 'no' ) ) {
            $hide_tabs_signup[] = 'branding';
        }
        if ( 'yes' === get_option( 'wcvendors_hide_signup_tab_shipping', 'no' ) ) {
            $hide_tabs_signup[] = 'shipping';
        }
        if ( 'yes' === get_option( 'wcvendors_hide_signup_tab_social', 'no' ) ) {
            $hide_tabs_signup[] = 'social';
        }
        if ( 'yes' === get_option( 'wcvendors_hide_signup_tab_seo', 'no' ) ) {
            $hide_tabs_signup[] = 'seo';
        }
        if (
            ! wc_string_to_bool( get_option( 'wcvendors_allow_policy_override', 'yes' ) ) ||
            wc_string_to_bool( get_option( 'wcvendors_hide_signup_tab_policies', 'no' ) )
            || (
                wc_string_to_bool( get_option( 'wcvendors_hide_signup_store_policy_privacy', 'no' ) )
                && wc_string_to_bool( get_option( 'wcvendors_hide_signup_store_policy_terms', 'no' ) )
                && wc_string_to_bool( get_option( 'wcvendors_hide_signup_shipping_shipping_policy', 'no' ) )
                && wc_string_to_bool( get_option( 'wcvendors_hide_signup_shipping_return_policy', 'no' ) )
            )
        ) {
            $hide_tabs_signup[] = 'policies';
        }

        return $hide_tabs_signup;
    }
}
