<?php
/**
 * Output the report product button and popup
 *
 * @package  WCVendors_Pro
 * @author   WC Vendors
 * @since    1.8.3
 * @version  1.8.3
 */

/**
 * Class WCVendors_Pro_Product_Reports_Form
 */
class WCVendors_Pro_Product_Reports_Form {

    /**
     * Add report button and popup under product meta.
     */
    public function product_reports_form_content() {
        global $post;

        if ( 'yes' !== get_option( 'wcvendors_pro_product_reports_enabled', 'no' ) ) {
            return;
        }

        $is_logged_in        = is_user_logged_in();
        $show_loggedin       = get_option( 'wcvendors_pro_product_reports_visibility_loggedin', 'yes' );
        $show_loggedout      = get_option( 'wcvendors_pro_product_reports_visibility_loggedout', 'no' );
        $is_admin            = current_user_can( 'manage_woocommerce' );
        $display_to_loggedin = $is_logged_in && 'yes' === $show_loggedin;
        $display_to_guest    = ! $is_logged_in && 'yes' === $show_loggedout;

        if ( ! $is_admin && ! $display_to_loggedin && ! $display_to_guest ) {
            return;
        }

        $post_author = (int) get_post_field( 'post_author', $post->ID );
        $vendor_id   = (int) WCV_Vendors::get_vendor_from_product( $post->ID );
        $reporter_id = (int) get_current_user_id();

        if ( ! $is_admin && ( $post_author === $reporter_id || $vendor_id === $reporter_id ) ) {
            return;
        }

        if ( $is_logged_in && ! $is_admin && ! self::maybe_report( (int) $post->ID ) ) {
            return;
        }

        $default_btn_label           = __( 'Report this product', 'wcvendors-pro' );
        $default_popup_title         = __( 'Report an issue with this product or seller', 'wcvendors-pro' );
        $default_select_reason_label = __( 'Select a reason', 'wcvendors-pro' );
        $default_description_label   = __( 'Tell us about the issue', 'wcvendors-pro' );

        $button_label            = get_option( 'wcvendors_pro_product_reports_button_label', $default_btn_label );
        $popup_title             = get_option( 'wcvendors_pro_product_reports_popup_title', $default_popup_title );
        $select_reason_label     = get_option( 'wcvendors_pro_product_reports_select_reason_label', $default_select_reason_label );
        $description_input_label = get_option( 'wcvendors_pro_product_reports_description_input_label', $default_description_label );
        $display_style           = get_option( 'wcvendors_pro_product_reports_display_style', 'link' );

        /**
         * Backward compatibility filters
         */
        $button_label = apply_filters( 'wcvendors_product_reports_button_label', $button_label );
        $popup_title  = apply_filters( 'wcvendors_product_reports_popup_title', $popup_title ) . ': ' . get_the_title( $post->ID );

        wc_get_template(
            'product-reports.php',
            array(
                'button_label'                  => $button_label,
                'popup_title'                   => $popup_title,
                'select_reason_label'           => $select_reason_label,
                'select_reason_placeholder'     => apply_filters( 'wcvendors_product_reports_select_reason_placeholder', __( 'Select a reason', 'wcvendors-pro' ) ),
                'description_input_label'       => $description_input_label,
                'description_input_placeholder' => apply_filters( 'wcvendors_product_reports_description_input_placeholder', __( 'Tell us about the issue', 'wcvendors-pro' ) ),
                'display_style'                 => $display_style,
                'is_logged_in'                  => $is_logged_in,
            ),
            'wc-vendors/product/',
            WCV_PRO_ABSPATH_TEMPLATES . '/product/'
        );
    }

    /**
     * Output the report product reason field
     *
     * @param string $label The label text for the reason field.
     * @param string $placeholder The placeholder text for the reason field.
     *
     * @since 2.0.2
     */
    public static function report_reason_field( $label, $placeholder ) {
        $default_reason = wcv_default_product_report_reason();
        $user_reason    = get_option( 'wcvendors_pro_product_reports_reasons', $default_reason );
        $report_reason  = apply_filters(
            'wcvendors_product_reports_reason',
            $user_reason
        );
        $report_reason  = array_combine( $report_reason, $report_reason );
        WCVendors_Pro_Form_Helper::select(
            apply_filters(
                'wcvendors_product_reports_reason_field',
                array(
                    'id'                => '_wcv_product_reports_reason',
                    'label'             => $label,
                    'options'           => $report_reason,
                    'value'             => '',
                    'wrapper_tag'       => 'div',
                    'wrapper_class'     => 'form-row form-row-wide',
                    'field_class'       => 'select2',
                    'show_option_none'  => $placeholder,
                    'custom_attributes' => array(
                        'data-placeholder' => $placeholder,
                        'data-allow-clear' => 'true',
                    ),
                )
            )
        );
    }

    /**
     * Output the report product description input label
     *
     * @param string $label The label text for the description input field.
     * @param string $placeholder The placeholder text for the description input field.
     *
     * @since 2.0.2
     */
    public static function report_description_input_field( $label, $placeholder ) {
        WCVendors_Pro_Form_Helper::textarea(
            apply_filters(
                'wcvendors_product_reports_description_input_label',
                array(
                    'id'            => '_wcv_product_reports_description',
                    'label'         => $label,
                    'placeholder'   => $placeholder,
                    'value'         => '',
                    'wrapper_tag'   => 'div',
                    'wrapper_class' => 'control-group',
                    'field_class'   => 'control',
                )
            )
        );
    }

    /**
     * Output the report product nonce field
     */
    public static function product_reports_nonce_field() {
        WCVendors_Pro_Form_Helper::input(
            apply_filters(
                'wcvendors_product_reports_nonce_field',
                array(
                    'id'    => '_wcv_product_reports_nonce',
                    'value' => wp_create_nonce( 'wcv-product-reports' ),
                    'type'  => 'hidden',
                )
            )
        );
    }

    /**
     * Output the report product submit button
     */
    public static function product_reports_submit_button() {
        WCVendors_Pro_Form_Helper::submit(
            apply_filters(
                'wcvendors_product_reports_submit_button',
                array(
                    'id'            => '_wcv_product_reports_submit',
                    'label'         => __( 'Submit report', 'wcvendors-pro' ),
                    'type'          => 'submit',
                    'value'         => __( 'Submit report', 'wcvendors-pro' ),
                    'wrapper_tag'   => 'div',
                    'wrapper_class' => 'form-row form-row-wide',
                    'field_class'   => '',
                )
            )
        );
    }

    /**
     * Check if the product is reported by the current user
     *
     * @param  int $product_id Product id.
     */
    public static function maybe_report( $product_id ) {
        global $wpdb;
        $result = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}wcv_product_report WHERE product_id = %d AND reporter_id = %d", $product_id, get_current_user_id() ) ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

        if ( $result ) {
            return false;
        }

        return true;
    }
}
