<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://www.webtoffee.com/
 * @since      3.0.0
 *
 * @package    WebToffee
 * @subpackage CookieConsent/Frontend
 */

namespace WebToffee\CookieConsent\Lite\Frontend;

use WebToffee\CookieConsent\Lite\Admin\Modules\Banners\Includes\Controller;
use WebToffee\CookieConsent\Lite\Admin\Modules\Settings\Includes\Settings;

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    WebToffee
 * @subpackage WebToffee\CookieConsent\Lite\Frontend
 * @author     WebToffee <info@webtoffee.com>
 */
class Frontend {


	/**
	 * The ID of this plugin.
	 *
	 * @since    3.0.0
	 * @access   private
	 * @var      string    $plugin_name  The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    3.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Admin modules of the plugin
	 *
	 * @var array
	 */
	private static $modules;

	/**
	 * Currently active modules
	 *
	 * @var array
	 */
	private static $active_modules;

	/**
	 * Existing modules
	 *
	 * @var array
	 */
	public static $existing_modules;

	/**
	 * Banner object
	 *
	 * @var object
	 */
	protected $banner;
	/**
	 * Banners array
	 *
	 * array
	 */
	protected $banners = array();

	/**
	 * Plugin settings
	 *
	 * @var object
	 */
	protected $settings;
	/**
	 * Banner template
	 *
	 * @var object
	 */
	protected $template;

	/**
	 * Banner templates array
	 *
	 * array
	 */
	protected $templates = array();

	/**
	 * Providers list
	 *
	 * @var array
	 */
	protected $providers = array();
	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    3.0.0
	 * @param      string $plugin_name       The name of the plugin.
	 * @param      string $version    The version of this plugin.
	 */
	/**
	 * Status google consent mode
	 *
	 * @since    3.1.0
	 * @access   protected
	 * @var      boolean    $google_consent_mode    The gcm value.
	 */
	protected $google_consent_mode = false;
	/**
	 * Value of gtag_behavior
	 *
	 * @since    3.1.0
	 * @access   protected
	 * @var      boolean    $gtag_behavior    The gtag_behavior value.
	 */
	protected $gtag_behavior = false;
	/**
	 * Value of url_passthrough
	 *
	 * @since    3.1.0
	 * @access   protected
	 * @var      boolean    $url_passthrough    The url_passthrough value.
	 */
	protected $url_passthrough = false;
	/**
	 * Value of gcm_debug_mode
	 *
	 * @since    3.1.0
	 * @access   protected
	 * @var      boolean    $gcm_debug_mode    The gcm_debug_mode value.
	 */
	protected $gcm_debug_mode = false;
	/**
	 * Value of redact_ads_data
	 *
	 * @since    3.1.0
	 * @access   protected
	 * @var      boolean    $redact_ads_data    The redact_ads_data value.
	 */
	protected $redact_ads_data = false;
	/**
	 * Microsoft Clarity status
	 *
	 * @since    3.3.1
	 * @access   protected
	 * @var      boolean    $microsoft_clarity    The microsoft_clarity value.
	 */
	protected $microsoft_clarity = false;
	/**
	 * Microsoft UET status
	 *
	 * @since    3.4.0
	 * @access   protected
	 * @var      boolean    $microsoft_uet    The microsoft_uet value.
	 */
	protected $microsoft_uet = false;
	/**
	 * Value of gtm_template
	 *
	 * @since    3.4.0
	 * @access   protected
	 * @var      boolean    $gtm_template    The gtm_template value.
	 */
	protected $gtm_template = false;

	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version     = $version;
		$this->load_modules();
		$this->settings = new Settings();
		$options        = get_option( 'wcc_settings' );
		add_action( 'init', array( $this, 'load_banner' ) );
		add_action( 'wp_footer', array( $this, 'banner_html' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ), 1 );
		add_action( 'wp_head', array( $this, 'insert_styles' ) );
		$this->google_consent_mode = isset( $options['google_consent_mode']['status'] ) ? $options['google_consent_mode']['status'] : true;
		$this->gtag_behavior       = isset( $options['gtag_behavior']['selected'] ) ? $options['gtag_behavior']['selected'] : false;
		$this->url_passthrough     = isset( $options['url_passthrough']['status'] ) ? $options['url_passthrough']['status'] : false;
		$this->gcm_debug_mode      = isset( $options['gcm_debug_mode']['status'] ) ? $options['gcm_debug_mode']['status'] : false;
		$this->redact_ads_data     = isset( $options['redact_ads_data']['status'] ) ? $options['redact_ads_data']['status'] : false;
		$this->microsoft_clarity   = isset( $options['microsoft_clarity']['status'] ) ? $options['microsoft_clarity']['status'] : false;
		$this->microsoft_uet       = isset( $options['microsoft_uet']['status'] ) ? $options['microsoft_uet']['status'] : false;
		$this->gtm_template        = isset( $options['gtm_template']['status'] ) ? $options['gtm_template']['status'] : false;
	}

	/**
	 * Get the default modules array
	 *
	 * @since 3.0.0
	 * @return array
	 */
	public function get_default_modules() {
		$modules = array( 'iab' );
		return $modules;
	}

	/**
	 * Load all the modules
	 *
	 * @return void
	 */
	public function load_modules() {
		if ( true === wcc_disable_banner() ) {
			return;
		}
		foreach ( $this->get_default_modules() as $module ) {
			$parts      = explode( '_', $module );
			$class      = implode( '_', $parts );
			$class      = str_ireplace( '-', '_', $class );
			$module     = str_ireplace( '-', '_', $module );
			$class_name = 'WebToffee\CookieConsent\Lite\\Frontend\\Modules\\' . ucwords( $module, '_' ) . '\\' . ucwords( $class, '_' );
			if ( class_exists( $class_name ) ) {
				$module_obj = new $class_name( $module );
				if ( $module_obj instanceof $class_name ) {
					if ( $module_obj->is_active() ) {
						$module_obj->init();
						self::$active_modules[ $module ] = true;
					}
				}
			}
		}
	}

	/**
	 * Enqeue front end scripts
	 *
	 * @return void
	 */
	public function enqueue_scripts() {
		if ( true === wcc_disable_banner() ) {
			return;
		}
		if ( ! $this->templates ) {
			return;
		}
		$css    = array();
		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
		if ( is_array( $this->templates ) ) {
			foreach ( $this->templates as $key => $template ) {
				$css[ $key ] = isset( $template['styles'] ) ? $template['styles'] : '';
			}
		}
		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/script' . $suffix . '.js', array(), $this->version, false );
		wp_localize_script( $this->plugin_name, '_wccConfig', $this->get_store_data() );
		wp_localize_script( $this->plugin_name, '_wccStyles', array( 'css' => $css ) );
		wp_localize_script(
			$this->plugin_name,
			'_wccApi',
			array(
				'base'  => esc_url_raw( rest_url( 'wcc/v1/' ) ),
				'nonce' => wp_create_nonce( 'wp_rest' ),
			)
		);

		if ( true === $this->google_consent_mode ) {
			wp_localize_script( $this->plugin_name, '_wccGCMConfig', $this->get_gcm_data() );
			wp_enqueue_script( $this->plugin_name . '-gcm', plugin_dir_url( __FILE__ ) . 'js/gcm' . $suffix . '.js', array(), $this->version, false );
		}
		/*
		 * Enqueue the WP Consent API script if the WP Consent API plugin is enabled.
		 */
		if ( true === $this->is_wpconsentapi_enabled() ) {
			wp_localize_script( $this->plugin_name, '_wccWcaConfig', $this->get_wca_data() );
			wp_enqueue_script( $this->plugin_name . '-wca', plugin_dir_url( __FILE__ ) . 'js/wca' . $suffix . '.js', array(), $this->version, false );
		}
		/*
		 * Enqueue the Microsoft consent mode script if the Microsoft Clarity Or UET is enabled.
		 * @since 3.3.1
		 */
		if ( true === $this->microsoft_clarity || true === $this->microsoft_uet ) {
			wp_localize_script( $this->plugin_name, '_wccMCMConfig', $this->get_mcm_data() );
			wp_enqueue_script( $this->plugin_name . 'ms-clarity-consent', plugin_dir_url( __FILE__ ) . 'js/mcm' . $suffix . '.js', array(), $this->version, false );
		}
	}

	/**
	 * Add inline styles to the head
	 *
	 * @return void
	 */
	public function insert_styles() {
		if ( true === wcc_disable_banner() || is_admin() ) {
			return;
		}
		echo '<style id="wcc-style-inline">[data-tag]{visibility:hidden;}</style>';
	}

	/**
	 * Load active banner.
	 *
	 * @return void
	 */
	public function load_banner() {
		if ( true === wcc_disable_banner() || ! wcc_is_front_end_request() ) {
			return;
		}
		$controller    = Controller::get_instance();
		$this->banners = $controller->get_active_banners();
		if ( empty( $this->banners ) ) {
			return;
		}
		foreach ( $this->banners as $key => $banner ) {
			$this->templates[ $key ] = $banner->get_template();
			$properties              = $banner->get_settings();
			$settings                = isset( $properties['settings'] ) ? $properties['settings'] : array();
			if ( count( $this->banners ) > 1 ) {
				$active_law = apply_filters( 'wcc_alter_active_law', 'gdpr' );
				if ( $banner && ( $settings['applicableLaw'] ?? '' ) === $active_law ) {
					$version_id = $settings['versionID'] ?? 'default';
					$shortcodes = new \WebToffee\CookieConsent\Lite\Frontend\Modules\Shortcodes\Shortcodes( $banner, $version_id );
				}
			} else {
				$version_id = $settings['versionID'] ?? 'default';
				$shortcodes = new \WebToffee\CookieConsent\Lite\Frontend\Modules\Shortcodes\Shortcodes( $banner, $version_id );
			}
		}
	}

	/**
	 * Print banner HTML as script template using
	 * type="text/template" attribute
	 *
	 * @return void
	 */
	public function banner_html() {
		if ( ! $this->templates || true === wcc_disable_banner() ) {
			return;
		}
		if ( is_array( $this->templates ) ) {
			foreach ( $this->templates as $key => $template ) {
				$html = isset( $template['html'] ) ? $template['html'] : '';
				// Get the site URL to determine the protocol (http or https)
				$site_url      = get_site_url();
				$site_protocol = wp_parse_url( $site_url, PHP_URL_SCHEME ); // 'http' or 'https'
				// Replace the protocol in src attributes with the site's protocol
				$html = preg_replace(
					'/src=["\']http:\/\/(.*?)(\/wp-content\/)/',
					'src="' . $site_protocol . '://$1$2',
					$html
				);
				echo '<script id="wccBannerTemplate_' . esc_attr( $key ) . '" type="text/template">';
				echo wp_kses( $html, wcc_allowed_html() );
				echo '</script>';
			}
		}
	}

	/**
	 * Get store data
	 *
	 * @return array
	 */
	public function get_store_data() {
		$store     = array();
		$settings  = get_option( 'wcc_settings' );
		$providers = array();
		$store     = array(
			'_ipData'             => array(),
			'_assetsURL'          => WCC_PLUGIN_URL . 'frontend/images/',
			'_publicURL'          => get_site_url(),
			'_categories'         => $this->get_cookie_groups(),
			'_activeLaw'          => 'gdpr',
			'_rootDomain'         => $this->get_cookie_domain(),
			'_block'              => true,
			'_showBanner'         => true,
			'_bannerConfig'       => $this->prepare_config(),
			'_version'            => $this->version,
			'_logConsent'         => isset( $settings['consent_logs']['status'] ) && true === $settings['consent_logs']['status'] ? true : false,
			'_tags'               => $this->prepare_tags(),
			'_rtl'                => $this->is_rtl(),
			'_lawSelected'        => isset( $settings['banners']['selected'] ) ? $settings['banners']['selected'] : array(),
			'_restApiUrl'         => esc_url( 'https://directory.cookieyes.com/api/v1/ip' ),
			'_renewConsent'       => isset( $settings['renew_date']['date'] ) ? $settings['renew_date']['date'] : '',
			'_restrictToCA'       => apply_filters( 'wcc_restrict_ccpa_banner_to_ca', false ),
			'_customEvents'       => apply_filters( 'wcc_custom_events', false ),
			'_ccpaAllowedRegions' => apply_filters( 'wcc_ccpa_allowed_regions', array() ),
			'_gdprAllowedRegions' => apply_filters( 'wcc_gdpr_allowed_regions', array() ),
			'_closeButtonAction'  => apply_filters( 'wcc_close_button_action', 'reject' ),
			'_ssl'                => apply_filters( 'wcc_ssl_enabled', is_ssl() ),
		);
		/*
		 * Filter the store data and set _lawSelected in ['GDPR', 'CCPA'] order if two where selected.
		 */
		if ( ! empty( $store['_lawSelected'] ) ) {
			$store['_lawSelected'] = array_values( array_intersect( array( 'GDPR', 'CCPA' ), $store['_lawSelected'] ) );
		}

		$providers = array();
		foreach ( $this->providers as $key => $value ) {
			$providers[] = array(
				're'         => $key,
				'categories' => $value,
			);
		}
		//Unblock google scripts on when gtag_behavior is advanced
		if ( $this->google_consent_mode && 'advanced' === $this->gtag_behavior ) {
			$providers = $this->exclude_google_tag_scripts( $providers );
		}
		//Unblock microsoft scripts on when microsoft_clarity is enabled
		if ( $this->microsoft_clarity ) {
			$providers = $this->exclude_microsoft_scripts( $providers );
		}
		$store['_providersToBlock'] = $providers;

		return $store;
	}

	/**
	 * Get gcm data
	 *
	 * @since 3.1.0
	 * @return array
	 */
	public function get_gcm_data() {
		$settings        = array();
		$region_settings = get_option( 'wcc_gcm_default_consents', array() );
		$settings        = array(
			'_mode'           => $this->gtag_behavior,
			'_urlPassthrough' => $this->url_passthrough,
			'_debugMode'      => $this->gcm_debug_mode,
			'_redactData'     => $this->redact_ads_data,
			'_regions'        => $region_settings,
			'_wccBypass'      => isset( $_GET['wcc_bypass'] ) ? true : false, // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			'wait_for_update' => apply_filters( 'wcc_gcm_update_waiting_time', 500 ),
			'_isGTMTemplate'  => $this->gtm_template,
		);
		return $settings;
	}
	/**
	 * Get WP Consent API category list
	 *
	 * @since 3.2.0
	 * @return array
	 */
	public function get_wca_data() {
		return apply_filters(
			'wcc_wca_category_map',
			array(
				'functional'    => array( 'functional', 'preferences' ),
				'analytics'     => array( 'statistics', 'statistics-anonymous' ),
				'advertisement' => 'marketing',
			)
		);
	}
	/**
	 * Return cookie domain
	 *
	 * @return string
	 */
	public function get_cookie_domain() {
		return apply_filters( 'wcc_cookie_domain', '' );
	}
	/**
	 * Get cookie groups
	 *
	 * @return array
	 */
	public function get_cookie_groups() {
		$cookie_groups = array();
		$categories    = \WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Includes\Category_Controller::get_instance()->get_items();

		foreach ( $categories as $category ) {
			$category        = new \WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Includes\Cookie_Categories( $category );
			$cookie_groups[] = array(
				'name'                => $category->get_name( wcc_current_language() ),
				'slug'                => $category->get_slug(),
				'isNecessary'         => 'necessary' === $category->get_slug() ? true : false,
				'ccpaDoNotSell'       => $category->get_sell_personal_data(),
				'cookies'             => $this->get_cookies( $category ),
				'active'              => true,
				'defaultConsent'      => array(
					'gdpr' => $category->get_prior_consent(),
					'ccpa' => 'necessary' === $category->get_slug() || $category->get_sell_personal_data() === false ? true : false,
				),
				'foundNoCookieScript' => $this->check_no_cookie_script_exist( $category->get_slug() ),
			);
		}
		return $cookie_groups;
	}
	/**
	 * Get cookies by category
	 *
	 * @param object $category Category object.
	 * @return array
	 */
	public function get_cookies( $category = '' ) {
		$cookies  = array();
		$cat_slug = $category->get_slug();
		$items    = \WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Includes\Cookie_Controller::get_instance()->get_items_by_category( $category->get_id() );
		foreach ( $items as $item ) {
			// Skip items where cookie_status is 0.
			if ( empty( $item->cookie_status ) ) {
				continue;
			}

			$cookie    = new \WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Includes\Cookie( $item->cookie_id );
			$cookies[] = array(
				'cookieID' => $cookie->get_name(),
				'domain'   => $cookie->get_domain(),
				'provider' => $cookie->get_url_pattern(),
			);
			$provider  = $cookie->get_url_pattern();
			$this->build_provider_list( $provider, $cat_slug );
		}
		return $cookies;
	}
	/**
	 * Add to provider list
	 * @since 3.2.0
	 * @param string $provider Provider name.
	 * @param string $cat_slug Category slug.
	 * @return void
	 */
	public function build_provider_list( $provider, $cat_slug ) {
		if ( '' !== $provider && 'necessary' !== $cat_slug ) {
			if ( ! isset( $this->providers[ $provider ] ) ) {
				$this->providers[ $provider ] = array();
			}
			if ( isset( $this->providers[ $provider ] ) && ! in_array( $cat_slug, $this->providers[ $provider ], true ) ) {
				$this->providers[ $provider ][] = $cat_slug;
			}
		}
	}
	/**
	 * Prepare the HTML elements tags for front-end script.
	 *
	 * @return array
	 */
	public function prepare_tags() {
		$data            = array();
		$banner_selected = $this->banners;
		if ( ! $banner_selected ) {
			return;
		}
		if ( is_array( $banner_selected ) ) {
			foreach ( $banner_selected as $banner_id => $banner ) {
				$settings  = $banner->get_settings();
				$configs   = isset( $settings['config'] ) ? $settings['config'] : array();
				$supported = array(
					'accept-button',
					'reject-button',
					'settings-button',
					'readmore-button',
					'donotsell-button',
					'accept-button',
					'revisit-consent',
					'detail-category-always-enabled',
				);
				foreach ( $supported as $tag ) {
					$config = wcc_array_search( $configs, 'tag', $tag );
					$data[] = array(
						'tag'    => $tag,
						'styles' => isset( $config['styles'] ) ? $config['styles'] : array(),
					);
				}
			}
		}
		return $data;
	}

	/**
	 * Prepare config for the front-end processing
	 *
	 * @return array
	 */
	public function prepare_config() {
		$data            = array();
		$data_arr        = array();
		$banner_selected = $this->banners;
		if ( ! $banner_selected ) {
			return $data_arr;
		}
		if ( is_array( $banner_selected ) ) {
			foreach ( $banner_selected as $banner_id => $banner ) {
				$properties                                   = $banner->get_settings();
				$data['settings']['type']                     = ( 'push-down' !== $properties['settings']['preferenceCenter'] ) ? $properties['settings']['type'] : 'classic';
				$data['settings']['position']                 = $properties['settings']['position'];
				$data['settings']['applicableLaw']            = $properties['settings']['applicableLaw'];
				$data['settings']['preferenceCenter']         = $properties['settings']['preferenceCenter'];
				$data['behaviours']['reloadBannerOnAccept']   = $properties['behaviours']['reloadBannerOnAccept']['status'];
				$data['behaviours']['loadAnalyticsByDefault'] = $properties['behaviours']['loadAnalyticsByDefault']['status'];
				$data['behaviours']['animations']             = $properties['behaviours']['animations'];
				$data['config']['revisitConsent']             = $properties['config']['revisitConsent'];
				$data['config']['preferenceCenter']['toggle'] = $properties['config']['preferenceCenter']['elements']['categories']['elements']['toggle'];
				// Use different poweredBy config based on banner_id.
				if ( $banner_id === 'CCPA' ) {
					$data['config']['preferenceCenter']['poweredBy'] = isset( $properties['config']['optoutPopup']['elements']['poweredBy'] ) ? $properties['config']['optoutPopup']['elements']['poweredBy'] : false;
				} else {
					$data['config']['preferenceCenter']['poweredBy'] = isset( $properties['config']['preferenceCenter']['elements']['poweredBy'] ) ? $properties['config']['preferenceCenter']['elements']['poweredBy'] : false;
				}
				$data['config']['categoryPreview']['status']  = $properties['config']['categoryPreview']['status'];
				$data['config']['categoryPreview']['toggle']  = $properties['config']['categoryPreview']['elements']['toggle'];
				$data['config']['videoPlaceholder']['status'] = $properties['config']['videoPlaceholder']['status'];
				$data['config']['videoPlaceholder']['styles'] = array_merge( $properties['config']['videoPlaceholder']['styles'], $properties['config']['videoPlaceholder']['elements']['title']['styles'] );
				$data['config']['readMore']                   = $properties['config']['notice']['elements']['buttons']['elements']['readMore'];
				$data['config']['auditTable']['status']       = $properties['config']['auditTable']['status'];
				$data['config']['optOption']['status']        = $properties['config']['optoutPopup']['elements']['optOption']['status'];
				$data['config']['optOption']['toggle']        = $properties['config']['optoutPopup']['elements']['optOption']['elements']['toggle'];
				$data['config']['optOption']['gpcOption']     = $properties['config']['optoutPopup']['elements']['gpcOption']['status'];
				$data['settings']['selectedRegion']           = $properties['config']['selectedRegion']['region'];
				$data['settings']['consentExpiry']            = isset( $properties['settings']['consentExpiry']['value'] ) ? absint( $properties['settings']['consentExpiry']['value'] ) : 365;
				$data['settings']['shortcodes']               = $this->prepare_shortcodes( $banner, $properties );
				$data['settings']['bannerEnabled']            = $properties['config']['bannerEnabled']['status'];
				$data_arr[ $banner_id ]                       = $data;
			}
		}
		return $data_arr;
	}

	/**
	 * Prepare shortcodes to be used on visitor side.
	 *
	 * @param array $properties Banner properties.
	 * @return array
	 */
	public function prepare_shortcodes( $banner, $properties = array() ) {
		$settings   = isset( $properties['settings'] ) ? $properties['settings'] : array();
		$version_id = isset( $settings['versionID'] ) ? $settings['versionID'] : 'default';
		$shortcodes = new \WebToffee\CookieConsent\Lite\Frontend\Modules\Shortcodes\Shortcodes( $banner, $version_id );
		$data       = array();
		$configs    = ( isset( $properties['config'] ) && is_array( $properties['config'] ) ) ? $properties['config'] : array();
		$config     = wcc_array_search( $configs, 'tag', 'readmore-button' );
		$attributes = array();
		if ( isset( $config['meta']['noFollow'] ) && true === $config['meta']['noFollow'] ) {
			$attributes['rel'] = 'nofollow';
		}
		if ( isset( $config['meta']['newTab'] ) && true === $config['meta']['newTab'] ) {
			$attributes['target'] = '_blank';
		}
		$data[] = array(
			'key'        => 'wcc_readmore',
			'content'    => do_shortcode( '[wcc_readmore]' ),
			'tag'        => 'readmore-button',
			'status'     => isset( $config['status'] ) && true === $config['status'] ? true : false,
			'attributes' => $attributes,
		);
		$data[] = array(
			'key'        => 'wcc_show_desc',
			'content'    => do_shortcode( '[wcc_show_desc]' ),
			'tag'        => 'show-desc-button',
			'status'     => true,
			'attributes' => array(),
		);
		$data[] = array(
			'key'        => 'wcc_hide_desc',
			'content'    => do_shortcode( '[wcc_hide_desc]' ),
			'tag'        => 'hide-desc-button',
			'status'     => true,
			'attributes' => array(),
		);
		$data[] = array(
			'key'        => 'wcc_category_toggle_label',
			'content'    => do_shortcode( '[wcc_category_toggle_label]' ),
			'tag'        => '',
			'status'     => true,
			'attributes' => array(),
		);
		$data[] = array(
			'key'        => 'wcc_enable_category_label',
			'content'    => do_shortcode( '[wcc_enable_category_label]' ),
			'tag'        => '',
			'status'     => true,
			'attributes' => array(),
		);
		$data[] = array(
			'key'        => 'wcc_disable_category_label',
			'content'    => do_shortcode( '[wcc_disable_category_label]' ),
			'tag'        => '',
			'status'     => true,
			'attributes' => array(),
		);

		$data[] = array(
			'key'        => 'wcc_video_placeholder',
			'content'    => do_shortcode( '[wcc_video_placeholder]' ),
			'tag'        => '',
			'status'     => true,
			'attributes' => array(),
		);
		$data[] = array(
			'key'        => 'wcc_enable_optout_label',
			'content'    => do_shortcode( '[wcc_enable_optout_label]' ),
			'tag'        => '',
			'status'     => true,
			'attributes' => array(),
		);
		$data[] = array(
			'key'        => 'wcc_disable_optout_label',
			'content'    => do_shortcode( '[wcc_disable_optout_label]' ),
			'tag'        => '',
			'status'     => true,
			'attributes' => array(),
		);
		$data[] = array(
			'key'        => 'wcc_optout_toggle_label',
			'content'    => do_shortcode( '[wcc_optout_toggle_label]' ),
			'tag'        => '',
			'status'     => true,
			'attributes' => array(),
		);
		$data[] = array(
			'key'        => 'wcc_optout_option_title',
			'content'    => do_shortcode( '[wcc_optout_option_title]' ),
			'tag'        => '',
			'status'     => true,
			'attributes' => array(),
		);
		$data[] = array(
			'key'        => 'wcc_optout_close_label',
			'content'    => do_shortcode( '[wcc_optout_close_label]' ),
			'tag'        => '',
			'status'     => true,
			'attributes' => array(),
		);
		return $data;
	}

	/**
	 * Determines whether the current/given language code is right-to-left (RTL)
	 *
	 * @param string $language Current language.
	 * @return boolean
	 */
	public function is_rtl( $language = '' ) {
		if ( ! $language ) {
			$language = wcc_current_language();
		}
		return in_array( $language, array( 'ar', 'az', 'dv', 'he', 'ku', 'fa', 'ur' ), true );
	}
	/**
	 * Exclude GoogleTagScripts from providerslist
	 * @param array $providers
	 * @return array
	 * @since  3.0.0
	 */
	public function exclude_google_tag_scripts( $providers ) {
		$google_tag_script_patterns = array(
			'ads.google.com',
			'google-analytics.com',
			'googletagmanager.com',
			'googleadservices.com',
			'googlesyndication.com/pagead/js/adsbygoogle.js',
		);

		foreach ( $providers as $key => $provider ) {
			foreach ( $google_tag_script_patterns as $pattern ) {
				if ( strpos( $provider['re'], $pattern ) !== false ) {
					unset( $providers[ $key ] );
					break;
				}
			}
		}

		// Reindex the array to maintain numeric keys
		return array_values( $providers );
	}
	/**
	 * Checks if a specific category exists within the no-cookie script options.
	 *
	 * @since 3.2.0
	 * @param string $category The category name to check.
	 * @return bool True if the category exists and contains items, otherwise false.
	 */
	public function check_no_cookie_script_exist( $category ) {
		$options = get_option( 'wcc_no_cookie_script', array() );
		// Iterate through all categories in the options
		foreach ( $options as $cat_slug => $providers ) {
			// Check if providers is an array and iterate through each provider
			if ( is_array( $providers ) ) {
				foreach ( $providers as $provider ) {
					$this->build_provider_list( $provider, $cat_slug );
				}
			}
		}
		// Check if the category exists in options and has a non-empty array
		return ! empty( $options[ $category ] ) && is_array( $options[ $category ] );
	}
	/**
	 * Check if WP Consent API plugin is enabled.
	 *
	 * @since 3.2.0
	 * @return bool True if the WP Consent API plugin is enabled, otherwise false.
	 */
	public function is_wpconsentapi_enabled() {
		return class_exists( 'WP_CONSENT_API' );
	}
	/**
	 * Exclude Microsoft scripts from providerslist
	 * @param array $providers
	 * @return array
	 * @since  3.3.1
	 */
	public function exclude_microsoft_scripts( $providers ) {
		$microsoft_script_patterns = array(
			'microsoft.com',
			'msn.com',
			'bing.com',
			'clarity.ms',
		);
		foreach ( $providers as $key => $provider ) {
			foreach ( $microsoft_script_patterns as $pattern ) {
				if ( strpos( $provider['re'], $pattern ) !== false ) {
					unset( $providers[ $key ] );
				}
			}
		}
		return array_values( $providers );
	}
	/**
	 * Get Microsoft Clarity Consent Mode data
	 * @since 3.4.0
	 * @return array
	 */
	public function get_mcm_data() {
		return array(
			'clarity' => $this->microsoft_clarity,
			'uet'     => $this->microsoft_uet,
		);
	}
}
