<?php

/**
 * Handles shortcodes used by the plugin.
 *
 * @link       https://www.webtoffee.com/
 * @since      3.0.0
 *
 * @package    WebToffee\CookieConsent\Lite\Includes
 */

namespace WebToffee\CookieConsent\Lite\Frontend\Modules\Shortcodes;

use WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Includes\Category_Controller;
use WebToffee\CookieConsent\Lite\Frontend\Modules\Iab\Iab;
use WebToffee\CookieConsent\Lite\Admin\Modules\Banners\Includes\Controller;


if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Handles shortcodes
 *
 * @class       Shortcodes
 * @version     3.0.0
 * @package     WebToffee\CookieConsent\Lite\Frontend\Modules\Shortcodes
 */
class Shortcodes {



	/**
	 * Banner properties
	 *
	 * @var array
	 */
	protected $properties;

	/**
	 * Banner contents
	 *
	 * @var array
	 */
	protected $contents;

	/**
	 * Banner template
	 *
	 * @var array
	 */
	protected $template;

	/**
	 * Banner default language
	 *
	 * @var string
	 */
	protected $language = 'en';

	/**
	 * Shortcode data, loads based on versions.
	 *
	 * @var array
	 */
	protected $shortcode_data;

	/**
	 * Available shortcodes
	 *
	 * @var array
	 */
	protected $shortcodes;

	/**
	 * Check if connected to web app.
	 *
	 * @var boolean
	 */
	protected $connected;
	/**
	 * Check if iab enabled
	 *
	 * @var boolean
	 */
	protected $is_iab_enabled;
	/**
	 * Check if gac enabled
	 *
	 * @var boolean
	 */
	protected $is_gac_enabled;

	/**
	 * Check if google privacy policy enabled
	 *
	 * @var boolean
	 */
	protected $is_gpp_enabled;

	/**
	 * Check if preview mode is active.
	 *
	 * @var boolean
	 */
	private $preview = false;

	/**
	 * Check if preview mode is active.
	 *
	 * @var boolean
	 */
	private $law = 'gdpr';

	/**
	 * Default constructor
	 *
	 * @param object  $banner Banner object.
	 * @param boolean $template Banner template.
	 */
	public function __construct( $banner, $template = false ) {
		$contents         = $banner->get_contents();
		$settings         = $banner->get_settings();
		$this->preview    = defined( 'WCC_PREVIEW_REQUEST' ) && WCC_PREVIEW_REQUEST;
		$this->connected  = wcc_is_cloud_request();
		$this->language   = $banner->get_language();
		$this->template   = $template;
		$this->properties = $settings;
		$this->law        = $banner->get_law();
		$this->contents   = isset( $contents[ $this->language ] ) ? $contents[ $this->language ] : '';
		// Set the IAB status
		$this->is_iab_enabled = ! empty( $settings['settings']['iab_enabled'] ) && wp_validate_boolean( $settings['settings']['iab_enabled'] );
		// Set the Google additional consent status
		$this->is_gac_enabled = ! empty( $settings['settings']['gacm_enabled'] ) && wp_validate_boolean( $settings['settings']['gacm_enabled'] );
		// Set the Google privacy policy status
		$this->is_gpp_enabled = ! empty( $settings['config']['preferenceCenter']['elements']['googlePrivacyPolicy']['status'] ) && wp_validate_boolean( $settings['config']['preferenceCenter']['elements']['googlePrivacyPolicy']['status'] );
		$this->load_shortcodes();
		$this->init();
	}

	/**
	 * Load shortcodes from a json file
	 *
	 * @return void
	 */
	private function load_shortcodes() {
		$this->shortcodes = wcc_read_json_file( __DIR__ . '/versions/' . esc_html( $this->template ) . '/shortcodes.json' );
	}
	/**
	 * Init shortcodes.
	 */
	public function init() {
		$shortcodes = ( isset( $this->shortcodes ) && is_array( $this->shortcodes ) ) ? $this->shortcodes : array();
		if ( empty( $shortcodes ) ) {
			return false;
		}
		foreach ( $shortcodes as $shortcode ) {
			$code     = $shortcode['key'];
			$function = array( $this, $code );
			if ( method_exists( $this, $code ) ) {
				add_shortcode( apply_filters( "wcc_{$code}_shortcode_tag", $code ), $function );
			}
		}
	}

	/**
	 * Return notice title
	 *
	 * @return string
	 */
	public function wcc_notice_title() {
		return isset( $this->contents['notice']['elements']['title'] ) ? $this->contents['notice']['elements']['title'] : '';
	}

	/**
	 * Return notice description
	 *
	 * @return string
	 */
	public function wcc_notice_description() {
		return isset( $this->contents['notice']['elements']['description'] ) ? do_shortcode( $this->contents['notice']['elements']['description'] ) : '';
	}

	/**
	 * Return accept button text
	 *
	 * @return string
	 */
	public function wcc_accept_text() {
		return isset( $this->contents['notice']['elements']['buttons']['elements']['accept'] ) ? $this->contents['notice']['elements']['buttons']['elements']['accept'] : '';
	}

	/**
	 * Return reject button text
	 *
	 * @return string
	 */
	public function wcc_reject_text() {
		return isset( $this->contents['notice']['elements']['buttons']['elements']['reject'] ) ? $this->contents['notice']['elements']['buttons']['elements']['reject'] : '';
	}

	/**
	 * Return settings button text
	 *
	 * @return string
	 */
	public function wcc_settings_text() {
		return isset( $this->contents['notice']['elements']['buttons']['elements']['settings'] ) ? $this->contents['notice']['elements']['buttons']['elements']['settings'] : '';
	}

	/**
	 * Return readmore button text
	 *
	 * @return string
	 */
	public function wcc_readmore_text() {
		return isset( $this->contents['notice']['elements']['buttons']['elements']['readMore'] ) ? $this->contents['notice']['elements']['buttons']['elements']['readMore'] : '';
	}

	/**
	 * Returns donotsell button text
	 *
	 * @return string
	 */
	public function wcc_donotsell_text() {
		return isset( $this->contents['notice']['elements']['buttons']['elements']['donotSell'] ) ? $this->contents['notice']['elements']['buttons']['elements']['donotSell'] : '';
	}

	/**
	 * Preference title
	 *
	 * @return string
	 */
	public function wcc_preference_title() {
		return isset( $this->contents['preferenceCenter']['elements']['title'] ) ? $this->contents['preferenceCenter']['elements']['title'] : '';
	}

	/**
	 * Return preference description
	 *
	 * @return string
	 */
	public function wcc_preference_description() {
		return isset( $this->contents['preferenceCenter']['elements']['description'] ) ? $this->contents['preferenceCenter']['elements']['description'] : '';
	}

	/**
	 * Return preference accept button text
	 *
	 * @return string
	 */
	public function wcc_preference_accept_text() {
		return isset( $this->contents['preferenceCenter']['elements']['buttons']['elements']['accept'] ) ? $this->contents['preferenceCenter']['elements']['buttons']['elements']['accept'] : '';
	}

	/**
	 * Return preference reject button text
	 *
	 * @return string
	 */
	public function wcc_preference_reject_text() {
		return isset( $this->contents['preferenceCenter']['elements']['buttons']['elements']['reject'] ) ? $this->contents['preferenceCenter']['elements']['buttons']['elements']['reject'] : '';
	}

	/**
	 * Return preference save button text
	 *
	 * @return string
	 */
	public function wcc_preference_save_text() {
		return isset( $this->contents['preferenceCenter']['elements']['buttons']['elements']['save'] ) ? $this->contents['preferenceCenter']['elements']['buttons']['elements']['save'] : '';
	}

	/**
	 * Return preference always enabled text
	 *
	 * @return string
	 */
	public function wcc_preference_always_enabled() {
		return isset( $this->contents['preferenceCenter']['elements']['category']['elements']['alwaysEnabled'] ) ? $this->contents['preferenceCenter']['elements']['category']['elements']['alwaysEnabled'] : '';
	}

	/**
	 * Callback for the shortcode [wcc_revisit_title]
	 *
	 * @return string
	 */
	public function wcc_revisit_title() {
		return isset( $this->contents['revisitConsent']['elements']['title'] ) ? $this->contents['revisitConsent']['elements']['title'] : '';
	}

	/**
	 * Callback for the shortcode [wcc_preview_save_text]
	 *
	 * @return string
	 */
	public function wcc_preview_save_text() {
		return isset( $this->contents['categoryPreview']['elements']['buttons']['elements']['save'] ) ? $this->contents['categoryPreview']['elements']['buttons']['elements']['save'] : '';
	}
	/**
	 * Return accept button HTML
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string
	 */
	public function wcc_accept( $atts ) {
		return $this->get_btn_html( 'accept-button' );
	}

	/**
	 * Return reject button HTML
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string
	 */
	public function wcc_reject( $atts ) {
		return $this->get_btn_html( 'reject-button' );
	}

	/**
	 * Return settings button HTML
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string
	 */
	public function wcc_settings( $atts ) {
		return $this->get_btn_html( 'settings-button' );
	}

	/**
	 * Return readmore button HTML
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string
	 */
	public function wcc_readmore( $atts ) {
		return $this->get_btn_html( 'readmore-button' );
	}

	/**
	 * Return donotsell button HTML
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string
	 */
	public function wcc_donot_sell( $atts ) {
		return $this->get_btn_html( 'donotsell-button' );
	}

	/**
	 * Return button HTML
	 *
	 * @param string $tag Shortcode tag.
	 * @return string
	 */
	public function get_btn_html( $tag = 'settings-button' ) {

		$config         = wcc_array_search( $this->properties['config'], 'tag', $tag );
		$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', $tag );

		if ( false === $shortcode_data ) {
			return '';
		}
		$btn_html = isset( $shortcode_data['content']['button'] ) ? $shortcode_data['content']['button'] : '';
		if ( isset( $config['type'] ) && 'link' === $config['type'] ) {
			$btn_html = isset( $shortcode_data['content']['link'] ) ? wp_kses( $shortcode_data['content']['link'], wcc_allowed_html() ) : '';
		}
		return do_shortcode( $btn_html );
	}

	/**
	 * Return preference table HTML
	 *
	 * @return string
	 */
	public function wcc_preference_category() {
		$html           = '';
		$categories     = Category_Controller::get_instance()->get_items();
		$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'detail-categories' );
		$content        = isset( $shortcode_data['content']['container'] ) ? wp_kses( $shortcode_data['content']['container'], wcc_allowed_html() ) : '';

		if ( '' === $content ) {
			return $html;
		}

		foreach ( $categories as $category ) {
			$category = new \WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Includes\Cookie_Categories( $category );
			if ( false === $category->get_visibility() && false === $this->preview ) {
				continue;
			}
			$audit_table = $this->wcc_audit_table( $category->get_cookies() );
			$description = $category->get_description( $this->language );
			$name        = $category->get_name( $this->language );

			$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'detail-category-toggle' );

			$html .= str_replace(
				array(
					'[wcc_preference_{{category_slug}}_title]',
					'[wcc_preference_{{category_slug}}_status]',
					'[wcc_preference_{{category_slug}}_description]',
					'{{category_slug}}',
					'[wcc_audit_table]',
				),
				array(
					esc_html( $name ),
					esc_html( $category->get_prior_consent() ),
					wp_kses_post( $description ),
					esc_html( $category->get_slug() ),
					$audit_table,
				),
				$content
			);
		}
		return do_shortcode( $html );
	}

	/**
	 * Return category list HTML
	 *
	 * @return string
	 */
	public function wcc_category_list() {
		$html           = '';
		$categories     = Category_Controller::get_instance()->get_items();
		$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'categories-list' );
		$content        = isset( $shortcode_data['content']['container'] ) ? wp_kses( $shortcode_data['content']['container'], wcc_allowed_html() ) : '';

		if ( '' === $content ) {
			return $html;
		}

		foreach ( $categories as $category ) {
			$category = new \WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Includes\Cookie_Categories( $category );
			if ( false === $category->get_visibility() && false === $this->preview ) {
				continue;
			}
			$audit_table = $this->wcc_audit_table_by_category( $category );
			$description = $category->get_description( $this->language );
			$name        = $category->get_name( $this->language );

			$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'detail-category-toggle' );

			$html .= str_replace(
				array(
					'[wcc_category_{{category_slug}}_title]',
					'[wcc_category_{{category_slug}}_description]',
					'{{category_slug}}',
					'[wcc_cookie_audit_table]',
				),
				array(
					esc_html( $name ),
					wp_kses_post( $description ),
					esc_html( $category->get_slug() ),
					wp_kses( $audit_table, wcc_allowed_html() ),
				),
				$content
			);
		}
		return do_shortcode( $html );
	}

	/**
	 * Cookie prefrence button
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string
	 */
	public function wcc_preference_button( $atts ) {
		$atts = shortcode_atts(
			array(
				'data-style' => 'color:#000000; border-color:#000000; background-color:transparent; text-decoration:underline;', // Default styles
			),
			$atts
		);

		$revisit_title = $this->wcc_revisit_title();
		$button_html   = sprintf(
			'<a role="button" class="wcc-preference-btn" data-tag="preference-btn" style="%s">%s</a>',
			esc_attr( $atts['data-style'] ),
			esc_html( $revisit_title )
		);

		return $button_html;
	}

	/**
	 * Cookie audit table.
	 *
	 * @param array $cookies Cookies array.
	 * @return string
	 */
	public function wcc_audit_table( $cookies = array() ) {
		$html = '';

		$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'audit-table' );
		$config         = wcc_array_search( $this->properties['config'], 'tag', 'audit-table' );

		if ( isset( $config['status'] ) && false === $config['status'] ) {
			return;
		}

		$container = isset( $shortcode_data['content']['container'] ) ? $shortcode_data['content']['container'] : '';
		if ( '' === $shortcode_data ) {
			return $html;
		}
		$contents = isset( $this->contents['auditTable']['elements'] ) ? $this->contents['auditTable']['elements'] : '';

		if ( empty( $cookies ) ) {
			$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'audit-table-empty' );
			$container      = isset( $shortcode_data['content']['container'] ) ? $shortcode_data['content']['container'] : '';
			$html           = do_shortcode( $container );
			return $html;
		}
		foreach ( $cookies as $cookie ) {
			// Check if cookie is present in the cookie list after last update.
			if ( false === $cookie['cookie_status'] ) {
				continue;
			}

			$description = $cookie['description'];
			$duration    = $cookie['duration'];
			$description = isset( $description[ $this->language ] ) ? $description[ $this->language ] : '';
			$duration    = isset( $duration[ $this->language ] ) ? $duration[ $this->language ] : '';

			// Define standard fields array
			$standard_fields = array(
				array(
					'label'  => isset( $contents['headers']['elements']['id'] ) ? $contents['headers']['elements']['id'] : '',
					'value'  => $cookie['name'],
					'escape' => 'html',
				),
				array(
					'label'  => isset( $contents['headers']['elements']['duration'] ) ? $contents['headers']['elements']['duration'] : '',
					'value'  => $duration,
					'escape' => 'html',
				),
				array(
					'label'  => isset( $contents['headers']['elements']['description'] ) ? $contents['headers']['elements']['description'] : '',
					'value'  => $description,
					'escape' => 'kses',
				),
			);

			// Apply filter to modify standard fields
			$fields = apply_filters( 'wcc_audit_table_fields', $standard_fields, $cookie, $this->language, $contents );

			// Generate table body from fields array
			$table_body = '';
			foreach ( $fields as $field ) {
				if ( isset( $field['label'] ) && isset( $field['value'] ) ) {
					$table_body .= '<li>';
					$table_body .= '<div>' . esc_html( $field['label'] ) . '</div>';

					// Handle different escape types
					if ( isset( $field['escape'] ) && $field['escape'] === 'kses' ) {
						$table_body .= '<div>' . wp_kses( $field['value'], wcc_allowed_html() ) . '</div>';
					} else {
						$table_body .= '<div>' . esc_html( $field['value'] ) . '</div>';
					}

					$table_body .= '</li>';
				}
			}

			$html .= str_replace(
				array(
					'[CONTENT]',
				),
				array(
					$table_body,
				),
				$container
			);
		}
		return $html;
	}

	/**
	 * Return cookie list of all categories HTML
	 *
	 * @return string
	 */
	public function wcc_cookies_audit_table( $atts ) {
		$html           = '';
		$categories     = Category_Controller::get_instance()->get_items();
		$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'categories-audit-list' );
		$content        = isset( $shortcode_data['content']['container'] ) ? wp_kses( $shortcode_data['content']['container'], wcc_allowed_html() ) : '';
		$contents       = isset( $this->contents['auditTable']['elements'] ) ? $this->contents['auditTable']['elements'] : '';
		if ( empty( $content ) ) {
			return $html;
		}

		// Parse shortcode attributes
		$atts = shortcode_atts(
			array(
				'category' => '',
			),
			$atts,
			'wcc_cookies_audit_table'
		);

		$filter_categories = array_filter( array_map( 'trim', explode( ',', $atts['category'] ) ) );

		// Define standard table headers
		$standard_headers = array(
			array(
				'label' => isset( $contents['headers']['elements']['id'] ) ? $contents['headers']['elements']['id'] : '',
			),
			array(
				'label' => isset( $contents['headers']['elements']['duration'] ) ? $contents['headers']['elements']['duration'] : '',
			),
			array(
				'label' => isset( $contents['headers']['elements']['description'] ) ? $contents['headers']['elements']['description'] : '',
			),
		);

		// Apply filter to modify table headers
		$headers = apply_filters( 'wcc_cookies_audit_table_headers', $standard_headers, $contents );

		// Generate table head
		$table_head = '<thead><tr>';
		foreach ( $headers as $header ) {
			if ( isset( $header['label'] ) ) {
				$table_head .= '<th>' . esc_html( $header['label'] ) . '</th>';
			}
		}
		$table_head .= '</tr></thead>';

		$table_body    = '<tbody>';
		$cookies_found = false; // Flag to check if any cookies are found
		foreach ( $categories as $category ) {
			$category_obj = new \WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Includes\Cookie_Categories( $category );
			if ( false === $category_obj->get_visibility() && false === $this->preview ) {
				continue;
			}
			// Filter categories if filter_categories is not empty
			if ( ! empty( $filter_categories ) && ! in_array( $category_obj->get_slug(), $filter_categories, true ) ) {
				continue;
			}
			$cookies = $category_obj->get_cookies();
			if ( empty( $cookies ) ) {
				continue;
			}
			$cookies_found = true; // Set flag to true if cookies are found
			foreach ( $cookies as $cookie ) {
				// Skip cookies with cookie_status = false
				if ( empty( $cookie['cookie_status'] ) ) {
					continue;
				}
				$description = $cookie['description'];
				$duration    = $cookie['duration'];
				$description = isset( $description[ $this->language ] ) ? $description[ $this->language ] : '';
				$duration    = isset( $duration[ $this->language ] ) ? $duration[ $this->language ] : '';

				// Define standard fields array
				$standard_fields = array(
					array(
						'label'  => isset( $contents['headers']['elements']['id'] ) ? $contents['headers']['elements']['id'] : '',
						'value'  => $cookie['name'],
						'escape' => 'html',
					),
					array(
						'label'  => isset( $contents['headers']['elements']['duration'] ) ? $contents['headers']['elements']['duration'] : '',
						'value'  => $duration,
						'escape' => 'html',
					),
					array(
						'label'  => isset( $contents['headers']['elements']['description'] ) ? $contents['headers']['elements']['description'] : '',
						'value'  => $description,
						'escape' => 'kses',
					),
				);

				// Apply filter to modify table fields
				$fields = apply_filters( 'wcc_cookies_audit_table_fields', $standard_fields, $cookie, $this->language, $contents, $category_obj );

				// Generate table row
				$table_body .= '<tr>';
				foreach ( $fields as $field ) {
					if ( isset( $field['value'] ) ) {
						// Handle different escape types
						if ( isset( $field['escape'] ) && $field['escape'] === 'kses' ) {
							$table_body .= '<td>' . wp_kses( $field['value'], wcc_allowed_html() ) . '</td>';
						} else {
							$table_body .= '<td>' . esc_html( $field['value'] ) . '</td>';
						}
					}
				}
				$table_body .= '</tr>';
			}
		}
		if ( ! $cookies_found ) {
			$colspan     = count( $headers );
			$table_body .= '<tr><td colspan="' . $colspan . '" style="text-align:center;">' . $this->wcc_audit_table_empty_text() . '</td></tr>';
		}
		$table_body    .= '</tbody>';
		$audit_table    = $table_head . $table_body;
		$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'detail-category-toggle' );
		$html          .= str_replace( '[wcc_cookie_audit_table]', wp_kses( $audit_table, wcc_allowed_html() ), $content );
		return do_shortcode( $html );
	}

	/**
	 * Category detail preview.
	 *
	 * @return string
	 */
	public function wcc_preview_category() {
		$html           = '';
		$categories     = Category_Controller::get_instance()->get_items();
		$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'detail-category-preview' );
		$container      = isset( $shortcode_data['content']['container'] ) ? $shortcode_data['content']['container'] : '';
		foreach ( $categories as $category ) {
			$object = new \WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Includes\Cookie_Categories( $category );
			$name   = $object->get_name( $this->language );
			$html  .= str_replace(
				array(
					'[wcc_preview_{{category_slug}}_title]',
					'{{category_slug}}',
				),
				array(
					$name,
					$object->get_slug(),
				),
				$container
			);
		}
		return $html;
	}

	/**
	 * Callback for the shortcode [wcc_privacy_link]
	 *
	 * @return string
	 */
	public function wcc_privacy_link() {
		return isset( $this->contents['notice']['elements']['privacyLink'] ) && '' !== $this->contents['notice']['elements']['privacyLink'] ? esc_url( $this->contents['notice']['elements']['privacyLink'] ) : '#';
	}

	/**
	 * Callback for the shortcode [wcc_show_desc]
	 *
	 * @return string
	 */
	public function wcc_show_desc() {
		return $this->get_btn_html( 'show-desc-button' );
	}

	/**
	 * Callback for the shortcode [wcc_hide_desc]
	 *
	 * @return string
	 */
	public function wcc_hide_desc() {
		return $this->get_btn_html( 'hide-desc-button' );
	}

	/**
	 * Callback for the shortcode [wcc_showmore_text]
	 *
	 * @return string
	 */
	public function wcc_showmore_text() {
		$key = 'ccpa' === $this->law ? 'optoutPopup' : 'preferenceCenter';
		return isset( $this->contents[ $key ]['elements']['showMore'] ) ? $this->contents[ $key ]['elements']['showMore'] : '';
	}

	/**
	 * Callback for the shortcode [wcc_showless_text]
	 *
	 * @return string
	 */
	public function wcc_showless_text() {
		$key = 'ccpa' === $this->law ? 'optoutPopup' : 'preferenceCenter';
		return isset( $this->contents[ $key ]['elements']['showLess'] ) ? $this->contents[ $key ]['elements']['showLess'] : '';
	}

	/**
	 * Callback for the shortcode [wcc_enable_category_label]
	 *
	 * @return string
	 */
	public function wcc_enable_category_label() {
		return isset( $this->contents['preferenceCenter']['elements']['category']['elements']['enable'] ) ? $this->contents['preferenceCenter']['elements']['category']['elements']['enable'] : '';
	}

	/**
	 * Callback for the shortcode [wcc_disable_category_label]
	 *
	 * @return string
	 */
	public function wcc_disable_category_label() {
		return isset( $this->contents['preferenceCenter']['elements']['category']['elements']['disable'] ) ? $this->contents['preferenceCenter']['elements']['category']['elements']['disable'] : '';
	}

	/**
	 * Callback for the shortcode [wcc_audit_table_empty_text]
	 *
	 * @return string
	 */
	public function wcc_audit_table_empty_text() {
		return isset( $this->contents['auditTable']['elements']['message'] ) ? $this->contents['auditTable']['elements']['message'] : '';
	}

	/**
	 * Callback for the shortcode [wcc_notice_close_label]
	 *
	 * @return string
	 */
	public function wcc_notice_close_label() {
		return isset( $this->contents['notice']['elements']['closeButton'] ) ? $this->contents['notice']['elements']['closeButton'] : '';
	}

	/**
	 * Callback for the shortcode [wcc_preference_close_label]
	 *
	 * @since 3.3.1
	 * @return string
	 */
	public function wcc_preference_close_label() {
		return isset( $this->contents['preferenceCenter']['elements']['closeButtonAriaLabel'] ) ? $this->contents['preferenceCenter']['elements']['closeButtonAriaLabel'] : __( 'Close cookie preference modal', 'webtoffee-cookie-consent' );
	}

	/**
	 * Callback for the shortcode [wcc_optout_cancel_text]
	 *
	 * @return string
	 */
	public function wcc_optout_cancel_text() {
		return isset( $this->contents['optoutPopup']['elements']['buttons']['elements']['cancel'] ) ? $this->contents['optoutPopup']['elements']['buttons']['elements']['cancel'] : '';
	}

	/**
	 * Callback for the shortcode [wcc_optout_confirm_text]
	 *
	 * @return string
	 */
	public function wcc_optout_confirm_text() {
		return isset( $this->contents['optoutPopup']['elements']['buttons']['elements']['confirm'] ) ? $this->contents['optoutPopup']['elements']['buttons']['elements']['confirm'] : '';
	}

	/**
	 * Callback for the shortcode [wcc_optout_confirmation]
	 *
	 * @return string
	 */
	public function wcc_optout_confirmation() {
		return isset( $this->contents['optoutPopup']['elements']['confirmation'] ) ? $this->contents['optoutPopup']['elements']['confirmation'] : '';
	}

	/**
	 * Callback for the shortcode [wcc_category_toggle_label]
	 *
	 * @return string
	 */
	public function wcc_category_toggle_label() {
		$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'detail-category-toggle' );
		return isset( $shortcode_data['content']['container'] ) ? $shortcode_data['content']['container'] : '';
	}

	/**
	 * Callback for the shortcode [wcc_video_placeholder]
	 *
	 * @return string
	 */
	public function wcc_video_placeholder() {
		$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'video-placeholder' );
		return do_shortcode( isset( $shortcode_data['content']['container'] ) ? $shortcode_data['content']['container'] : '' );
	}

	/**
	 * Callback for the shortcode [wcc_video_placeholder_title]
	 *
	 * @return string
	 */
	public function wcc_video_placeholder_title() {
		return isset( $this->contents['videoPlaceholder']['elements']['title'] ) ? $this->contents['videoPlaceholder']['elements']['title'] : '';
	}

	/**
	 * Populate audit table.
	 *
	 * @return string
	 */
	public function wcc_outside_audit_table() {
		$html           = '';
		$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'outside-audit-table' );
		$container      = isset( $shortcode_data['content']['container'] ) ? $shortcode_data['content']['container'] : '';
		$categories     = Category_Controller::get_instance()->get_items();

		if ( empty( $categories ) ) {
			return $html;
		}

		foreach ( $categories as $category ) {
			$category = new \WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Includes\Cookie_Categories( $category );
			if ( false === $category->get_visibility() ) {
				continue;
			}
			$audit_table = $this->wcc_audit_table_by_category( $category );
			$name        = $category->get_name( $this->language );
			$html       .= str_replace(
				array(
					'[wcc_preference_{{category_slug}}_title]',
					'[CONTENT]',
				),
				array(
					esc_html( $name ),
					wp_kses( $audit_table, wcc_allowed_html() ),
				),
				$container
			);
		}
		return do_shortcode( $html );
	}

	/**
	 * Create audit-table for each category.
	 *
	 * @param object $category Category object.
	 * @return string
	 */
	public function wcc_audit_table_by_category( $category ) {
		$cookies = $category->get_cookies();

		// First check if there are any valid cookies (cookie_status = true)
		$valid_cookies = array_filter(
			$cookies,
			function ( $cookie ) {
				return ! empty( $cookie['cookie_status'] );
			}
		);

		$contents   = isset( $this->contents['auditTable']['elements'] ) ? $this->contents['auditTable']['elements'] : '';
		$html       = '';
		$thead_style = '';
		if ( empty( $valid_cookies ) ) {
			$thead_style = 'style="display:none;"';
		}

		$table_head = '<thead ' . $thead_style . '><tr>
			<th>' . esc_html( isset( $contents['headers']['elements']['id'] ) ? $contents['headers']['elements']['id'] : '' ) . '</th>
			<th>' . esc_html( isset( $contents['headers']['elements']['duration'] ) ? $contents['headers']['elements']['duration'] : '' ) . '</th>
			<th>' . esc_html( isset( $contents['headers']['elements']['description'] ) ? $contents['headers']['elements']['description'] : '' ) . '</th>
			</tr></thead>';

		// If no valid cookies, return empty table message
		if ( empty( $valid_cookies ) ) {
			return $table_head . '<tbody><tr><td colspan="3">' . $this->wcc_audit_table_empty_text() . '</td></tr></tbody>';
		}
		$table_body = '<tbody>';

		foreach ( $cookies as $cookie ) {
			// Skip cookies with cookie_status = false
			if ( empty( $cookie['cookie_status'] ) ) {
				continue;
			}

			$description = $cookie['description'];
			$duration    = $cookie['duration'];
			$description = isset( $description[ $this->language ] ) ? $description[ $this->language ] : '';
			$duration    = isset( $duration[ $this->language ] ) ? $duration[ $this->language ] : '';

			$table_body .= '<tr>';
			$table_body .= '<td>' . esc_html( $cookie['name'] ) . '</td>';
			$table_body .= '<td>' . esc_html( $duration ) . '</td>';
			$table_body .= '<td>' . wp_kses( $description, wcc_allowed_html() ) . '</td>';
			$table_body .= '</tr>';
		}

		$table_body .= '</tbody>';
		$html        = $table_head . $table_body;
		return $html;
	}

	public function wcc_optout_title() {
		return isset( $this->contents['optoutPopup']['elements']['title'] ) ? $this->contents['optoutPopup']['elements']['title'] : '';
	}
	public function wcc_optout_description() {
		return isset( $this->contents['optoutPopup']['elements']['description'] ) ? $this->contents['optoutPopup']['elements']['description'] : '';
	}
	public function wcc_optout_option_title() {
		return isset( $this->contents['optoutPopup']['elements']['optOption']['elements']['title'] ) ? $this->contents['optoutPopup']['elements']['optOption']['elements']['title'] : '';
	}
	public function wcc_optout_gpc_description() {
		return isset( $this->contents['optoutPopup']['elements']['gpcOption']['elements']['description'] ) ? $this->contents['optoutPopup']['elements']['gpcOption']['elements']['description'] : '';
	}
	public function wcc_enable_optout_label() {
		return isset( $this->contents['optoutPopup']['elements']['optOption']['elements']['enable'] ) ? $this->contents['optoutPopup']['elements']['optOption']['elements']['enable'] : '';
	}
	public function wcc_disable_optout_label() {
		return isset( $this->contents['optoutPopup']['elements']['optOption']['elements']['disable'] ) ? $this->contents['optoutPopup']['elements']['optOption']['elements']['disable'] : '';
	}
	public function wcc_optout_toggle_label() {
		$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'optout-option-toggle' );
		return isset( $shortcode_data['content']['container'] ) ? $shortcode_data['content']['container'] : '';
	}
	public function wcc_optout_close_label() {
		return isset( $this->contents['optoutPopup']['elements']['closeButton'] ) ? $this->contents['optoutPopup']['elements']['closeButton'] : '';
	}
	/**
	 * Return IAB Title
	 *
	 * @return string
	 */
	public function wcc_iab_notice_title() {
		return isset( $this->contents['iabDescription']['elements']['title'] ) ? $this->contents['iabDescription']['elements']['title'] : '';
	}
	/**
	 * Return IAB description
	 *
	 * @return string
	 */
	public function wcc_iab_notice_description() {
		return isset( $this->contents['iabDescription']['elements']['description'] ) ? $this->contents['iabDescription']['elements']['description'] : '';
	}
	/**
	 * Return IAB preference description
	 *
	 * @return string
	 */
	public function wcc_iab_preference_description() {
		if ( $this->is_gac_enabled ) {
			return isset( $this->contents['iabPreferenceDescription']['gacm']['description'] ) ? $this->contents['iabPreferenceDescription']['gacm']['description'] : '';
		}
		return isset( $this->contents['iabPreferenceDescription']['elements']['description'] ) ? $this->contents['iabPreferenceDescription']['elements']['description'] : '';
	}
	/**
	 * Return IAB cookie category title
	 *
	 * @return string
	 */
	public function wcc_iab_nav_item_cookie_categories() {
		return isset( $this->contents['iabCookieCategories']['elements']['title'] ) ? $this->contents['iabCookieCategories']['elements']['title'] : '';
	}
	/**
	 * Return IAB purpose and features title
	 *
	 * @return string
	 */
	public function wcc_iab_nav_item_purposes_n_features() {
		return isset( $this->contents['iabPurposesFeatures']['elements']['title'] ) ? $this->contents['iabPurposesFeatures']['elements']['title'] : '';
	}
	/**
	 * Return IAB vendors title
	 *
	 * @return string
	 */
	public function wcc_iab_nav_item_vendors() {
		return isset( $this->contents['iabVendorTitle']['elements']['title'] ) ? $this->contents['iabVendorTitle']['elements']['title'] : '';
	}
	/**
	 * Return vendor list array
	 *
	 * @return array
	 */
	public function get_iab_vendor_list() {
		if ( ! $this->is_iab_enabled ) {
			return array();
		}
		$vendors_arr = wp_cache_get( 'wcc_iab_vendor_list', 'wcc_vendor_list' );
		if ( ! $vendors_arr ) {
			$vendors_arr = Iab::get_instance()->get_vendor_list( $this->language );
			wp_cache_set( 'wcc_iab_vendor_list', $vendors_arr, 'wcc_vendor_list', 12 * HOUR_IN_SECONDS );
		}
		return $vendors_arr;
	}
	/**
	 * Return vendor list key array
	 *
	 * @return array
	 */
	public function get_iab_vendor_list_keys() {
		$key_arr     = array();
		$vendors_arr = $this->get_iab_vendor_list();
		if ( ! empty( $vendors_arr ) ) {
			foreach ( $vendors_arr as $item_key => $item ) {
				$allowed_key_arr = array( 'purposes', 'specialPurposes', 'features', 'specialFeatures', 'dataCategories', 'vendors' );
				if ( in_array( $item_key, $allowed_key_arr, true ) ) {
					if ( is_array( $item ) ) {
						foreach ( $item as $key => $value ) {
							if ( 'dataCategories' === $item_key ) {
								$key_arr['dataDeclaration'][ $key ] = $value;
							} else {
								$key_arr[ $item_key ][ $key ] = $value;
							}
						}
					}
				}
			}
		}
		return $key_arr;
	}
	/**
	 * Return IAB purposes and feactures section content HTML
	 *
	 * @return string
	 */
	public function wcc_iab_purpose_n_features_section() {
		$vendors_arr = array();
		$result_arr  = array();
		$html        = '';
		if ( ! $this->is_iab_enabled ) {
			return $html;
		}
		$vendors_arr   = $this->get_iab_vendor_list();
		$show_checkbox = true;
		if ( empty( $vendors_arr ) ) {
			return '';
		}
		/* Setting array which doesn't need legitimate interest */
		$not_in_legitimate_interest_arr = array( 1, 3, 4, 5, 6 );
		if ( ! empty( $vendors_arr ) ) {
			$allowed_key_arr = array( 'purposes', 'specialPurposes', 'features', 'specialFeatures' );
			foreach ( $vendors_arr as $v_key => $value ) {
				if ( in_array( $v_key, $allowed_key_arr, true ) ) {
					$result_arr[ $v_key ] = $value;
				}
			}
		}
		$legitimate_interest_txt = isset( $this->contents['iabDescription']['buttons']['elements']['legitimateInterest'] ) ? $this->contents['iabDescription']['buttons']['elements']['legitimateInterest'] : __( 'Legitimate Interest', 'webtoffee-cookie-consent' );
		foreach ( $result_arr as $item_key => $item ) {
			switch ( $item_key ) {
				case 'purposes':
					$title         = isset( $this->contents['iabPurposesFeatures']['elements'][ $item_key ] ) ? $this->contents['iabPurposesFeatures']['elements'][ $item_key ] : __( 'Purposes', 'webtoffee-cookie-consent' );
					$item_key      = 'purpose';
					$show_checkbox = true;
					break;
				case 'specialPurposes':
					$title         = isset( $this->contents['iabPurposesFeatures']['elements'][ $item_key ] ) ? $this->contents['iabPurposesFeatures']['elements'][ $item_key ] : __( 'Special Purposes', 'webtoffee-cookie-consent' );
					$item_key      = 'specialpurpose';
					$show_checkbox = false;
					break;
				case 'features':
					$title         = isset( $this->contents['iabPurposesFeatures']['elements'][ $item_key ] ) ? $this->contents['iabPurposesFeatures']['elements'][ $item_key ] : __( 'Features', 'webtoffee-cookie-consent' );
					$item_key      = 'feature';
					$show_checkbox = false;
					break;
				case 'specialFeatures':
					$title         = isset( $this->contents['iabPurposesFeatures']['elements'][ $item_key ] ) ? $this->contents['iabPurposesFeatures']['elements'][ $item_key ] : __( 'Special Features', 'webtoffee-cookie-consent' );
					$item_key      = 'specialfeature';
					$show_checkbox = true;
					break;
				default:
					$title = '';
					break;
			}
			$html .= '<div class="wcc-accordion" id="accordion-' . esc_attr( $item_key ) . '" aria-expanded="false">';
			$html .= $this->wcc_generate_accordion_header( $item_key, $title, $show_checkbox );
			$html .= '<div class="wcc-accordion-body" aria-hidden="true">';
			if ( ! is_array( $item ) ) {
				return '';
			}
			$purpose_feature_data     = get_option( 'wcc_purpose_features', array() );
			$allowed_purpose_features = array( 'specialfeature', 'specialpurpose', 'feature' );
			if ( is_array( $item ) ) {
				foreach ( $item as $key => $value ) {
					$title             = $item_key;
					$name              = isset( $value['name'] ) ? sanitize_text_field( $value['name'] ) : '';
					$description       = isset( $value['description'] ) ? sanitize_text_field( $value['description'] ) : '';
					$illustration_arr  = isset( $value['illustrations'] ) ? $value['illustrations'] : '';
					$illustration_html = '';
					if ( is_array( $illustration_arr ) && ! empty( $illustration_arr ) ) {
						$illustration_html .= '<label class="wcc-tab-content-illustration-header">' . esc_html( 'Illustrations', 'webtoffee-cookie-consent' ) . '</label> <ul class="wcc-iab-illustrations-des">';
						foreach ( $illustration_arr as $ills_key => $ills_value ) {
							$illustration_html .= '<li>' . esc_html( $ills_value, 'webtoffee-cookie-consent' ) . '</li>';
						}
						$illustration_html .= '</ul></label>';
					}
					$vendor_consent_text = __( 'Number of Vendors seeking consent or relying on legitimate interest:', 'webtoffee-cookie-consent' );
					// phpcs:ignore WordPress.PHP.StrictInArray.FoundNonStrictFalse
					if ( in_array( $key, $not_in_legitimate_interest_arr, true ) || in_array( $item_key, $allowed_purpose_features, true ) ) {
						$legitimate_interest = false;
					} else {
						$legitimate_interest = true;
					}
					// If purpose and features are configured in admin dashboard
					if ( ! empty( $purpose_feature_data ) ) {
						if ( isset( $purpose_feature_data[ $item_key ] ) ) {
							foreach ( $purpose_feature_data[ $item_key ] as $item ) {
								if ( isset( $item['id'] ) && $item['id'] === $key ) {
									$html .= '<div class="wcc-child-accordion" id="wcc-' . esc_attr( $item_key ) . '-inner-accordion-' . esc_attr( $key ) . '" aria-expanded="false">';
									$html .= $this->wcc_generate_accordion_body( $item_key, $name, $description, $illustration_html, $vendor_consent_text, $legitimate_interest, $key, $show_checkbox );
									$html .= '</div>'; // Closing inner accordion

								}
							}
						}
					} else {
						$html .= '<div class="wcc-child-accordion" id="wcc-' . esc_attr( $item_key ) . '-inner-accordion-' . esc_attr( $key ) . '" aria-expanded="false">';
						$html .= $this->wcc_generate_accordion_body( $item_key, $name, $description, $illustration_html, $vendor_consent_text, $legitimate_interest, $key, $show_checkbox );
						$html .= '</div>'; // Closing inner accordion
					}
				}
			}
			$html .= '</div>'; // Closing accordion body
			$html .= '</div>'; // Closing accordion
		}
		return $html;
	}
	/**
	 * Return IAB vendors section content
	 *
	 * @return string
	 */
	public function wcc_iab_vendors_section() {
		$html = '';
		if ( ! $this->is_iab_enabled ) {
			return $html;
		}
		$title = isset( $this->contents['iabVendorTitle']['elements']['third_party_subtitle'] ) ? $this->contents['iabVendorTitle']['elements']['third_party_subtitle'] : __( 'Third Party Vendors', 'webtoffee-cookie-consent' );
		$html .= '<div class="wcc-accordion" id="accordion-vendors" aria-expanded="false">';
		$html .= $this->wcc_generate_accordion_header( 'vendors', $title, true );
		$html .= '<div class="wcc-accordion-body" aria-hidden="true">';
		$html .= '<div class="wcc-vendor-storage-disclosure-section"></div></div>'; // Closing accordion body
		$html .= '</div>'; // Closing accordion
		if ( $this->is_gac_enabled ) {
			$title = isset( $this->contents['iabVendorTitle']['elements']['google_vendors_subtitle'] ) ? $this->contents['iabVendorTitle']['elements']['google_vendors_subtitle'] : __( 'Google Ad Tech Providers', 'webtoffee-cookie-consent' );
			$html .= '<div class="wcc-accordion" id="accordion-google-vendors" aria-expanded="false">';
			$html .= $this->wcc_generate_accordion_header( 'google-vendors', $title, true );
			$html .= '<div class="wcc-accordion-body" aria-hidden="true">';
			$html .= '</div>'; // Closing accordion body
			$html .= '</div>'; // Closing accordion
		}
		return $html;
	}
	/**
	 * Generate accordion header html
	 *
	 * @return string
	 */
	public function wcc_generate_accordion_header( $item_key, $title, $show_checkbox = true ) {
		$html = '<div class="wcc-accordion-iab-item">
					<div class="wcc-accordion-chevron"><i class="wcc-chevron-right"></i></div>
					<div class="wcc-accordion-header-wrapper">
						<div class="wcc-accordion-header">
							<div class="wcc-accordion-title">
								<button class="wcc-accordion-btn" aria-label="' . esc_attr( $title ) . '"
								data-tag="detail-category-title">' . esc_html( $title ) . '</button>';
		if ( 'vendors' === $item_key ) {
			$vendor_list_length = count( Iab::get_instance()->get_allowed_vendor_list() );
			$html              .= '<div class="wcc-vendor-count">&nbsp;(' . esc_html( $vendor_list_length ) . ')</div>';
		} elseif ( 'google-vendors' === $item_key ) {
			$gac_vendor_list_length = count( Iab::get_instance()->get_allowed_google_vendor_list() );
			$html                  .= '<div class="wcc-vendor-count">&nbsp;(' . esc_html( $gac_vendor_list_length ) . ')</div>';
		}
		$html .= '</div>';
		if ( $show_checkbox ) {
			$html .= '<div class="wcc-switch-wrapper">
												<div class="wcc-switch wcc-consent-switch-wrapper wcc-switch-sm" data-tag="detail-category-toggle">
													<label for="wcc-' . esc_attr( $item_key ) . '-consents-checkbox" class="wcc-switch" aria-label="Consent for ' . esc_html( $title ) . '"></label>
													<input type="checkbox" id= "wcc-' . esc_attr( $item_key ) . '-consents-checkbox" class="wcc-consents-checkbox" data-tag="' . esc_attr( $item_key ) . '">
												</div>
											</div>';
		}

		$html .= '</div>
					</div>
				</div>';
		return $html;
	}
	/**
	 * Generate accordion body html
	 *
	 * @return string
	 */
	public function wcc_generate_accordion_body( $purpose_feature_key, $name, $description, $illustration_html, $vendor_consent_text, $legitimate_interest, $item_key, $show_checkbox ) {
		$legitimate_interest_txt = isset( $this->contents['iabDescription']['buttons']['elements']['legitimateInterest'] ) ? $this->contents['iabDescription']['buttons']['elements']['legitimateInterest'] : __( 'Legitimate Interest', 'webtoffee-cookie-consent' );
		$consent_txt             = isset( $this->contents['iabDescription']['buttons']['elements']['consent'] ) ? $this->contents['iabDescription']['buttons']['elements']['consent'] : __( 'Consent', 'webtoffee-cookie-consent' );
		$html                    = '<div class="wcc-child-accordion-item">
					<div class="wcc-child-accordion-chevron"><i class="wcc-chevron-right"></i></div>
					<div class="wcc-child-accordion-header-wrapper">
						<button class="wcc-child-accordion-btn" aria-label="' . esc_attr( $name ) . '"
							data-tag="detail-category-title">' . esc_html( $name ) . '</button>
							<div class="wcc-switch-wrapper">';
		if ( $legitimate_interest ) {
			$html .= '<div class="wcc-switch wcc-legitimate-switch-wrapper wcc-switch-sm wcc-switch-separator" data-tag="detail-category-toggle">
									<label for="wcc-' . esc_attr( $purpose_feature_key ) . '-legitimate-interests-checkbox-item-' . esc_attr( $item_key ) . '" class="wcc-legitimate-interest">' . esc_html( $legitimate_interest_txt ) . '</label>
									<input type="checkbox" id="wcc-' . esc_attr( $purpose_feature_key ) . '-legitimate-interests-checkbox-item-' . esc_attr( $item_key ) . '" class=" wcc-' . esc_attr( $purpose_feature_key ) . '-legitimate-interests-checkbox">
								</div>';
		}
		if ( $show_checkbox ) {
			$html .= '<div class="wcc-switch wcc-consent-switch-wrapper wcc-switch-sm" data-tag="detail-category-toggle">
								<label for="wcc-' . esc_attr( $purpose_feature_key ) . '-consents-checkbox-item-' . esc_attr( $item_key ) . '" class="wcc-switch wcc-checkbox-label">' . esc_html( $consent_txt ) . '</label>
								<input type="checkbox" id="wcc-' . esc_attr( $purpose_feature_key ) . '-consents-checkbox-item-' . esc_attr( $item_key ) . '" class="wcc-iab-checkbox wcc-' . esc_attr( $purpose_feature_key ) . '-checkbox">
							</div>';
		}
		$html .= '</div>
					</div>
				</div>
				<div class="wcc-child-accordion-body" aria-hidden="true">
					<div class="wcc-accordion-header-des" data-tag="detail-category-description" style="color:#212121;">
						<p>' . esc_html( $description ) . '</p>
					</div>';
					if ( ! empty( $illustration_html ) ) {
						$html .= '<div class="wcc-accordion-header-des" data-tag="detail-category-description" style="color:#212121;">
							<p>' . wp_kses_post( $illustration_html ) . '</p>
						</div>';
					}
					$html .='<div class="wcc-accordion-header-des">
                    	<label>' . esc_html( $vendor_consent_text, 'webtoffee-cookie-consent' ) . '</label>
                    	<span class="wcc-vendors-seek-count"></span>
                    </div>
				</div>';
		return $html;
	}
	/**
	 * Return google privacy policy description, link HTML
	 *
	 * @return string
	 */
	public function wcc_google_privacy_policy() {
		$html = '';
		if ( ! $this->is_gpp_enabled ) {
			return $html;
		}
		$shortcode_data = wcc_array_search( $this->shortcodes, 'uiTag', 'google-privacy-policy' );

		$content = isset( $shortcode_data['content']['container'] ) ? wp_kses( $shortcode_data['content']['container'], wcc_allowed_html() ) : '';
		if ( '' === $content ) {
			return $html;
		}
		$description = isset( $this->contents['googlePrivacyPolicy']['elements']['description'] ) ? $this->contents['googlePrivacyPolicy']['elements']['description'] : '';
		$link_text   = isset( $this->contents['googlePrivacyPolicy']['elements']['linkText'] ) ? $this->contents['googlePrivacyPolicy']['elements']['linkText'] : '';
		$link_url    = isset( $this->contents['googlePrivacyPolicy']['elements']['linkURL'] ) ? $this->contents['googlePrivacyPolicy']['elements']['linkURL'] : '';

		// Create the link in the desired format
		$link_html = '<a class="wcc-link" href="' . esc_url( $link_url ) . '" target="_blank">' . esc_html( $link_text ) . '</a>';

		$html .= str_replace(
			array(
				'[wcc_google_privacy_desc]',
				'[wcc_google_privacy_url]',
			),
			array(
				esc_html( $description ),
				$link_html,
			),
			$content
		);
		return do_shortcode( $html );
	}
}
