<?php
/**
 * Translation helper functions
 *
 * @link       https://www.webtoffee.com/
 * @since      3.0.0
 * @package    WebToffee\CookieConsent\Lite\Includes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
if ( ! function_exists( 'wcc_default_language' ) ) {

	/**
	 * Check if a request is a rest request
	 *
	 * @return string
	 */
	function wcc_default_language() {
		$settings = get_option( 'wcc_settings' );
		return isset( $settings['languages']['default'] ) ? wcc_sanitize_text( $settings['languages']['default'] ) : 'en';
	}
}
if ( ! function_exists( 'wcc_selected_languages' ) ) {

	/**
	 * Check if a request is a rest request
	 *
	 * @param string $language Language to add temporarily to the existing list.
	 * @return array
	 */
	function wcc_selected_languages( $language = '' ) {
		$settings  = get_option( 'wcc_settings' );
		$languages = isset( $settings['languages']['selected'] ) ? wcc_sanitize_text( $settings['languages']['selected'] ) : array();
		if ( ! in_array( wcc_default_language(), $languages, true ) ) {
			array_push( $languages, wcc_default_language() );
		}
		if ( '' !== $language && ! in_array( $language, $languages, true ) ) {
			array_push( $languages, $language );
		}
		return $languages;
	}
}

if ( ! function_exists( 'wcc_i18n_is_multilingual' ) ) {

	/**
	 * Return true if multilingual plugin is active
	 *
	 * @return boolean
	 */
	function wcc_i18n_is_multilingual() {
		$status = false;

		if ( defined( 'ICL_LANGUAGE_CODE' ) || defined( 'POLYLANG_FILE' ) || class_exists( 'TRP_Translate_Press' ) ) {
			$status = true;
		}
		return $status;
	}
}

if ( ! function_exists( 'wcc_current_language' ) ) {
	/**
	 * Returns the current language code of the site
	 *
	 * @return string
	 */
	function wcc_current_language() {
		$current_language = null;
		if ( wcc_i18n_is_multilingual() ) {
			// If the plugin used is Polylang.
			if ( function_exists( 'pll_current_language' ) ) {

				$current_language = pll_current_language();
				// If current_language is still empty, we have to get the default language.
				if ( empty( $current_language ) ) {
					$current_language = pll_default_language();
				}
			} elseif ( class_exists( 'TRP_Translate_Press' ) ) { // If the plugin used is TranslatePress.
				// phpcs:ignore WordPress.NamingConventions.ValidVariableName.VariableNotSnakeCase
				global $TRP_LANGUAGE;
				$trp = TRP_Translate_Press::get_trp_instance();
				if ( $trp && method_exists( $trp, 'get_component' ) && $trp->get_component( 'url_converter' ) ) {
					// phpcs:ignore WordPress.NamingConventions.ValidVariableName.VariableNotSnakeCase
					$current_language = $TRP_LANGUAGE;
					if ( empty( $current_language ) ) {
						$current_language = $trp->get_component( 'url_converter' )->get_lang_from_url_string( add_query_arg( array() ) );
					}
					$current_language = wcc_trp_map_language( $current_language );
				}
				// phpcs:enable WordPress.NamingConventions.ValidVariableName
			} else { // If the plugin used is WPML.
				$current_language = apply_filters( 'wpml_current_language', null );
			}
			// Fallback if neither WPML, Polylang, nor TranslatePress is used.
			if ( 'all' === $current_language ) {
				$current_language = wcc_default_language();
			}
		} else {
			$current_language = wcc_default_language();
		}
		$map              = wcc_get_lang_map();
		$current_language = isset( $map[ $current_language ] ) ? $map[ $current_language ] : $current_language;
		if ( in_array( $current_language, wcc_selected_languages(), true ) === false ) {
			$current_language = wcc_default_language();
		}
		return apply_filters( 'wcc_current_language', $current_language );
	}
}

if ( ! function_exists( 'wcc_get_lang_map' ) ) {
	/**
	 * Returns the current language code of the site
	 *
	 * @return string
	 */
	function wcc_get_lang_map() {
		$map = array(
			'pt-pt' => 'pt',
		);

		return apply_filters( 'wcc_language_map', $map );
	}
}

if ( ! function_exists( 'wcc_trp_map_language' ) ) {
	/**
	 * Maps TranslatePress language codes to simplified format
	 * Converts codes like 'fr_FR' to 'fr'
	 *
	 * @since 3.2.0
	 * @param string $language_code The language code to map (e.g. 'fr_FR', 'it_IT')
	 * @return string The simplified language code (e.g. 'fr', 'it')
	 */
	function wcc_trp_map_language( $language_code ) {
		if ( empty( $language_code ) ) {
			return '';
		}
		// Split on underscore and return first part
		$parts = explode( '_', $language_code );
		return $parts[0];
	}
}

if ( ! function_exists( 'wcc_i18n_default_language' ) ) {
	/**
	 * Returns the current language code of the site
	 *
	 * @return string
	 */
	function wcc_i18n_default_language() {
		if ( wcc_i18n_is_multilingual() ) {
			if ( function_exists( 'pll_default_language' ) ) {
				$default = pll_default_language();
			} else {
				$null    = null;
				$default = apply_filters( 'wpml_default_language', $null );
			}
		} else {
			$default = wcc_default_language();
		}
		return $default;
	}
}
if ( ! function_exists( 'wcc_i18n_term_by_language' ) ) {
	/**
	 * Returns the current language code of the site
	 *
	 * @param integer $term_id Original term id.
	 * @param string  $language Language code.
	 * @return object
	 */
	function wcc_i18n_term_by_language( $term_id, $language ) {
		$term = false;
		if ( wcc_i18n_is_multilingual() ) {
			if ( function_exists( 'pll_get_term_translations' ) ) {
				$terms = pll_get_term_translations( $term_id );
				if ( isset( $terms[ $language ] ) ) {
					$original_term_id = $terms[ $language ];
					$term             = get_term_by( 'id', $original_term_id, 'cookielawinfo-category' );
				}
			} else {
				if ( function_exists( 'icl_object_id' ) ) {
					global $sitepress;
					if ( $sitepress ) {
						if ( version_compare( ICL_SITEPRESS_VERSION, '3.2.0' ) >= 0 ) {
							$original_term_id = apply_filters( 'wpml_object_id', $term_id, 'category', true, $language );
						} else {
							$original_term_id = icl_object_id( $term_id, 'category', true, $language );
						}
						remove_filter( 'get_term', array( $sitepress, 'get_term_adjust_id' ), 1 );
						$term = get_term_by( 'id', $original_term_id, 'cookielawinfo-category' );
						add_filter( 'get_term', array( $sitepress, 'get_term_adjust_id' ), 1, 1 );
					}
				}
			}
		}
		return $term;
	}
}

if ( ! function_exists( 'wcc_i18n_post_by_language' ) ) {
	/**
	 * Returns the current language code of the site
	 *
	 * @param integer $term_id Original term id.
	 * @param string  $language Language code.
	 * @return object
	 */
	function wcc_i18n_post_by_language( $post_id, $language ) {
		$post = false;
		if ( wcc_i18n_is_multilingual() ) {
			if ( function_exists( 'pll_get_post_translations' ) ) {
				$posts = pll_get_post_translations( $post_id );
				if ( isset( $posts[ $language ] ) ) {
					$original_post_id = $posts[ $language ];
					$post             = get_post( $original_post_id );
				}
			} else {
				if ( function_exists( 'icl_object_id' ) ) {
					$type = apply_filters( 'wpml_element_type', get_post_type( $post_id ) );
					$trid = apply_filters( 'wpml_element_trid', false, $post_id, $type );

					$translations = apply_filters( 'wpml_get_element_translations', array(), $trid, $type );
					if ( isset( $translations[ $language ] ) ) {
						$original_post_id = isset( $translations[ $language ]->element_id ) ? $translations[ $language ]->element_id : false;
						if ( $original_post_id ) {
							$post = get_post( $original_post_id );
						}
					}
				}
			}
		}
		return $post;
	}
}

if ( ! function_exists( 'wcc_wpml_active' ) ) {
	function wcc_wpml_active() {
		return class_exists( 'SitePress' );
	}
}

if ( ! function_exists( 'wcc_pll_active' ) ) {
	function wcc_pll_active() {
		return function_exists( 'pll_current_language' );
	}
}

if ( ! function_exists( 'wcc_i18n_selected_languages' ) ) {
	function wcc_i18n_selected_languages() {
		$languages = array( wcc_i18n_default_language() );
		if ( wcc_i18n_is_multilingual() ) {
			if ( wcc_wpml_active() ) {
				return wcc_i18n_wpml_languages();
			} elseif ( wcc_pll_active() ) {
				return wcc_i18n_pll_languages();
			}
		}
		return $languages;
	}
}

if ( ! function_exists( 'wcc_i18n_pll_languages' ) ) {
	function wcc_i18n_pll_languages() {
		$languages = array();
		if ( function_exists( 'pll_languages_list' ) ) {
			$configured = pll_languages_list();
			if ( empty( $configured ) ) {
				return $languages;
			}
			foreach ( $configured as $language ) {
				$languages[] = $language;
			}
		}
		return $languages;
	}
}
if ( ! function_exists( 'wcc_i18n_wpml_languages' ) ) {
	function wcc_i18n_wpml_languages() {
		$languages  = array();
		$configured = apply_filters( 'wpml_active_languages', null );
		if ( empty( $configured ) ) {
			return $languages;
		}
		foreach ( $configured as $key => $language ) {
			$languages[] = $key;
		}
		return $languages;
	}
}
if ( ! function_exists( 'wcc_i18n_wpml_languages_list' ) ) {
	function wcc_i18n_wpml_languages_list() {
		$languages  = array();
		$configured = apply_filters( 'wpml_active_languages', null );
		if ( empty( $configured ) ) {
			return $languages;
		}
		foreach ( $configured as $key => $language ) {
			$languages[ $key ] = $language;
		}
		return $languages;
	}
}
if ( ! function_exists( 'wcc_i18n_pll_languages_list' ) ) {
	function wcc_i18n_pll_languages_list() {
		$languages = array();
		if ( function_exists( 'pll_languages_list' ) ) {
			//$configured = pll_the_languages( array( 'raw' => 1 ) );
			$languages = pll_languages_list( array( 'fields' => '' ) );
			if ( empty( $configured ) ) {
				return $languages;
			}
			foreach ( $configured as $language ) {
				$languages[ $language['slug'] ] = $language;
			}
		}
		return $languages;
	}
}

if ( ! function_exists( 'wcc_i18n_translate_string' ) ) {
	function wcc_i18n_translate_string( $translatable_string, $key, $language, $context = 'CookieLawInfo-0.9' ) {
		if ( function_exists( 'pll_translate_string' ) ) {
			return pll_translate_string( $translatable_string, $language );
		} else {
			return apply_filters( 'wpml_translate_single_string', $translatable_string, "admin_texts_{$context}", "[{$context}]" . $key, $language );
		}
	}
}

if ( ! function_exists( 'wcc_i18n_term_language' ) ) {
	function wcc_i18n_term_language( $term ) {
		$language = wcc_i18n_default_language();
		if ( wcc_i18n_is_multilingual() ) {
			if ( function_exists( 'pll_get_term_language' ) ) {
				$language = pll_get_term_language( $term );
			}
		}
		return $language;
	}
}
