<?php
/**
 * Utility functions class
 *
 * @link       https://www.webtoffee.com/
 * @since      3.0.0
 *
 * @author     Sarath GP <sarath.gp@mozilor.com>
 * @package    WebToffee\CookieConsent\Lite\Includes
 */

use WebToffee\CookieConsent\Lite\Includes\Filesystem;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
if ( ! function_exists( 'wcc_parse_url' ) ) {
	/**
	 * Return parsed URL
	 *
	 * @param string $url URL string to be parsed.
	 * @return array URL parts.
	 */
	function wcc_parse_url( $url ) {
		return function_exists( 'wp_parse_url' )
			? wp_parse_url( $url )
			: parse_url( $url ); // phpcs:ignore WordPress.WP.AlternativeFunctions.parse_url_parse_url
	}
}
if ( ! function_exists( 'wcc_read_json_file' ) ) {
	/**
	 * Processes a json file from the specified path
	 * and returns an array with its contents, or a void array if none found.
	 *
	 * @since 3.0.0
	 *
	 * @param string $file_path Path to file. Empty if no file.
	 * @return array Contents from json file.
	 */
	function wcc_read_json_file( $file_path = '' ) {
		$config = array();

		$file_system = Filesystem::get_instance();
		$json        = $file_system->get_contents( $file_path );
		if ( ! $json ) {
			return $config;
		}
		$decoded_file        = json_decode(
			$json,
			true
		);
		$json_decoding_error = json_last_error();
		if ( JSON_ERROR_NONE !== $json_decoding_error ) {
			return $config;
		}
		if ( is_array( $decoded_file ) ) {
			$config = $decoded_file;
		}
		return $config;
	}
}

if ( ! function_exists( 'wcc_i18n_date' ) ) {
	/**
	 * Get localized date.
	 *
	 * @param string $date Date in time stamped format.
	 * @return string
	 */
	function wcc_i18n_date( $date = '' ) {
		return date_i18n( 'd/m/Y g:i:s', $date );
	}
}
if ( ! function_exists( 'wcc_is_admin_request' ) ) {
	/**
	 * Get localized date.
	 *
	 * @return boolean
	 */
	function wcc_is_admin_request() {
		return is_admin() && ! wcc_is_ajax_request();
	}
}
if ( ! function_exists( 'wcc_is_ajax_request' ) ) {
	/**
	 * Get localized date.
	 *
	 * @return boolean
	 */
	function wcc_is_ajax_request() {
		if ( function_exists( 'wp_doing_ajax' ) ) {
			return wp_doing_ajax();
		} else {
			return ( defined( 'DOING_AJAX' ) && DOING_AJAX );
		}
	}
}
if ( ! function_exists( 'wcc_is_rest_request' ) ) {

	/**
	 * Check if a request is a rest request
	 *
	 * @return boolean
	 */
	function wcc_is_rest_request() {
		if ( empty( $_SERVER['REQUEST_URI'] ) ) {
			return false;
		}
		$rest_prefix = trailingslashit( rest_get_url_prefix() );
		$request     = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : false;
		if ( ! $request ) {
			return false;
		}
		$is_rest_api_request = ( false !== strpos( $request, $rest_prefix ) );

		return apply_filters( 'wcc_is_rest_api_request', $is_rest_api_request );
	}
}
if ( ! function_exists( 'wcc_is_cloud_request' ) ) {

	/**
	 * Check if a request is a rest request
	 *
	 * @return boolean
	 */
	function wcc_is_cloud_request() {
		return ( defined( 'WCC_CLOUD_REQUEST' ) && WCC_CLOUD_REQUEST );
	}
}
if ( ! function_exists( 'wcc_array_search' ) ) {

	/**
	 * Get settings of element from banner properties by using the tag "data-tag"
	 *
	 * @param array  $input_array Array to be searched.
	 * @param string $key Tag to be used for searching.
	 * @param string $value  Tag name.
	 * @return array
	 */
	function wcc_array_search( $input_array = array(), $key = '', $value = '' ) {

		$results = array();
		if ( is_array( $input_array ) ) {
			if ( isset( $input_array[ $key ] ) && $input_array[ $key ] === $value ) {
				$results = $input_array;
			}
			foreach ( $input_array as $sub_array ) {
				$results = array_merge( $results, wcc_array_search( $sub_array, $key, $value ) );
			}
		}
		return $results;
	}
}
if ( ! function_exists( 'wcc_first_time_install' ) ) {

	/**
	 * Check if the plugin is activated for the first time.
	 *
	 * @return boolean
	 */
	function wcc_first_time_install() {
		return (bool) get_site_transient( '_wcc_first_time_install' ) || (bool) get_option( 'wcc_first_time_activated_plugin' );
	}
}

if ( ! function_exists( 'wcc_is_admin_page' ) ) {

	/**
	 * Check if the plugin is activated for the first time.
	 *
	 * @return boolean
	 */
	function wcc_is_admin_page() {
		if ( ! is_admin() ) {
			return false;
		}
		if ( function_exists( 'get_current_screen' ) && ! empty( get_current_screen() ) ) {
			$screen = get_current_screen();
			$page   = isset( $screen->id ) ? $screen->id : false;
			if ( false !== strpos( $page, 'toplevel_page_cookie-consent' ) ) {
				return true;
			}
			if ( ! empty( $screen->parent_base ) && false !== strpos( $screen->parent_base, 'cookie-consent' ) ) {
				return true;
			}
		} else {
			$page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		}
		return false !== strpos( $page, 'cookie-consent' );
	}
}

if ( ! function_exists( 'wcc_is_front_end_request' ) ) {

	/**
	 * Check if request coming from front-end.
	 *
	 * @return boolean
	 */
	function wcc_is_front_end_request() {
		if ( is_admin() || wcc_is_rest_request() || wcc_is_ajax_request() ) {
			return false;
		}
		return true;
	}
}
if ( ! function_exists( 'wcc_disable_banner' ) ) {

	/**
	 * Check if request coming from front-end.
	 *
	 * @return boolean
	 */
	function wcc_disable_banner() {
		global $wp_customize;
		$options = get_option( 'wcc_settings' );
		// Define the array of specific page URLs on which the banner should be hidden
		$pages_to_hide = isset( $options['hide_banner_on_pages']['selected'] ) ? $options['hide_banner_on_pages']['selected'] : array();
		// Get the current page URI path
		$request_uri = isset( $_SERVER['REQUEST_URI'] ) ? wp_unslash( $_SERVER['REQUEST_URI'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$current_page_path = wp_parse_url( esc_url_raw( $request_uri ), PHP_URL_PATH );

		// Loop through $pages_to_hide and compare paths
		foreach ( $pages_to_hide as $page_url ) {
			$page_path = wp_parse_url( $page_url, PHP_URL_PATH );
			if ( $current_page_path === $page_path ) {
				return true;
			}
		}
		if ( isset( $_GET['et_fb'] ) || isset( $_GET['et_fb'] ) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		|| ( defined( 'ET_FB_ENABLED' ) && ET_FB_ENABLED )
		|| isset( $_GET['elementor-preview'] ) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		|| isset( $_POST['cs_preview_state'] ) // phpcs:ignore WordPress.Security.NonceVerification.Missing
		|| isset( $wp_customize ) ) {
			return true;
		}
		/* Allow filtering the banner disable status
		 * @since 3.4.3
		 * @param boolean $should_disable_banner Whether to disable the banner.
		 * @param string $current_page_path The current page path.
		 * @return boolean
		*/
		return apply_filters( 'wcc_should_disable_banner', false, $current_page_path );
	}
}
if ( ! function_exists( 'wcc_missing_tables' ) ) {

	/**
	 * Check if request coming from front-end.
	 *
	 * @return array
	 */
	function wcc_missing_tables() {
		return get_option( 'wcc_missing_tables', array() );
	}
}
if ( ! function_exists( 'wcc_convert_seconds_to_days' ) ) {

	/**
	 * return days from seconds
	 *
	 * @return string
	 */
	function wcc_convert_seconds_to_days( $sec ) {
		$days = floor( $sec / 86400 );
		return $days;
	}
}