<?php
/**
 * REST API for availability rules (v1).
 *
 * Handles requests to the `/availabilities` endpoint. This controller
 * provides read-only access to availability rules via a flexible `filter`
 * array that is passed to the availability data store.
 *
 * Endpoint
 * - `GET /{namespace}/availabilities` (defaults to v1 namespace)
 *
 * Permissions
 * - Public read (`permission_callback` = `__return_true`).
 *
 * Query Parameters
 * - `filter` (array of objects): Low-level constraints executed by the data store.
 *   - Item shape: `{ key: string, compare: string, value: scalar|array }`
 *   - Common keys: `id`, `kind`, `kind_id`, `event_id`, `title`, `range_type`,
 *     `from_date`, `to_date`, `from_range`, `to_range`, `appointable`,
 *     `priority`, `qty`, `ordering`, `rrule`, `date_created`, `date_modified`.
 *   - Computed keys: `start_date`, `end_date` for date-window filtering.
 *   - Comparators (examples): `=`, `!=`, `>`, `>=`, `<`, `<=`, `IN`, `NOT IN`,
 *     `BETWEEN`, `NOT BETWEEN`, `LIKE`, `NOT LIKE`, `REGEXP`, `NOT REGEXP`,
 *     `IS NULL`, `IS NOT NULL`.
 *
 * Response
 * - Array of rule objects with fields like `id`, `kind`, `kind_id`, `range_type`,
 *   `from_date`, `to_date`, `from_range`, `to_range`, `appointable`, `priority`,
 *   `qty`, `ordering`, `rrule`, `date_created`, `date_modified`.
 *
 * Notes
 * - CRUD support is available in v2 (`/wc-appointments/v2/availabilities`).
 * - For time-window reads at scale, prefer the v2 Index controller.
 *
 * @package WooCommerce\Appointments\Rest\Controller
 */

/**
 * REST API Products controller class.
 */
class WC_Appointments_REST_Availabilities_Controller extends WC_REST_Controller {

	use WC_Appointments_Rest_Permission_Check;

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = WC_Appointments_REST_API::V1_NAMESPACE;

	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'availabilities';

	/**
	 * Register the route for availabilities slots.
	 */
	public function register_routes(): void {
		register_rest_route(
		    $this->namespace,
		    '/' . $this->rest_base,
		    [
				[
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => [ $this, 'get_items' ],
					'permission_callback' => '__return_true',
				],
			],
		);
	}

	/**
	 * List availability rules.
	 *
	 * Accepts a `filter` array passed directly to the availability
	 * data store. See class doc for allowed comparators and keys.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function get_items( $request ): WP_REST_Response {
		$prepared_args = [];

		// Set up arguments if passed.
		if ( ! empty( $request['filter'] ) ) {
			$prepared_args = $request['filter'];
			#error_log( var_export( $prepared_args, true ) );
		}

		// Example.
		#filter[0][key]=kind&filter[0][compare]=IN&filter[0][value][]=availability%23global&filter[0][value][]=availability%23staff

		/**
		 * Filter arguments, before passing to WC_Appointments_Availability_Data_Store, when querying availabilites via the REST API.
		 *
		 * @param array           $prepared_args Array of arguments for WC_Appointments_Availability_Data_Store.
		 * @param WP_REST_Request $request       The current request.
		 */
		$prepared_args = apply_filters( 'woocommerce_rest_availabilities_query', $prepared_args, $request );

		$availabilities = WC_Data_Store::load( 'appointments-availability' )->get_all_as_array( $prepared_args );

		return rest_ensure_response( $availabilities );
	}

	/**
	 * JSON Schema placeholder.
	 *
	 * v1 controller does not expose an item schema; v2 does.
	 * Left as parent for compatibility.
	 */
	public function get_item_schema(): array {
		return parent::get_item_schema();
	}
}
