<?php
/**
 * WooCommerce Appointments Blocks Integration.
 */

namespace WooCommerce\Appointments\Blocks;

defined( 'ABSPATH' ) || exit;

use Automattic\WooCommerce\Blocks\Assets\Api as AssetApi;
use Automattic\WooCommerce\Blocks\Package;
use Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry;
use Automattic\WooCommerce\Blocks\Registry\Container;
use Automattic\WooCommerce\StoreApi\Schemas\ExtendSchema;
use Automattic\WooCommerce\StoreApi\StoreApi;

/**
 * This class is responsible for integrating a new payment method when using WooCommerce Blocks.
 */
class WC_Appointments_Blocks {

	/**
	 * Constructor
	 */
	public function __construct() {
		// Include needed files.
		$this->includes();

		// Add woocommerce blocks support.
		$this->add_woocommerce_block_support();

		/**
		 * This function enables block based product list components to change
		 * the Add to cart button into a link to the product detail page.
		 */
		add_filter(
		    'woocommerce_product_has_options',
		    function ( $has_options, $product ) {
				if ( 'appointment' === $product->get_type() ) {
					return true;
				}
				return $has_options;
			},
		    10,
		    2,
		);
	}

	/**
	 * Include needed files
	 */
	public function includes(): void {
		include_once __DIR__ . '/class-wc-appointments-block-gateway.php';
		// Availability filter block.
		include_once __DIR__ . '/class-wc-appointments-block-availability.php';
		// Appointment form block.
		include_once __DIR__ . '/class-wc-appointments-block-appointment-form.php';
	}

	/**
	 * Add payment method block support
	 */
	public function add_woocommerce_block_support(): void {
		if ( class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
			// Register payment method integrations.
			add_action( 'woocommerce_blocks_payment_method_type_registration', [ $this, 'register_payment_method_integrations' ] );
			$this->register_payment_methods();
			$this->register_payment_requirements();
		}
	}

	/**
	 * Register payment method integration
	 *
	 * @param PaymentMethodRegistry $payment_method_registry Payment method registry object.
	 */
	public function register_payment_method_integrations( PaymentMethodRegistry $payment_method_registry ): void {
		$payment_method_registry->register(
		    Package::container()->get( Appointments_Block_Gateway::class ),
		);
	}

	/**
	 * Register payment method
	 *
	 * @return void
	 */
	protected function register_payment_methods() {
		$container = Package::container();

		$container->register(
		    Appointments_Block_Gateway::class,
		    function( Container $container ): \WooCommerce\Appointments\Blocks\Appointments_Block_Gateway {
				$asset_api = $container->get( AssetApi::class );
				return new Appointments_Block_Gateway( $asset_api );
			},
		);
	}

	/**
     * Register the payment requirements for blocks
     */
    public function register_payment_requirements(): void {
		if ( class_exists( 'Automattic\WooCommerce\StoreApi\StoreApi' ) ) {
			// Get extend class from the container.
			$extend = StoreApi::container()->get( ExtendSchema::class );

			// Add payment requirements for appointment availability carts.
			$extend->register_payment_requirements(
			    [
					'data_callback' => [ $this, 'add_appointment_availability_payment_requirement' ],
				],
			);
		}
	}

	/**
     * Adds appointment availability payment requirement for carts that contain a product that requires it.
     */
    public function add_appointment_availability_payment_requirement(): array {
		if ( wc_appointment_cart_requires_confirmation() ) {
			return [ 'appointment_availability' ];
		}
		return [];
	}
}

new WC_Appointments_Blocks();
