<?php
// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Constants for WooCommerce Appointments.
 *
 * Provides named constants for magic values used throughout the plugin,
 * making the codebase more maintainable and AI-friendly.
 *
 * @since 5.0.0
 */
class WC_Appointments_Constants {

	/**
	 * Duration unit constants.
	 */
	public const DURATION_MINUTE = 'minute';
	public const DURATION_HOUR   = 'hour';
	public const DURATION_DAY    = 'day';
	public const DURATION_NIGHT  = 'night';
	public const DURATION_MONTH  = 'month';

	/**
	 * Time conversion constants (seconds).
	 */
	public const SECONDS_PER_MINUTE = 60;
	public const SECONDS_PER_HOUR   = 3600;
	public const SECONDS_PER_DAY    = 86400;
	public const SECONDS_PER_WEEK   = 604800;

	/**
	 * Minutes conversion constants.
	 */
	public const MINUTES_PER_HOUR = 60;
	public const MINUTES_PER_DAY  = 1440;
	public const MINUTES_PER_WEEK = 10080;
	public const MINUTES_PER_MONTH = 43200; // Approximate (30 days)

	/**
	 * Appointment status constants.
	 */
	public const STATUS_UNPAID               = 'unpaid';
	public const STATUS_PENDING_CONFIRMATION = 'pending-confirmation';
	public const STATUS_CONFIRMED            = 'confirmed';
	public const STATUS_PAID                 = 'paid';
	public const STATUS_CANCELLED            = 'cancelled';
	public const STATUS_COMPLETE             = 'complete';
	public const STATUS_IN_CART              = 'in-cart';
	public const STATUS_WAS_IN_CART          = 'was-in-cart';

	/**
	 * Customer status constants.
	 */
	public const CUSTOMER_STATUS_EXPECTED = 'expected';
	public const CUSTOMER_STATUS_ARRIVED  = 'arrived';
	public const CUSTOMER_STATUS_NO_SHOW  = 'no-show';

	/**
	 * Staff assignment type constants.
	 */
	public const STAFF_ASSIGNMENT_CUSTOMER = 'customer';
	public const STAFF_ASSIGNMENT_AUTOMATIC = 'automatic';
	public const STAFF_ASSIGNMENT_ALL = 'all';

	/**
	 * Get all valid duration units.
	 *
	 * @since 5.0.0
	 *
	 * @return array<string> Array of valid duration unit strings.
	 */
	public static function get_duration_units(): array {
		return [
			self::DURATION_MINUTE,
			self::DURATION_HOUR,
			self::DURATION_DAY,
			self::DURATION_NIGHT,
			self::DURATION_MONTH,
		];
	}

	/**
	 * Get all valid appointment statuses.
	 *
	 * @since 5.0.0
	 *
	 * @return array<string> Array of valid appointment status strings.
	 */
	public static function get_appointment_statuses(): array {
		return [
			self::STATUS_UNPAID,
			self::STATUS_PENDING_CONFIRMATION,
			self::STATUS_CONFIRMED,
			self::STATUS_PAID,
			self::STATUS_CANCELLED,
			self::STATUS_COMPLETE,
			self::STATUS_IN_CART,
			self::STATUS_WAS_IN_CART,
		];
	}

	/**
	 * Get all valid customer statuses.
	 *
	 * @since 5.0.0
	 *
	 * @return array<string> Array of valid customer status strings.
	 */
	public static function get_customer_statuses(): array {
		return [
			self::CUSTOMER_STATUS_EXPECTED,
			self::CUSTOMER_STATUS_ARRIVED,
			self::CUSTOMER_STATUS_NO_SHOW,
		];
	}

	/**
	 * Convert duration unit to seconds.
	 *
	 * @since 5.0.0
	 *
	 * @param string $unit Duration unit (minute, hour, day, month).
	 * @param int    $value Optional. Value to multiply. Default 1.
	 *
	 * @return int Seconds equivalent, or 0 if unit is invalid.
	 */
	public static function duration_unit_to_seconds( string $unit, int $value = 1 ): int {
		switch ( $unit ) {
			case self::DURATION_MINUTE:
				return $value * self::SECONDS_PER_MINUTE;
			case self::DURATION_HOUR:
				return $value * self::SECONDS_PER_HOUR;
			case self::DURATION_DAY:
				return $value * self::SECONDS_PER_DAY;
			case self::DURATION_MONTH:
				// Approximate: 30 days
				return $value * ( self::SECONDS_PER_DAY * 30 );
			default:
				return 0;
		}
	}

	/**
	 * Convert duration unit to minutes.
	 *
	 * @since 5.0.0
	 *
	 * @param string $unit Duration unit (minute, hour, day, month).
	 * @param int    $value Optional. Value to multiply. Default 1.
	 *
	 * @return int Minutes equivalent, or 0 if unit is invalid.
	 */
	public static function duration_unit_to_minutes( string $unit, int $value = 1 ): int {
		switch ( $unit ) {
			case self::DURATION_MINUTE:
				return $value;
			case self::DURATION_HOUR:
				return $value * self::MINUTES_PER_HOUR;
			case self::DURATION_DAY:
				return $value * self::MINUTES_PER_DAY;
			case self::DURATION_MONTH:
				return $value * self::MINUTES_PER_MONTH;
			default:
				return 0;
		}
	}
}


















