<?php
/**
 * WooCommerce PayPal Payments Integration
 *
 * Handles compatibility with the official WooCommerce PayPal Payments plugin.
 * Disables PayPal smart buttons on appointment product pages because appointments
 * require specific form data (date, time, staff, etc.) that smart buttons bypass.
 *
 * @package WooCommerce Appointments
 * @since 5.1.0
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * PayPal Payments Integration class.
 *
 * @since 5.1.0
 */
class WC_Appointments_Integration_PayPal_Payments implements WC_Appointments_Integration_Interface {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->add_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @return void
	 */
	public function add_hooks(): void {
		// Disable PayPal smart buttons (Pay Now buttons) on appointment product pages.
		// These buttons bypass the add-to-cart form which is required for appointments.
		add_filter( 'woocommerce_paypal_payments_product_supports_payment_request_button', [ $this, 'disable_smart_buttons_for_appointments' ], 10, 2 );

		// Also handle the mini cart button context.
		add_filter( 'woocommerce_paypal_payments_checkout_button_renderer_hook', [ $this, 'maybe_disable_mini_cart_button' ], 10, 1 );
	}

	/**
	 * Disable PayPal smart buttons on appointment product pages.
	 *
	 * PayPal's "Pay Now Experience" smart buttons bypass the normal WooCommerce
	 * add-to-cart form submission. For appointment products, this is problematic
	 * because appointments require specific form data (date, time, staff selection,
	 * duration, etc.) that must be submitted through the form.
	 *
	 * When smart buttons are used, the appointment data is never collected,
	 * resulting in items appearing in the mini cart but not actually being
	 * added to the WooCommerce cart session properly.
	 *
	 * @since 5.1.0
	 *
	 * @param bool       $is_supported Whether the product supports smart buttons.
	 * @param WC_Product $product      The product object.
	 * @return bool False if the product is an appointment product, otherwise the original value.
	 */
	public function disable_smart_buttons_for_appointments( $is_supported, $product ): bool {
		if ( ! $product instanceof WC_Product ) {
			return $is_supported;
		}

		// Disable smart buttons for appointment products.
		if ( is_wc_appointment_product( $product ) ) {
			return false;
		}

		return $is_supported;
	}

	/**
	 * Maybe disable the mini cart PayPal button when cart contains appointments.
	 *
	 * If the cart contains appointment products that haven't been properly
	 * validated through the form, we should be cautious about express checkout.
	 *
	 * @since 5.1.0
	 *
	 * @param string $hook The hook location for the button.
	 * @return string The hook location, potentially modified.
	 */
	public function maybe_disable_mini_cart_button( $hook ): string {
		// This filter can be used to modify button placement.
		// For now, we rely on the product-level filter which should handle most cases.
		return $hook;
	}
}
