<?php
// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * WC_Appointments_Export_Manager Class.
 *
 * Manages appointment exporters and provides a unified API for exporting appointments
 * in various formats (CSV, ICS, etc.).
 *
 * @package WooCommerce\Appointments\Admin
 * @since 5.0.0
 */
class WC_Appointments_Export_Manager {

	/**
	 * Array of registered exporters.
	 *
	 * @var array<string, WC_Appointments_Exporter_Interface>
	 */
	private array $exporters = [];

	/**
	 * Register an exporter.
	 *
	 * @param WC_Appointments_Exporter_Interface $exporter Exporter instance.
	 * @return void
	 */
	public function register_exporter( WC_Appointments_Exporter_Interface $exporter ): void {
		$this->exporters[ $exporter->get_format() ] = $exporter;
	}

	/**
	 * Export appointments using the specified format.
	 *
	 * @param string $format Format identifier (e.g., 'csv', 'ics').
	 * @param array  $appointments Array of WC_Appointment objects.
	 * @param array  $args Optional export arguments.
	 * @return string|false File path on success, false on failure.
	 */
	public function export_appointments( string $format, array $appointments, array $args = [] ): string|false {
		if ( ! isset( $this->exporters[ $format ] ) ) {
			return false;
		}

		return $this->exporters[ $format ]->export( $appointments, $args );
	}

	/**
	 * Get all available export formats.
	 *
	 * @return array<string> Array of format identifiers.
	 */
	public function get_available_formats(): array {
		return array_keys( $this->exporters );
	}

	/**
	 * Get exporter instance for a specific format.
	 *
	 * @param string $format Format identifier.
	 * @return WC_Appointments_Exporter_Interface|null Exporter instance or null if not found.
	 */
	public function get_exporter( string $format ): ?WC_Appointments_Exporter_Interface {
		return $this->exporters[ $format ] ?? null;
	}

	/**
	 * Check if a format is available.
	 *
	 * @param string $format Format identifier.
	 * @return bool True if format is available, false otherwise.
	 */
	public function is_format_available( string $format ): bool {
		return isset( $this->exporters[ $format ] );
	}
}
