<?php
/**
 * Appointment Form Block for WooCommerce Appointments.
 */

defined( 'ABSPATH' ) || exit;

class WC_Appointments_Block_Appointment_Form {
	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'init', [ $this, 'register_block' ], 10, 0 );
		add_action( 'init', [ $this, 'register_assets' ], 9, 0 );
	}

	/**
	 * Register block and server-side renderer.
	 */
	public function register_block(): void {
		if ( function_exists( 'register_block_type' ) ) {
			register_block_type(
			    'woocommerce-appointments/appointment-form',
			    [
					'render_callback' => [ $this, 'render_block' ],
					'editor_script'   => 'wc-appointments-appointment-form-block-editor',
					'editor_style'    => 'wc-appointments-appointment-form-block-editor-style',
				],
			);
		}
	}

	/**
	 * Register and localize assets required by the block editor.
	 */
	public function register_assets(): void {
		if (defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG) {
        }

		// Editor script for block registration in Gutenberg.
		wp_register_script(
		    'wc-appointments-appointment-form-block-editor',
		    WC_APPOINTMENTS_PLUGIN_URL . '/assets/js/blocks/appointments-block-appointment-form.js',
		    [ 'wp-blocks', 'wp-element', 'wp-i18n', 'wp-block-editor', 'wp-components', 'wp-api-fetch' ],
		    WC_APPOINTMENTS_VERSION,
		    true,
		);

		// Ensure translations for editor strings.
		if ( function_exists( 'wp_set_script_translations' ) ) {
			wp_set_script_translations(
			    'wc-appointments-appointment-form-block-editor',
			    'woocommerce-appointments',
			    WC_APPOINTMENTS_PLUGIN_PATH . '/languages',
			);
		}

		// Provide locale data directly for JS i18n as a fallback when JSON catalogs are missing.
		$locale = function_exists( 'determine_locale' ) ? determine_locale() : get_locale();
		$wc_appointments_form_locale_data = [
			'' => [
				'domain' => 'woocommerce-appointments',
				'lang'   => $locale,
			],
			'Appointment Form' => [ __( 'Appointment Form', 'woocommerce-appointments' ) ],
			'Render the appointment form for a product.' => [ __( 'Render the appointment form for a product.', 'woocommerce-appointments' ) ],
			'Product' => [ __( 'Product', 'woocommerce-appointments' ) ],
			'Search Products' => [ __( 'Search Products', 'woocommerce-appointments' ) ],
			'Select a product' => [ __( 'Select a product', 'woocommerce-appointments' ) ],
			'Loading…' => [ __( 'Loading…', 'woocommerce-appointments' ) ],
			'Display Options' => [ __( 'Display Options', 'woocommerce-appointments' ) ],
			'Behavior' => [ __( 'Behavior', 'woocommerce-appointments' ) ],
			'Show Title' => [ __( 'Show Title', 'woocommerce-appointments' ) ],
			'Show Rating' => [ __( 'Show Rating', 'woocommerce-appointments' ) ],
			'Show Price' => [ __( 'Show Price', 'woocommerce-appointments' ) ],
			'Show Excerpt' => [ __( 'Show Excerpt', 'woocommerce-appointments' ) ],
			'Show Meta' => [ __( 'Show Meta', 'woocommerce-appointments' ) ],
			'Show Sharing' => [ __( 'Show Sharing', 'woocommerce-appointments' ) ],
			'Auto-select Availability' => [ __( 'Auto-select Availability', 'woocommerce-appointments' ) ],
			'Customer Timezones' => [ __( 'Customer Timezones', 'woocommerce-appointments' ) ],
			'Default' => [ __( 'Default', 'woocommerce-appointments' ) ],
			'Enable' => [ __( 'Enable', 'woocommerce-appointments' ) ],
			'Disable' => [ __( 'Disable', 'woocommerce-appointments' ) ],
			// In-block selector strings
			'Search for a product to use with the appointment form.' => [ __( 'Search for a product to use with the appointment form.', 'woocommerce-appointments' ) ],
			'No results found.' => [ __( 'No results found.', 'woocommerce-appointments' ) ],
			'Start typing to search.' => [ __( 'Start typing to search.', 'woocommerce-appointments' ) ],
			'Replace product' => [ __( 'Replace product', 'woocommerce-appointments' ) ],
			'Edit product' => [ __( 'Edit product', 'woocommerce-appointments' ) ],
			'View product' => [ __( 'View product', 'woocommerce-appointments' ) ],
			'Selected product not found.' => [ __( 'Selected product not found.', 'woocommerce-appointments' ) ],
			// Inline preview labels
			'Calendar Preview' => [ __( 'Calendar Preview', 'woocommerce-appointments' ) ],
			'Price' => [ __( 'Price', 'woocommerce-appointments' ) ],
			'Rating' => [ __( 'Rating', 'woocommerce-appointments' ) ],
			'Product ID' => [ __( 'Product ID', 'woocommerce-appointments' ) ],
			// Add inline sharing and add-to-cart labels
			'Share' => [ __( 'Share', 'woocommerce-appointments' ) ],
			'Quantity' => [ __( 'Quantity', 'woocommerce-appointments' ) ],
			'Add to cart' => [ __( 'Add to cart', 'woocommerce-appointments' ) ],
		];
		wp_localize_script(
		    'wc-appointments-appointment-form-block-editor',
		    'wcAppointmentsAppointmentFormLocaleData',
		    $wc_appointments_form_locale_data,
		);

		// Editor style for block polish in Gutenberg.
		wp_register_style(
		    'wc-appointments-appointment-form-block-editor-style',
		    WC_APPOINTMENTS_PLUGIN_URL . '/assets/css/blocks/appointments-block-appointment-form-editor.css',
		    [],
		    WC_APPOINTMENTS_VERSION,
		);
	}

	/**
	 * Server-side render callback.
	 *
	 * Outputs the appointment form using the shortcode handler.
	 */
	public function render_block( array $attributes = [], $content = '' ): string {
		$atts = [];

		// Map selected product ID.
		if ( ! empty( $attributes['productId'] ) ) {
			$atts['id'] = absint( $attributes['productId'] );
		}

		// Toggles for product elements.
		$atts['show_title']   = isset( $attributes['showTitle'] ) ? ( $attributes['showTitle'] ? 1 : 0 ) : 1;
		$atts['show_rating']  = isset( $attributes['showRating'] ) ? ( $attributes['showRating'] ? 1 : 0 ) : 1;
		$atts['show_price']   = isset( $attributes['showPrice'] ) ? ( $attributes['showPrice'] ? 1 : 0 ) : 1;
		$atts['show_excerpt'] = isset( $attributes['showExcerpt'] ) ? ( $attributes['showExcerpt'] ? 1 : 0 ) : 1;
		$atts['show_meta']    = isset( $attributes['showMeta'] ) ? ( $attributes['showMeta'] ? 1 : 0 ) : 1;
		$atts['show_sharing'] = isset( $attributes['showSharing'] ) ? ( $attributes['showSharing'] ? 1 : 0 ) : 1;

		// Tri-state options: default (null), enable ('yes'), disable ('no').
		if ( isset( $attributes['availabilityAutoselectMode'] ) ) {
			if ( 'enable' === $attributes['availabilityAutoselectMode'] ) {
				$atts['availability_autoselect'] = 'yes';
			} elseif ( 'disable' === $attributes['availabilityAutoselectMode'] ) {
				$atts['availability_autoselect'] = 'no';
			} else {
				$atts['availability_autoselect'] = null;
			}
		}
		if ( isset( $attributes['customerTimezonesMode'] ) ) {
			if ( 'enable' === $attributes['customerTimezonesMode'] ) {
				$atts['customer_timezones'] = 'yes';
			} elseif ( 'disable' === $attributes['customerTimezonesMode'] ) {
				$atts['customer_timezones'] = 'no';
			} else {
				$atts['customer_timezones'] = null;
			}
		}

		// Delegate rendering to the shortcode handler to maintain consistency.
		if ( class_exists( 'WC_Appointments_Shortcodes' ) && method_exists( 'WC_Appointments_Shortcodes', 'appointment_form' ) ) {
			return WC_Appointments_Shortcodes::appointment_form( $atts );
		}

		return '';
	}
}

new WC_Appointments_Block_Appointment_Form();