<?php
// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Data object representing a single cached availability/appointment occurrence.
 *
 * @since 5.0.0
 */
class WC_Appointments_Availability_Cache extends WC_Data {

	/**
	 * Object type and data store.
	 */
	protected $object_type     = 'appointments_availability_cache';
	protected $data_store_name = 'appointments-availability-cache';
	protected $cache_group     = 'wc-appointments-availability-cache';

	/**
	 * Data array (matches cache table columns).
	 *
	 * @var array
	 */
	protected $data = [
		'source'       => '',   // 'availability' | 'appointment'
		'source_id'    => 0,
		'product_id'   => 0,
		'staff_id'     => 0,
		'scope'        => 'global', // 'global' | 'product' | 'staff'
		'appointable'  => 'yes',    // 'yes' | 'no'
		'priority'     => 10,
		'ordering'     => 0,
		'qty'          => 0,
		'start_ts'     => 0,
		'end_ts'       => 0,
		'range_type'   => '',
		'rule_kind'    => '',
		'status'       => '', // 'unpaid' | 'paid' | 'expired' ...
		'date_created' => '',
		'date_modified'=> '',
	];

	public function __construct( $id = 0 ) {
		parent::__construct( $id );
		$this->data_store = WC_Data_Store::load( $this->data_store_name );

		if ( $this->get_id() > 0 ) {
			$this->data_store->read( $this );
		} else {
			$this->set_object_read( true );
		}
	}

	// Generic getters.

	/**
	 * Get source type.
	 *
	 * @param  string $context Context: view or edit.
	 * @return string          Source type (e.g. 'availability', 'appointment').
	 */
	public function get_source( $context = 'view' ): string      { return $this->get_prop( 'source', $context ); }

	/**
	 * Get source ID.
	 *
	 * @param  string $context Context: view or edit.
	 * @return int             Source ID.
	 */
	public function get_source_id( $context = 'view' ): int   { return (int) $this->get_prop( 'source_id', $context ); }

	/**
	 * Get product ID.
	 *
	 * @param  string $context Context: view or edit.
	 * @return int             Product ID.
	 */
	public function get_product_id( $context = 'view' ): int  { return (int) $this->get_prop( 'product_id', $context ); }

	/**
	 * Get staff ID.
	 *
	 * @param  string $context Context: view or edit.
	 * @return int             Staff ID.
	 */
	public function get_staff_id( $context = 'view' ): int    { return (int) $this->get_prop( 'staff_id', $context ); }

	/**
	 * Get scope.
	 *
	 * @param  string $context Context: view or edit.
	 * @return string          Scope (e.g. 'global', 'product', 'staff').
	 */
	public function get_scope( $context = 'view' ): string       { return $this->get_prop( 'scope', $context ); }

	/**
	 * Get appointable status.
	 *
	 * @param  string $context Context: view or edit.
	 * @return string          Appointable status ('yes' or 'no').
	 */
	public function get_appointable( $context = 'view' ): string { return $this->get_prop( 'appointable', $context ); }

	/**
	 * Get priority.
	 *
	 * @param  string $context Context: view or edit.
	 * @return int             Priority level.
	 */
	public function get_priority( $context = 'view' ): int    { return (int) $this->get_prop( 'priority', $context ); }

	/**
	 * Get ordering.
	 *
	 * @param  string $context Context: view or edit.
	 * @return int             Ordering index.
	 */
	public function get_ordering( $context = 'view' ): int    { return (int) $this->get_prop( 'ordering', $context ); }

	/**
	 * Get quantity.
	 *
	 * @param  string $context Context: view or edit.
	 * @return int             Quantity available.
	 */
	public function get_qty( $context = 'view' ): int         { return (int) $this->get_prop( 'qty', $context ); }

	/**
	 * Get start timestamp.
	 *
	 * @param  string $context Context: view or edit.
	 * @return int             Start timestamp.
	 */
	public function get_start_ts( $context = 'view' ): int    { return (int) $this->get_prop( 'start_ts', $context ); }

	/**
	 * Get end timestamp.
	 *
	 * @param  string $context Context: view or edit.
	 * @return int             End timestamp.
	 */
	public function get_end_ts( $context = 'view' ): int      { return (int) $this->get_prop( 'end_ts', $context ); }

	/**
	 * Get range type.
	 *
	 * @param  string $context Context: view or edit.
	 * @return string          Range type description.
	 */
	public function get_range_type( $context = 'view' ): string  { return $this->get_prop( 'range_type', $context ); }

	/**
	 * Get rule kind.
	 *
	 * @param  string $context Context: view or edit.
	 * @return string          Rule kind identifier.
	 */
	public function get_rule_kind( $context = 'view' ): string   { return $this->get_prop( 'rule_kind', $context ); }

	/**
	 * Get status.
	 *
	 * @param  string $context Context: view or edit.
	 * @return string          Status string.
	 */
	public function get_status( $context = 'view' ): string      { return $this->get_prop( 'status', $context ); }

	/**
	 * Get date created.
	 *
	 * @param  string $context Context: view or edit.
	 * @return WC_DateTime|null Date created object.
	 */
	public function get_date_created( $context = 'view' ): ?WC_DateTime{ return $this->get_prop( 'date_created', $context ); }

	/**
	 * Get date modified.
	 *
	 * @param  string $context Context: view or edit.
	 * @return WC_DateTime|null Date modified object.
	 */
	public function get_date_modified( $context = 'view' ): ?WC_DateTime{ return $this->get_prop( 'date_modified', $context ); }

	// Setters.

	/**
	 * Set source type.
	 *
	 * @param string $v Source type.
	 */
	public function set_source( $v ): void       { $this->set_prop( 'source', $v ); }

	/**
	 * Set source ID.
	 *
	 * @param int $v Source ID.
	 */
	public function set_source_id( $v ): void    { $this->set_prop( 'source_id', (int) $v ); }

	/**
	 * Set product ID.
	 *
	 * @param int $v Product ID.
	 */
	public function set_product_id( $v ): void   { $this->set_prop( 'product_id', (int) $v ); }

	/**
	 * Set staff ID.
	 *
	 * @param int $v Staff ID.
	 */
	public function set_staff_id( $v ): void     { $this->set_prop( 'staff_id', (int) $v ); }

	/**
	 * Set scope.
	 *
	 * @param string $v Scope.
	 */
	public function set_scope( $v ): void        { $this->set_prop( 'scope', $v ); }

	/**
	 * Set appointable status.
	 *
	 * @param string $v Appointable status.
	 */
	public function set_appointable( $v ): void  { $this->set_prop( 'appointable', $v ); }

	/**
	 * Set priority.
	 *
	 * @param int $v Priority.
	 */
	public function set_priority( $v ): void     { $this->set_prop( 'priority', (int) $v ); }

	/**
	 * Set ordering.
	 *
	 * @param int $v Ordering.
	 */
	public function set_ordering( $v ): void     { $this->set_prop( 'ordering', (int) $v ); }

	/**
	 * Set quantity.
	 *
	 * @param int $v Quantity.
	 */
	public function set_qty( $v ): void          { $this->set_prop( 'qty', (int) $v ); }

	/**
	 * Set start timestamp.
	 *
	 * @param int $v Start timestamp.
	 */
	public function set_start_ts( $v ): void     { $this->set_prop( 'start_ts', (int) $v ); }

	/**
	 * Set end timestamp.
	 *
	 * @param int $v End timestamp.
	 */
	public function set_end_ts( $v ): void       { $this->set_prop( 'end_ts', (int) $v ); }

	/**
	 * Set range type.
	 *
	 * @param string $v Range type.
	 */
	public function set_range_type( $v ): void   { $this->set_prop( 'range_type', $v ); }

	/**
	 * Set rule kind.
	 *
	 * @param string $v Rule kind.
	 */
	public function set_rule_kind( $v ): void    { $this->set_prop( 'rule_kind', $v ); }

	/**
	 * Set status.
	 *
	 * @param string $v Status.
	 */
	public function set_status( $v ): void       { $this->set_prop( 'status', $v ); }

	/**
	 * Set date created.
	 *
	 * @param string|int|null $v Date created.
	 */
	public function set_date_created( $v ): void { $this->set_prop( 'date_created', $v ); }

	/**
	 * Set date modified.
	 *
	 * @param string|int|null $v Date modified.
	 */
	public function set_date_modified( $v ): void{ $this->set_prop( 'date_modified', $v ); }
}
