/**
 * WordPress dependencies
 */
import {
	store,
	getContext,
	getElement,
	getConfig,
	withSyncEvent,
} from '@wordpress/interactivity';

export type CalendarDay = {
	key: string;
	dayNumber: number;
	dateString: string;
	isCurrentMonth: boolean;
};

export type BookingInteractiveCalendarStore = {
	state: {
		viewMonthName: string;
		prevMonthLabel: string;
		isPreviousMonthDisabled: boolean;
		nextMonthLabel: string;
		isNextMonthDisabled: boolean;
		calendarDays: CalendarDay[];
		calendarKey: string;
	};
	actions: {
		navigateToPreviousMonth: () => void;
		navigateToNextMonth: () => void;
		onTouchStart: ( event: TouchEvent ) => void;
		onTouchMove: ( event: TouchEvent ) => void;
		onTouchEnd: () => void;
	};
};

export type Context = {
	monthNames: string[];
	weekStartsOn: number;
	touchStartX: number;
	touchCurrentX: number;
	isDragging: boolean;
};

export type BookingInteractiveCalendarContext = {
	selectedDate: string;
	viewMonth: number;
	viewYear: number;
};

const formatDateString = ( date: Date ) => {
	return `${ date.getFullYear() }-${ String( date.getMonth() + 1 ).padStart(
		2,
		'0'
	) }-${ String( date.getDate() ).padStart( 2, '0' ) }`;
};

const formatMonthString = ( date: Date ) => {
	return `${ date.getFullYear() }-${ String( date.getMonth() + 1 ).padStart(
		2,
		'0'
	) }`;
};

/**
 * Get the previous month and year.
 *
 * @param month The month.
 * @param year  The year.
 * @return The previous month and year.
 */
const getPreviousMonth = ( month: number, year: number ) => {
	const previousMonth = month === 1 ? 12 : month - 1;
	const previousYear = month === 1 ? year - 1 : year;
	return { previousMonth, previousYear };
};

/**
 * Get the next month and year.
 *
 * @param month The month.
 * @param year  The year.
 * @return The next month and year.
 */
const getNextMonth = ( month: number, year: number ) => {
	const nextMonth = month === 12 ? 1 : month + 1;
	const nextYear = month === 12 ? year + 1 : year;
	return { nextMonth, nextYear };
};

const universalLock =
	'I acknowledge that using a private store means my plugin will inevitably break on the next store release.';

// We store the selected data in the Add to Cart + Options store so it's
// accessible in sibling blocks. It's necessary to handle submitting the form.
const {
	state: addToCartWithOptionsState,
	actions: addToCartWithOptionsActions,
} = store(
	'woocommerce/add-to-cart-with-options',
	{
		state: {
			get selectedDate() {
				const context =
					getContext< BookingInteractiveCalendarContext >();
				return context.selectedDate;
			},
			get currentViewMonth() {
				const context =
					getContext< BookingInteractiveCalendarContext >();
				return context.viewMonth;
			},
			get currentViewYear() {
				const context =
					getContext< BookingInteractiveCalendarContext >();
				return context.viewYear;
			},
		},
		actions: {
			selectDate( dateString: string | null ) {
				const context =
					getContext< BookingInteractiveCalendarContext >();
				context.selectedDate = dateString;
			},
			setView( month: number, year: number ) {
				const context =
					getContext< BookingInteractiveCalendarContext >();
				context.viewMonth = month;
				context.viewYear = year;
			},
			handleViewByDate( dateString: string ) {
				const context =
					getContext< BookingInteractiveCalendarContext >();
				const month = parseInt( context.viewMonth, 10 );
				const year = parseInt( context.viewYear, 10 );
				const selectedDate = new Date( dateString );
				const selectedMonth = selectedDate.getMonth() + 1;
				const selectedYear = selectedDate.getFullYear();
				if ( selectedMonth !== month || selectedYear !== year ) {
					addToCartWithOptionsActions.setView(
						selectedMonth,
						selectedYear
					);
				}
			},
		},
	},
	{ lock: universalLock }
);

const { state, actions } = store< BookingInteractiveCalendarStore >(
	'woocommerce-bookings/booking-interactive-calendar',
	{
		state: {
			get viewMonthName() {
				const context = getContext< Context >();
				const viewMonth = addToCartWithOptionsState.currentViewMonth;
				return context.monthNames[ viewMonth - 1 ];
			},

			get prevMonthLabel() {
				const context = getContext< Context >();
				const viewMonth = addToCartWithOptionsState.currentViewMonth;
				const viewYear = addToCartWithOptionsState.currentViewYear;
				const { previousMonth, previousYear } = getPreviousMonth(
					viewMonth,
					viewYear
				);
				return `Go to ${
					context.monthNames[ previousMonth - 1 ]
				} ${ previousYear }`;
			},

			get nextMonthLabel() {
				const context = getContext< Context >();
				const viewMonth = addToCartWithOptionsState.currentViewMonth;
				const viewYear = addToCartWithOptionsState.currentViewYear;
				const { nextMonth, nextYear } = getNextMonth(
					viewMonth,
					viewYear
				);
				return `Go to ${
					context.monthNames[ nextMonth - 1 ]
				} ${ nextYear }`;
			},

			get calendarDays() {
				const context = getContext< Context >();
				const year = addToCartWithOptionsState.currentViewYear;
				const month = addToCartWithOptionsState.currentViewMonth;

				const firstDay = new Date( year, month - 1, 1 );
				const daysInMonth = new Date( year, month, 0 ).getDate();
				// Get the day of week (0 = Sunday, 1 = Monday, etc.)
				const firstDayOfWeek = firstDay.getDay();
				// Calculate offset based on week start setting
				const weekStartsOn = context.weekStartsOn || 0;
				const startingDayOfWeek =
					( firstDayOfWeek - weekStartsOn + 7 ) % 7;

				const days: CalendarDay[] = [];

				// Previous month days.
				const { previousMonth, previousYear } = getPreviousMonth(
					month,
					year
				);
				const prevMonthDays = new Date(
					previousYear,
					previousMonth,
					0
				).getDate();

				for ( let i = startingDayOfWeek - 1; i >= 0; i-- ) {
					const dayNumber = prevMonthDays - i;
					const dateString = `${ previousYear }-${ String(
						previousMonth
					).padStart( 2, '0' ) }-${ String( dayNumber ).padStart(
						2,
						'0'
					) }`;

					days.push( {
						key: `prev-${ dayNumber }`,
						dayNumber,
						dateString,
						isCurrentMonth: false,
					} );
				}

				// Current month days.
				for ( let day = 1; day <= daysInMonth; day++ ) {
					const dateString = `${ year }-${ String( month ).padStart(
						2,
						'0'
					) }-${ String( day ).padStart( 2, '0' ) }`;

					days.push( {
						key: `current-${ day }`,
						dayNumber: day,
						dateString,
						isCurrentMonth: true,
					} );
				}

				// Next month days.
				const totalCells = 35; // 7 x 5.
				const remainingCells = totalCells - days.length;
				if ( 0 >= remainingCells ) {
					return days;
				}

				const { nextMonth, nextYear } = getNextMonth( month, year );
				for ( let day = 1; day <= remainingCells; day++ ) {
					const dateString = `${ nextYear }-${ String(
						nextMonth
					).padStart( 2, '0' ) }-${ String( day ).padStart(
						2,
						'0'
					) }`;

					days.push( {
						key: `next-${ day }`,
						dayNumber: day,
						dateString,
						isCurrentMonth: false,
					} );
				}

				return days;
			},

			// Derived states for dynamic day properties
			get dayIsToday() {
				const context = getContext< Context & { day: CalendarDay } >();
				const today = new Date();
				const todayString = formatDateString( today );
				return context.day.dateString === todayString;
			},

			get dayIsSelected() {
				const context = getContext< Context & { day: CalendarDay } >();
				return (
					addToCartWithOptionsState.selectedDate ===
					context.day.dateString
				);
			},

			get dayIsDisabled() {
				// Always enable today's date, even if there are no time slots
				if ( state.dayIsToday ) {
					return false;
				}

				const context = getContext< Context & { day: CalendarDay } >();
				const dateString = context.day.dateString;

				// Create date objects at 00:00:00 for consistent day-based comparison
				const dateParts = dateString.split( '-' );
				const dateToCheck = new Date(
					parseInt( dateParts[ 0 ], 10 ),
					parseInt( dateParts[ 1 ], 10 ) - 1,
					parseInt( dateParts[ 2 ], 10 )
				);

				const minDate = new Date(
					addToCartWithOptionsState.wcBookingsMinDate
				);
				minDate.setHours( 0, 0, 0, 0 );
				const maxDate = new Date(
					addToCartWithOptionsState.wcBookingsMaxDate
				);
				maxDate.setHours( 0, 0, 0, 0 );

				if ( dateToCheck < minDate || dateToCheck > maxDate ) {
					return true;
				}

				const availabilityCache =
					addToCartWithOptionsState.availabilityCache as Record<
						string,
						Record< string, number >
					>;
				const cacheKey = formatMonthString( dateToCheck );
				const daySlots = availabilityCache[ cacheKey ]?.[
					dateString
				] as Record< string, unknown > | number | undefined;

				const hasSlots =
					daySlots && typeof daySlots === 'object'
						? Object.keys( daySlots ).length > 0
						: Boolean( daySlots );

				// Disabled when there are no slots for this date.
				return ! hasSlots;
			},

			get dayAriaLabel() {
				const context = getContext< Context & { day: CalendarDay } >();
				const dateString = context.day.dateString;
				const dateParts = dateString.split( '-' );
				const year = parseInt( dateParts[ 0 ], 10 );
				const month = parseInt( dateParts[ 1 ], 10 );
				const dayNumber = parseInt( dateParts[ 2 ], 10 );

				return `${
					context.monthNames[ month - 1 ]
				} ${ dayNumber }, ${ year }`;
			},

			get dayTabIndex() {
				return state.dayIsDisabled ? -1 : 0;
			},

			get calendarKey() {
				return `${ context.viewYear }-${ context.viewMonth }`;
			},

			get isPreviousMonthDisabled() {
				if ( addToCartWithOptionsState.isLoadingAvailability ) {
					return true;
				}

				const minDate = addToCartWithOptionsState.wcBookingsMinDate;
				const viewMonth = addToCartWithOptionsState.currentViewMonth;
				const viewYear = addToCartWithOptionsState.currentViewYear;

				if (
					( viewMonth <= minDate.getMonth() + 1 &&
						viewYear === minDate.getFullYear() ) ||
					viewYear < minDate.getFullYear()
				) {
					return true;
				}

				return false;
			},

			get isNextMonthDisabled() {
				if ( addToCartWithOptionsState.isLoadingAvailability ) {
					return true;
				}

				const maxDate = addToCartWithOptionsState.wcBookingsMaxDate;
				const viewMonth = addToCartWithOptionsState.currentViewMonth;
				const viewYear = addToCartWithOptionsState.currentViewYear;

				if (
					( viewMonth >= maxDate.getMonth() + 1 &&
						viewYear === maxDate.getFullYear() ) ||
					viewYear > maxDate.getFullYear()
				) {
					return true;
				}

				return false;
			},
		},

		actions: {
			async navigateToPreviousMonth() {
				const viewMonth = addToCartWithOptionsState.currentViewMonth;
				const viewYear = addToCartWithOptionsState.currentViewYear;
				if ( viewMonth === 1 ) {
					addToCartWithOptionsActions.setView( 12, viewYear - 1 );
				} else {
					addToCartWithOptionsActions.setView(
						viewMonth - 1,
						viewYear
					);
				}
			},
			async navigateToNextMonth() {
				const viewMonth = addToCartWithOptionsState.currentViewMonth;
				const viewYear = addToCartWithOptionsState.currentViewYear;
				if ( viewMonth === 12 ) {
					addToCartWithOptionsActions.setView( 1, viewYear + 1 );
				} else {
					addToCartWithOptionsActions.setView(
						viewMonth + 1,
						viewYear
					);
				}
			},
			handleSelectDate: withSyncEvent( ( event: Event ) => {
				const dayElement = getElement()?.ref as HTMLElement;
				if ( ! dayElement ) {
					return;
				}

				const isDisabled =
					dayElement.getAttribute( 'aria-disabled' ) === 'true';
				if ( isDisabled ) {
					event.preventDefault();
					return;
				}

				const dateString = dayElement.dataset.date;

				// Do nothing if clicked on the same date.
				if ( addToCartWithOptionsState.selectedDate === dateString ) {
					return;
				}

				addToCartWithOptionsActions.selectDate( dateString );
				addToCartWithOptionsActions.handleViewByDate( dateString );

				// Focus the time slots block if config requires time selection.
				const config = getConfig(
					'woocommerce/add-to-cart-with-options'
				);
				if ( config.wcBookingsRequiresTimeSelection ) {
					const form = dayElement.closest( 'form' );
					const timeSlotsBlock = form?.querySelector(
						'.wc-bookings-time-slots__grid'
					);
					if ( timeSlotsBlock ) {
						setTimeout( () => {
							const firstButton = (
								timeSlotsBlock as HTMLElement
							 )?.querySelector( 'button' );
							if ( firstButton ) {
								( firstButton as HTMLElement )?.focus();
							}
						}, 0 );
					}
				}
			} ),
			onTouchStart( event: TouchEvent ) {
				const context = getContext< Context >();
				const { clientX } = event.touches[ 0 ];

				context.touchStartX = clientX;
				context.touchCurrentX = clientX;
				context.isDragging = true;
			},
			onTouchMove( event: TouchEvent ) {
				const context = getContext< Context >();
				if ( ! context.isDragging ) {
					return;
				}

				const { clientX } = event.touches[ 0 ];
				context.touchCurrentX = clientX;

				// Only prevent default if there's significant horizontal movement
				const delta = clientX - context.touchStartX;
				if ( Math.abs( delta ) > 10 ) {
					event.preventDefault();
				}
			},
			onTouchEnd: () => {
				const context = getContext< Context >();
				if ( ! context.isDragging ) {
					return;
				}

				const SNAP_THRESHOLD = 0.4;
				const delta = context.touchCurrentX - context.touchStartX;
				const element = getElement()?.ref;
				const calendarWidth = element?.offsetWidth || 0;

				// Only trigger swipe actions if there was significant movement
				if ( Math.abs( delta ) > calendarWidth * SNAP_THRESHOLD ) {
					if ( delta > 0 && ! state.isPreviousMonthDisabled ) {
						actions.navigateToPreviousMonth();
					} else if ( delta < 0 && ! state.isNextMonthDisabled ) {
						actions.navigateToNextMonth();
					}
				}

				// Reset touch state
				context.isDragging = false;
				context.touchStartX = 0;
				context.touchCurrentX = 0;
			},
		},
		callbacks: {
			initCalendar() {
				// Preselect today's date if no date is already selected
				if ( ! addToCartWithOptionsState.selectedDate ) {
					const today = new Date();
					const todayString = formatDateString( today );
					addToCartWithOptionsActions.selectDate( todayString );
					addToCartWithOptionsActions.handleViewByDate( todayString );
				}
			},
		},
	}
);
