/**
 * WordPress dependencies
 */
import {
	store,
	getContext,
	getElement,
	getConfig,
} from '@wordpress/interactivity';

export type Context = {
	slots: object;
	currentPage: number;
	slotsPerPage: number;
	touchStartX: number;
	touchCurrentX: number;
	isDragging: boolean;
};

export type BookingTimeSlotsStore = {
	state: {
		isVisible: boolean;
		shouldShowPlaceholder: boolean;
		shouldShowPagination: boolean;
		totalPages: number;
		pages: {
			pageNumber: number;
			ariaLabel: string;
			isSelected: boolean;
		}[];
	};
	actions: {
		nextPage: () => void;
		prevPage: () => void;
		handleGoToPage: () => void;
		handleSelectTime: () => void;
		findNextAvailable: () => void;
		onTouchStart: ( event: TouchEvent ) => void;
		onTouchMove: ( event: TouchEvent ) => void;
		onTouchEnd: () => void;
	};
};

export type AddToCartWithOptionsContext = {
	selectedTime: string | null;
	timeslotsPage: number;
};

/**
 * Accepts HH:MM:SS needs to return e.g. 09:00 PM or 09:00 AM.
 *
 * // TODO: Replace this with formatted date from the API so we streamline locale and formatting settings.
 *
 * @param timeString HH:MM:SS
 * @return 09:00 PM or 09:00 AM
 */
const formatTimeString = ( timeString: string ) => {
	const [ hours, minutes ] = timeString.split( ':' ).map( Number );
	const ampm = hours >= 12 ? 'pm' : 'am';
	const hours12 = hours % 12 || 12;
	return `${ String( hours12 ).padStart( 2, '0' ) }:${ String(
		minutes
	).padStart( 2, '0' ) } ${ ampm }`;
};

const universalLock =
	'I acknowledge that using a private store means my plugin will inevitably break on the next store release.';

// We store the selected data in the Add to Cart + Options store so it's
// accessible in sibling blocks. It's necessary to handle submitting the form.
const {
	state: addToCartWithOptionsState,
	actions: addToCartWithOptionsActions,
} = store(
	'woocommerce/add-to-cart-with-options',
	{
		state: {
			get selectedTime() {
				const context = getContext< AddToCartWithOptionsContext >();
				return context.selectedTime;
			},
			get currentTimeSlotsPage() {
				const context = getContext< Context >();
				return context.timeslotsPage || 1;
			},
			get firstAvailableDateWithSlots() {
				const availabilityCache =
					addToCartWithOptionsState.availabilityCache as Record<
						string,
						Record< string, Record< string, number > | null >
					>;

				if ( ! availabilityCache ) {
					return null;
				}

				const minDate = addToCartWithOptionsState.wcBookingsMinDate;
				const maxDate = addToCartWithOptionsState.wcBookingsMaxDate;

				// Get all month keys and sort them chronologically
				const monthKeys = Object.keys( availabilityCache ).sort();

				// Iterate through months in chronological order
				for ( const monthKey of monthKeys ) {
					const monthData = availabilityCache[ monthKey ];
					if ( ! monthData ) {
						continue;
					}

					// Get all date keys for this month and sort them chronologically
					const dateKeys = Object.keys( monthData ).sort();

					// Iterate through dates in chronological order
					for ( const dateString of dateKeys ) {
						const dateSlots = monthData[ dateString ];

						// Skip if null or empty
						if (
							! dateSlots ||
							typeof dateSlots !== 'object' ||
							Object.keys( dateSlots ).length === 0
						) {
							continue;
						}

						// Check if date is within min/max range
						const dateParts = dateString.split( '-' );
						const dateToCheck = new Date(
							parseInt( dateParts[ 0 ], 10 ),
							parseInt( dateParts[ 1 ], 10 ) - 1,
							parseInt( dateParts[ 2 ], 10 )
						);
						dateToCheck.setHours( 0, 0, 0, 0 );

						const minDateCopy = new Date( minDate );
						minDateCopy.setHours( 0, 0, 0, 0 );
						const maxDateCopy = new Date( maxDate );
						maxDateCopy.setHours( 0, 0, 0, 0 );

						if (
							dateToCheck >= minDateCopy &&
							dateToCheck <= maxDateCopy
						) {
							return dateString;
						}
					}
				}

				return null;
			},
		},
		actions: {
			selectTime( time: string | null ) {
				const context = getContext< AddToCartWithOptionsContext >();
				context.selectedTime = time;
			},
			setTimeSlotsPage( page: number ) {
				const context = getContext< Context >();
				context.timeslotsPage = page;
			},
		},
	},
	{ lock: universalLock }
);

const { state, actions } = store< BookingTimeSlotsStore >(
	'woocommerce-bookings/booking-time-slots',
	{
		state: {
			get isVisible() {
				return !! getConfig( 'woocommerce/add-to-cart-with-options' )
					?.wcBookingsRequiresTimeSelection;
			},
			get shouldShowPlaceholder() {
				return (
					state.slots.length === 0 &&
					addToCartWithOptionsState.selectedDate &&
					! addToCartWithOptionsState.isLoadingAvailability
				);
			},
			get shouldShowPagination() {
				return state.totalPages > 1;
			},
			get totalPages() {
				const slotsPerPage = getContext< Context >().slotsPerPage;
				return Math.ceil( state.slots.length / slotsPerPage );
			},
			get pages() {
				return Array.from(
					{ length: state.totalPages },
					( _, index ) => ( {
						pageNumber: index + 1,
						ariaLabel: `Go to page ${ index + 1 }`,
						isSelected:
							index + 1 ===
							addToCartWithOptionsState.currentTimeSlotsPage,
					} )
				);
			},
			get slots() {
				const availabilityCache =
					addToCartWithOptionsState.availabilityCache;
				if ( ! availabilityCache ) {
					return [];
				}

				const selectedDate = addToCartWithOptionsState.selectedDate
					? new Date( addToCartWithOptionsState.selectedDate )
					: null;
				if ( ! selectedDate ) {
					return [];
				}

				selectedDate?.setHours( 0, 0, 0, 0 );

				// get YYYY-MM from string addToCartWithOptionsState.selectedDate.
				const monthKey =
					addToCartWithOptionsState.selectedDate.substring( 0, 7 );
				if ( ! availabilityCache[ monthKey ] ) {
					return [];
				}
				const slots =
					availabilityCache[ monthKey ][
						addToCartWithOptionsState.selectedDate
					];
				if ( ! slots ) {
					return [];
				}

				// Return slots with only static data - dynamic properties are computed as derived states
				const formattedSlots = Object.entries( slots ).map(
					( [ time, capacity ] ) => {
						return {
							time,
							capacity,
						};
					}
				);

				// Prevent selecting past times.
				const todayMidnight = new Date();
				todayMidnight.setHours( 0, 0, 0, 0 );
				const minDate = addToCartWithOptionsState.wcBookingsMinDate;
				const isTodaySelected =
					todayMidnight.getTime() === selectedDate?.getTime();

				const finalSlots = isTodaySelected
					? formattedSlots.filter( ( slot ) => {
							const slotDateTime = new Date(
								`${ addToCartWithOptionsState.selectedDate } ${ slot.time }`
							);
							return slotDateTime >= minDate;
					  } )
					: formattedSlots;

				if ( ! finalSlots.length ) {
					return [];
				}

				return finalSlots;
			},
			get slotsForPage() {
				const currentPage =
					addToCartWithOptionsState.currentTimeSlotsPage;
				const slotsPerPage = getContext< Context >().slotsPerPage;
				return state.slots.slice(
					( currentPage - 1 ) * slotsPerPage,
					currentPage * slotsPerPage
				);
			},
			get isPreviousPageDisabled() {
				return addToCartWithOptionsState.currentTimeSlotsPage === 1;
			},
			get isNextPageDisabled() {
				return (
					addToCartWithOptionsState.currentTimeSlotsPage ===
					state.totalPages
				);
			},
			// Derived states for dynamic slot properties
			get slotIsSelected() {
				const context = getContext<
					Context & { slot: { time: string; capacity: number } }
				>();
				const selectedTime = addToCartWithOptionsState.selectedTime;
				return selectedTime && selectedTime === context.slot.time;
			},
			get slotAriaLabel() {
				const context = getContext<
					Context & { slot: { time: string; capacity: number } }
				>();
				return `Select ${ formatTimeString( context.slot.time ) }`;
			},
			get slotTimeString() {
				const context = getContext<
					Context & { slot: { time: string; capacity: number } }
				>();
				return formatTimeString( context.slot.time );
			},
		},
		actions: {
			nextPage() {
				const nextPage =
					addToCartWithOptionsState.currentTimeSlotsPage ===
					state.totalPages
						? 1
						: addToCartWithOptionsState.currentTimeSlotsPage + 1;
				addToCartWithOptionsActions.setTimeSlotsPage( nextPage );
			},
			prevPage() {
				const nextPage =
					addToCartWithOptionsState.currentTimeSlotsPage === 1
						? state.totalPages
						: addToCartWithOptionsState.currentTimeSlotsPage - 1;
				addToCartWithOptionsActions.setTimeSlotsPage( nextPage );
			},
			handleGoToPage() {
				const elementButton = getElement()?.ref as HTMLElement;
				if ( ! elementButton ) {
					return;
				}
				const pageNumber = parseInt(
					elementButton.getAttribute( 'data-pageNumber' ) as string,
					10
				);
				if ( pageNumber < 1 || pageNumber > state.totalPages ) {
					return;
				}

				addToCartWithOptionsActions.setTimeSlotsPage( pageNumber );
			},

			handleSelectTime() {
				const timeElement = getElement()?.ref as HTMLElement;
				if ( ! timeElement ) {
					return;
				}

				const time = timeElement.dataset.time;

				// Do nothing if clicked on the same time.
				if ( addToCartWithOptionsState.selectedTime === time ) {
					return;
				}

				addToCartWithOptionsActions.selectTime( time );

				// Focus confirm button.
				const form = timeElement.closest( 'form' );
				const confirmButtonsBlock = form?.querySelector(
					'.wc-bookings-modal-buttons'
				);
				if ( confirmButtonsBlock ) {
					setTimeout( () => {
						const firstButton = (
							confirmButtonsBlock as HTMLElement
						 )?.querySelector( 'a' );
						if ( firstButton ) {
							( firstButton as HTMLElement )?.focus();
						}
					}, 0 );
				}
			},

			onTouchStart( event: TouchEvent ) {
				const context = getContext< Context >();
				const { clientX } = event.touches[ 0 ];

				context.touchStartX = clientX;
				context.touchCurrentX = clientX;
				context.isDragging = true;
			},
			onTouchMove( event: TouchEvent ) {
				const context = getContext< Context >();
				if ( ! context.isDragging ) {
					return;
				}

				const { clientX } = event.touches[ 0 ];
				context.touchCurrentX = clientX;

				// Only prevent default if there's significant horizontal movement
				const delta = clientX - context.touchStartX;
				if ( Math.abs( delta ) > 10 ) {
					event.preventDefault();
				}
			},
			onTouchEnd: () => {
				const context = getContext< Context >();
				if ( ! context.isDragging ) {
					return;
				}

				const SNAP_THRESHOLD = 0.3;
				const delta = context.touchCurrentX - context.touchStartX;
				const element = getElement()?.ref;
				const calendarWidth = element?.offsetWidth || 0;

				// Only trigger swipe actions if there was significant movement
				if ( Math.abs( delta ) > calendarWidth * SNAP_THRESHOLD ) {
					if ( delta > 0 ) {
						actions.prevPage();
					} else if ( delta < 0 ) {
						actions.nextPage();
					}
				}

				// Reset touch state
				context.isDragging = false;
				context.touchStartX = 0;
				context.touchCurrentX = 0;
			},

			findNextAvailable() {
				const firstAvailableDate =
					addToCartWithOptionsState.firstAvailableDateWithSlots;
				if ( firstAvailableDate ) {
					addToCartWithOptionsActions.selectDate(
						firstAvailableDate
					);
					addToCartWithOptionsActions.handleViewByDate(
						firstAvailableDate
					);
				}
			},
		},
		callbacks: {
			resetPaginationOnSelectedDate() {
				const selectedDate = addToCartWithOptionsState.selectedDate;
				if ( ! selectedDate ) {
					return;
				}

				// Reset pagination to first page when the selected date changes.
				addToCartWithOptionsActions.setTimeSlotsPage( 1 );

				if ( state?.slots?.length ) {
					const firstSlotTime = state.slots[ 0 ]?.time;
					if ( firstSlotTime ) {
						addToCartWithOptionsActions.selectTime( firstSlotTime );
					}
				} else {
					addToCartWithOptionsActions.selectTime( null );
				}
			},
		},
	}
);
