<?php
/**
 * WooCommerce Google Analytics Pro
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Google Analytics Pro to newer
 * versions in the future. If you wish to customize WooCommerce Google Analytics Pro for your
 * needs please refer to http://docs.woocommerce.com/document/woocommerce-google-analytics-pro/ for more information.
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2015-2024, SkyVerge, Inc.
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Google_Analytics_Pro\API;

use SkyVerge\WooCommerce\Google_Analytics_Pro\API\Collect_API\Response;
use SkyVerge\WooCommerce\Google_Analytics_Pro\API\Collect_API\Request;
use SkyVerge\WooCommerce\Google_Analytics_Pro\Plugin;
use SkyVerge\WooCommerce\PluginFramework\v5_15_11 as Framework;

defined( 'ABSPATH' ) or exit;

/**
 * The Collect API for GA4 wrapper class.
 *
 * This is an experimental wrapper class for tracking server-side events using the
 * https://www.google-analytics.com/g/collect endpoint, which is normally used by gtag.js on the frontend.
 *
 * The official, recommended way to track server-side events is using Measurement Protocol API, which seems to suffer
 * from session attribution issues (for example, purchase journeys have 100% drop-off after first step).
 *
 * In our testing, it seems that using Collect API / endpoint fixes these issues. However, there is no official
 * documentation for this endpoint, and the current implementation is reverse-engineered from gtag.js calls.
 *
 * Based on @see https://www.thyngster.com/app-web-google-analytics-measurement-protocol-version-2#introduction
 *
 * @since 2.1.0
 */
class Collect_API extends Framework\SV_WC_API_Base {


	/** @var string|null Google Analytics measurement ID */
	private ?string $measurement_id;


	/**
	 * Constructs the class.
	 *
	 * @since 2.1.0
	 *
	 * @param ?string $measurement_id
	 */
	public function __construct( ?string $measurement_id ) {

		$this->measurement_id = $measurement_id;
		$this->request_uri    = 'https://www.google-analytics.com/g/collect';
		$this->request_method = 'GET';
	}


	/**
	 * Collects (records) an event via the Collect API.
	 *
	 * @since 2.1.0
	 *
	 * @param array $data event data
	 * @return bool whether the event was collected or not
	 */
	public function collect( array $data ): bool {

		if (! $this->measurement_id || empty( $data['client_id'] ) ) {
			return false;
		}

		try {

			$this->set_response_handler( Response::class );

			$this->perform_request( $this->get_new_request( $data ) );

		} catch ( Framework\SV_WC_API_Exception $e ) {

			/* translators: Placeholders: %s - error message */
			$error = sprintf( __( 'Error tracking event: %s', 'woocommerce-google-analytics-pro' ), $e->getMessage() );

			if ( wc_google_analytics_pro()->get_integration()->debug_mode_on() ) {
				wc_google_analytics_pro()->log( $error );
			}

			return false;
		}

		return true;
	}


	/**
	 * Builds and returns a new API request object.
	 *
	 * @since 2.1.0
	 *
	 * @param array $args unused
	 *
	 * @return Request
	 */
	protected function get_new_request( $args = [] ): Request {

		return new Request( $this->measurement_id, $args );
	}


	/**
	 * Gets the request user agent.
	 *
	 * Checks for the presence of a browser to send to Google Analytics.
	 *
	 * @see Framework\SV_WC_API_Base::get_request_user_agent() for the default implementation
	 * @since 2.1.0
	 * @return string
	 */
	protected function get_request_user_agent(): string {

		return wc_get_user_agent() ?: parent::get_request_user_agent();
	}


	/**
	 * Gets the plugin instance.
	 *
	 * @see Framework\SV_WC_API_Base::get_plugin()
	 *
	 * @since 2.1.0
	 * @return Plugin
	 */
	protected function get_plugin(): Plugin {

		return wc_google_analytics_pro();
	}


	/**
	 * Gets the request URL query.
	 *
	 * This method is based on Framework\SV_WC_API_Base::get_request_query(), except it does not check the request method.
	 *
	 * @see Framework\SV_WC_API_Base::get_request_query() for the default implementation
	 * @since 2.1.0
	 * @return string
	 */
	protected function get_request_query(): string {

		$query   = '';
		$request = $this->get_request();

		if ( $request ) {

			$params = $request->get_params();

			if ( ! empty( $params ) ) {
				$query = http_build_query( $params, '', '&' );
			}
		}

		return $query;
	}


}
