<?php
/**
 * WooCommerce Google Analytics Pro
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Google Analytics Pro to newer
 * versions in the future. If you wish to customize WooCommerce Google Analytics Pro for your
 * needs please refer to http://docs.woocommerce.com/document/woocommerce-google-analytics-pro/ for more information.
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2015-2024, SkyVerge, Inc.
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking\Events\GA4;

use SkyVerge\WooCommerce\Google_Analytics_Pro\Helpers\Product_Helper;
use SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking;
use SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking\Adapters\Product_Item_Event_Data_Adapter;
use SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking\Events\GA4_Event;
use SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking\Events\Traits\Has_Deferred_AJAX_Trigger;
use SkyVerge\WooCommerce\PluginFramework\v5_15_11\SV_WC_Helper;
use WC_Product;
use WP_Block;

defined( 'ABSPATH' ) or exit;

/**
 * The "view item list" event.
 *
 * @since 2.0.0
 */
class View_Item_List_Event extends GA4_Event {

	use Has_Deferred_AJAX_Trigger;

	/** @var string the event ID */
	public const ID = 'view_item_list';

	/** @var WC_Product[] the products in the list to track */
	protected array $products = [];

	/** @var array<string, WC_Product[]> products to track, keyed by grid block name */
	protected array $grid_products = [];

	/** @var string|null current block being rendered */
	protected ?string $current_block = null;

	/** @var bool whether we're already tracking products in other blocks */
	protected bool $tracking_block_products = false;

	/** @var string the ajax action name  */
	protected string $ajax_action = 'wc_google_analytics_pro_view_item_list';

	/** @var string the ajax callback method name  */
	protected string $ajax_callback = 'track_via_ajax';


	/**
	 * @inheritdoc
	 */
	public function get_form_field_title(): string {

		return __( 'View Item List', 'woocommerce-google-analytics-pro' );
	}


	/**
	 * @inheritdoc
	 */
	public function get_form_field_description(): string {

		return __( 'Triggered when a customer views a list of products, for example the shop page or related products under a single product page.', 'woocommerce-google-analytics-pro' );
	}


	/**
	 * @inheritdoc
	 */
	public function get_default_name(): string {

		return 'view_item_list';
	}


	/**
	 * @inheritdoc
	 */
	public function register_hooks() : void {

		$this->register_ajax_hooks();

		add_action( 'woocommerce_before_shop_loop_item', [ $this, 'remember_product' ] );
		add_action( 'woocommerce_product_loop_end', [ $this, 'trigger_tracking' ] );

		// hooks for tracking events on product grid blocks
		add_filter( 'pre_render_block', [ $this, 'remember_product_grid' ], 10, 3 );
		add_filter( 'woocommerce_blocks_product_grid_item_html', [ $this, 'remember_product_grid_product' ], 10, 3 );
		add_filter( 'render_block', [ $this, 'track_in_product_grids' ], 10, 2 );

		// hooks for tracking events on the SPA-like "all products"
		add_filter( 'render_block_woocommerce/all-products', [ $this, 'track_in_products_block' ] );
	}


	/**
	 * Remember each product that is being looped over inside a product loop.
	 *
	 * This is necessary because at the end of the loop, we no longer have access to the list of products that were looped over.
	 *
	 * @internal
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 */
	public function remember_product(): void {
		global $product;

		if ( ! $product instanceof WC_Product ) {
			return;
		}

		$this->products[] = $product;
	}


	/**
	 * Triggers the tracking call at the end of each product loop.
	 *
	 * We're hooking into a filter because there is no equivalent action hook that is guaranteed to exist on all product loops.
	 *
	 * @internal
	 *
	 * @since 2.0.0
	 *
	 * @param string $loop_end the loop end HTML
	 * @return string
	 */
	public function trigger_tracking( $loop_end = '' ) : string {

		$this->track();

		return $loop_end;
	}


	/**
	 * @inheritdoc
	 */
	public function track( $products = null, $list_name = null ) : void {

		$products = $products ?? $this->products;

		if ( empty( $products ) || ! $this->should_track_item_list_view() ) {
			return;
		}

		if ( Tracking::not_page_reload() ) {

			$this->record_via_js( $this->get_event_params( $products, $list_name ) );
		}
	}


	protected function get_event_params( array $products, ?string $list_name = null ) {
		return [
			'category'       => 'Products',
			'item_list_name' => $list_name ?: Product_Helper::get_list_type(),
			'items'          => array_values( array_map( static function( $product ) {

				return ( new Product_Item_Event_Data_Adapter( $product ) )->convert_from_source();

			}, $products ) ),
		];
	}


	/**
	 * Determines whether we should track the item list view on the current page.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	protected function should_track_item_list_view() : bool {

		$track_on = (array) wc_google_analytics_pro()->get_integration()->get_option( 'track_item_list_views_on', [] );

		// bail if product impression tracking is disabled on product pages, and we're on a product page
		// note: this doesn't account for the [product_page] shortcode unfortunately
		if ( ! in_array( 'single_product_pages', $track_on, true ) && is_product() ) {
			return false;
		}

		// bail if product impression tracking is disabled on product archive pages, and we're on an archive page
		if ( ! in_array( 'archive_pages', $track_on, true ) && ( is_shop() || is_product_taxonomy() || is_product_category() || is_product_tag() ) ) {
			return false;
		}

		return true;
	}


	/**
	 * Remembers current block / product grid, so we can later associate products with product lists.
	 *
	 * @since 2.1.0
	 * @see \Automattic\WooCommerce\Blocks\BlockTypes\AbstractProductGrid
	 * @internal
	 *
	 * @param mixed $pre_render
	 * @param array $block
	 * @return mixed
	 */
	public function remember_product_grid( $pre_render, array $block ) {

		$block_name = $block['blockName'] ?? null;

		if ( Tracking::is_frontend_request() && $block_name && Product_Helper::is_product_grid_block( $block_name ) ) {

			$this->current_block                = $block_name;
			$this->grid_products[ $block_name ] = [];
		}

		return $pre_render;
	}


	/**
	 * Remembers each product being rendered inside a product grid.
	 *
	 * @since 2.1.0
	 * @see \Automattic\WooCommerce\Blocks\BlockTypes\AbstractProductGrid
	 * @internal
	 *
	 * @param string $html
	 * @param mixed $data
	 * @param WC_Product $product
	 * @return string
	 */
	public function remember_product_grid_product( string $html, $data, WC_Product $product ): string {

		if ( Tracking::is_frontend_request() && $this->current_block ) {

			$this->grid_products[ $this->current_block ][] = $product;
		}

		return $html;
	}


	/**
	 * Tracks the event in product grids.
	 *
	 * @since 2.1.0
	 * @see \Automattic\WooCommerce\Blocks\BlockTypes\AbstractProductGrid
	 * @internal
	 *
	 * @param string $block_content
	 * @param array $block
	 * @return string
	 */
	public function track_in_product_grids( string $block_content, array $block ) : string {

		$block_name = $block['blockName'] ?? null;

		if ( Tracking::is_frontend_request() && $this->current_block === $block_name ) {

			if ( ! empty( $products = $this->grid_products[ $block_name ] ?? [] ) ) {

				$this->track( $products, Product_Helper::get_list_type( $block_name ) );
			}

			$this->current_block = null;
		}

		return $block_content;
	}


	/**
	 * Tracks view item lists events when using the "all products" block
	 *
	 * @since 2.1.0
	 *
	 * @internal
	 */
	public function track_in_products_block( string $block_content ) : string {

		if ( Tracking::do_not_track() ) {
			return $block_content;
		}

		if ( ! $this->tracking_block_products ) {

			// Note: `listName` below is only provided in some rare occasions - it's mostly undefined
			$trigger = <<<JS
			wp.hooks.addAction( 'experimental__woocommerce_blocks-product-list-render', 'woocommerce-google-analytics-pro', function({ products, listName }) {
				"__INSERT_AJAX_CALL_HERE__"({ product_ids: products.map(product => product.id), list_name: listName });
			});
			JS;

			$this->trigger_via_ajax( $trigger );
			$this->tracking_grid_products = true;
		}

		return $block_content;
	}


	/**
	 * Tracks the event via AJAX call - used when tracking the event in blocks.
	 *
	 * @since 2.1.0
	 * @internal
	 */
	public function track_via_ajax() : void {

		check_ajax_referer( $this->ajax_action, 'security' );

		$product_ids = SV_WC_Helper::get_posted_value( 'product_ids' ) ?? [];
		$list_name   = SV_WC_Helper::get_posted_value( 'list_name' );

		if ( empty( $product_ids ) ) {
			return;
		}

		$products  = array_map( 'wc_get_product', $product_ids );
		$list_name = Product_Helper::get_list_type( $list_name );

		$this->record_via_api( $this->get_event_params( $products, $list_name ) );
	}


}
