<?php
/**
 * Plugin Name: Gravity Forms Product Add-Ons for WooCommerce
 * Plugin URI: http://woothemes.com/products/gravity-forms-add-ons/
 * Description: Allows you to use Gravity Forms on individual WooCommerce products. Requires the Gravity Forms plugin to work.
 * Version: 3.7.1
 * Author: Element Stark
 * Author URI: https://www.elementstark.com/
 * Developer: Lucas Stark
 * Developer URI: http://www.elementstark.com/
 * Requires at least: 3.1
 * Tested up to: 6.9
 * Text Domain: woocommerce-gravityforms-product-addons
 * Domain Path: /languages
 * Copyright: © 2012-2026 Element Stark LLC.
 * License: GNU General Public License v3.0
 * License URI: http://www.gnu.org/licenses/gpl-3.0.html
 * Requires Plugins: woocommerce, gravityforms
 *
 * WC requires at least: 9.0
 * WC tested up to: 10.4
 * Woo: 18633:a6ac0ab1a1536e3a357ccf24c0650ed0
 *
 * @package WooCommerce Gravity Forms Product Add-Ons
 **/

defined( 'ABSPATH' ) || exit;

const WC_GFPA_TEXTDOMAIN = 'woocommerce-gravityforms-product-addons';
const WC_GFPA_LEGACY_TEXTDOMAIN = 'wc_gf_addons';

/**
 * Required functions
 **/
if ( ! function_exists( 'is_woocommerce_active' ) ) {
    require_once 'woo-includes/woo-functions.php';
}

// Declare support for features.
add_action(
        'before_woocommerce_init',
        function () {
            if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
                \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
                \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'cart_checkout_blocks', __FILE__, true );
            }
        }
);

/**
 * i18n
 * - Loads new domain using standard WP behavior (supports both wp-content/languages/plugins + plugin /languages).
 * - Loads legacy domain (wc_gf_addons) only if legacy files exist.
 */
add_action( 'init', 'wc_gravityforms_product_addons_load_textdomain', 0 );
function wc_gravityforms_product_addons_load_textdomain(): void {
    $locale = determine_locale();

    // Legacy Loco/override location (standard).
    $legacy_plugins_mo = WP_LANG_DIR . "/plugins/" . WC_GFPA_LEGACY_TEXTDOMAIN . "-{$locale}.mo";
    if ( file_exists( $legacy_plugins_mo ) ) {
        load_textdomain( WC_GFPA_LEGACY_TEXTDOMAIN, $legacy_plugins_mo );
    }

    // Legacy WooCommerce directory. Incorrect old location, but some users may have used it.
    $legacy_wc_mo = WP_LANG_DIR . "/woocommerce/woocommerce-gravityforms-product-addons-{$locale}.mo";
    if ( file_exists( $legacy_wc_mo ) ) {
        // Note: loaded into the *legacy* domain only if you had old code using wc_gf_addons.
        // If your old installs used this WC path for the legacy domain, keep this line.
        load_textdomain( WC_GFPA_LEGACY_TEXTDOMAIN, $legacy_wc_mo );
    }

    // New standard domain.
    load_plugin_textdomain(
            WC_GFPA_TEXTDOMAIN,
            false,
            dirname( plugin_basename( __FILE__ ) ) . '/languages/'
    );
}

if ( is_woocommerce_active() ) {
    include 'compatibility.php';

    add_action( 'plugins_loaded', 'wc_gravityforms_product_addons_plugins_loaded' );
    function wc_gravityforms_product_addons_plugins_loaded() {
        if ( wc_gravityforms_is_plugin_active( 'gravityforms/gravityforms.php' ) || wc_gravityforms_is_plugin_active_for_network( 'gravityforms/gravityforms.php' ) ) {
            require_once 'gravityforms-product-addons-main.php';
        } else {
            add_action( 'admin_notices', 'wc_gravityforms_admin_install_notices' );
        }
    }

    function wc_gravityforms_is_plugin_active( $plugin ) {
        return in_array( $plugin, (array) get_option( 'active_plugins', array() ), true ) || wc_gravityforms_is_plugin_active_for_network( $plugin );
    }

    function wc_gravityforms_is_plugin_active_for_network( $plugin ) {
        if ( ! is_multisite() ) {
            return false;
        }

        $plugins = get_site_option( 'active_sitewide_plugins' );
        return isset( $plugins[ $plugin ] );
    }

    function wc_gfpa_get_plugin_url() {
        return plugins_url( basename( plugin_dir_path( __FILE__ ) ), basename( __FILE__ ) );
    }

    function wc_gravityforms_admin_install_notices() {
        ?>
        <div id="message" class="updated woocommerce-error wc-connect">
            <div class="squeezer">
                <h4>
                    <strong><?php esc_html_e( 'Gravity Forms Not Found', WC_GFPA_TEXTDOMAIN ); ?></strong>
                    <?php esc_html_e( '— The Gravity Forms Plugin is required to build and manage the forms for your products.', WC_GFPA_TEXTDOMAIN ); ?>
                </h4>
                <p class="submit">
                    <a href="https://www.gravityforms.com/" class="button-primary">
                        <?php esc_html_e( 'Get Gravity Forms', WC_GFPA_TEXTDOMAIN ); ?>
                    </a>
                </p>
            </div>
        </div>
        <?php
    }
}

/**
 * Migration notice (only shown if legacy translation files exist).
 */
add_action( 'admin_notices', function () {
    if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
        return;
    }

    // Handle dismiss.
    if ( isset( $_GET['wc_gfpa_i18n_dismiss'] ) ) {
        check_admin_referer( 'wc_gfpa_i18n_dismiss' );
        update_option( 'wc_gfpa_i18n_notice_dismissed', 1 );
        return;
    }

    if ( get_option( 'wc_gfpa_i18n_notice_dismissed' ) ) {
        return;
    }

    $locale = determine_locale();

    $legacy_files = [
            WP_LANG_DIR . "/plugins/" . WC_GFPA_LEGACY_TEXTDOMAIN . "-{$locale}.mo",
            WP_LANG_DIR . "/plugins/" . WC_GFPA_LEGACY_TEXTDOMAIN . "-{$locale}.po",
        // If you historically used this custom path, include it:
            WP_LANG_DIR . "/woocommerce/woocommerce-gravityforms-product-addons-{$locale}.mo",
    ];

    $has_legacy = false;
    foreach ( $legacy_files as $file ) {
        if ( file_exists( $file ) ) {
            $has_legacy = true;
            break;
        }
    }

    if ( ! $has_legacy ) {
        return;
    }

    $dismiss_url = wp_nonce_url(
            add_query_arg( 'wc_gfpa_i18n_dismiss', 1 ),
            'wc_gfpa_i18n_dismiss'
    );
    ?>
    <div class="notice notice-info">
        <p>
            <strong><?php esc_html_e( 'Translation update:', WC_GFPA_TEXTDOMAIN ); ?></strong>
            <?php esc_html_e( 'This plugin’s text domain was updated. If you previously created translations, you may need to re-save them so they are stored under the new domain.', WC_GFPA_TEXTDOMAIN ); ?>
        </p>
        <p>
            <?php esc_html_e( 'If you use Loco Translate: open this plugin’s language in Loco and click Save. Loco will write the files to the new standard location.', WC_GFPA_TEXTDOMAIN ); ?>
        </p>
        <p><a href="<?php echo esc_url( $dismiss_url ); ?>"><?php esc_html_e( 'Dismiss', WC_GFPA_TEXTDOMAIN ); ?></a></p>
    </div>
    <?php
} );
