<?php
/**
 * WP File Download
 *
 * @package WP File Download
 * @author  Joomunited
 * @version 1.0
 */

use Joomunited\WPFramework\v1_0_6\Application;
use Joomunited\WPFramework\v1_0_6\View;
use Joomunited\WPFramework\v1_0_6\Utilities;
use Joomunited\WPFramework\v1_0_6\Form;

defined('ABSPATH') || die();

/**
 * Class WpfdViewCategory
 */
class WpfdViewCategory extends View
{
    /**
     * Current theme name
     *
     * @var string
     */
    public $prefix;

    /**
     * Category files
     *
     * @var object|mixed
     */
    public $category;

    /**
     * Category params
     *
     * @var array|mixed
     */
    public $params;

    /**
     * Global main settings
     *
     * @var array|mixed
     */
    public $mainConfig;

    /**
     * Themes list
     *
     * @var array|mixed
     */
    public $themes;

    /**
     * Form settings
     *
     * @var array|mixed
     */
    public $form;

    /**
     * Mail settings
     *
     * @var array|mixed
     */
    public $mailOptions;

    /**
     * Render category
     *
     * @param null $tpl Template name
     *
     * @return void
     */
    public function render($tpl = null)
    {
        Application::getInstance('Wpfd');
        $id_category = Utilities::getInt('id');
        if (empty($id_category)) {
            echo '';
            wp_die();
        }
        $modelCat         = $this->getModel('category');
        $this->category   = $modelCat->getCategory($id_category);
        if (!$this->category) {
            echo '';
            wp_die();
        }
        $this->params     = (array) $this->category->params;
        $modelConfig      = $this->getModel('config');
        $this->mainConfig = $modelConfig->getConfig();
        $this->themes     = $modelConfig->getThemes();
        $mailOptions      = get_option('_wpfd_notifications', array());

        if (empty($mailOptions)) {
            $mailOptions = array();
        }

        $this->mailOptions = $mailOptions;

        if (Utilities::getInput('onlyTheme', 'GET', 'int')) {
            $newTheme           = Utilities::getInput('theme', 'GET', 'string');
            $this->prefix = $newTheme . '_';
            if ($newTheme === 'default') {
                $this->prefix = '';
            }
            $defaultThemeConfig = $modelConfig->getThemeParams($newTheme);
            $this->params       = wp_parse_args($this->params, $defaultThemeConfig);

            if (WpfdBase::checkExistTheme($newTheme)) {
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Print output html
                echo $this->loadTemplate('theme-' . $newTheme);
            } else {
                $dir = trailingslashit(dirname(wpfd_locate_theme($newTheme, 'theme.php')));
                $this->setPath($dir);
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Print output html
                echo $this->loadTemplate('theme-' . $newTheme);
            }
            die();
        }
        $defaultThemeConfig = $modelConfig->getThemeParams($this->params['theme']);
        $this->params       = wp_parse_args($this->params, $defaultThemeConfig);
        $form               = new Form();
        $this->prefix = $this->params['theme'] . '_';
        if ($this->params['theme'] === 'default') {
            $this->prefix = '';
        }
        if ($form->load('category', (array) $this->category->params)) {
            $this->form = $form->render();
        }
        parent::render($tpl);
    }

    /**
     * Helper functions to render a switcher field
     *
     * @param string  $field_name Option name
     * @param string  $label      Field label
     * @param string  $tooltip    Field tooltip
     * @param boolean $echo       Print or return
     *
     * @return string
     */
    public function renderSwitcher($field_name, $label, $tooltip = '', $echo = true)
    {
        $name    = esc_attr('params[' . $this->prefix . $field_name . ']');
        $value   = (isset($this->params[$this->prefix . $field_name]) && ((int) $this->params[$this->prefix . $field_name]) === 1) ? '1' : '0';
        $checked = ($value === '1') ? ' checked' : '';

        $html = '<div class="control-group">';
        $html .= '<label class="control-label" title="'. $tooltip .'" for="ref_' . $name . '">' . esc_html($label) . '</label>
        <div class="ju-switch-button">
            <label class="switch">
                <input type="checkbox" name="ref_' . $name . '" id="ref_' . $name . '" ' . esc_attr($checked) . ' />
                <span class="slider"></span>
            </label>
            <input type="hidden" name="' . $name . '" value="' . esc_attr($value) . '" />
        </div>';
        $html .= '</div>';
        if (true === $echo) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Print output html
            echo $html;
            return '';
        } else {
            return $html;
        }
    }

    /**
     * Helper functions to render a minicolors field
     *
     * @param string  $field_name Option name
     * @param string  $label      Field label
     * @param string  $tooltip    Field tooltip
     * @param boolean $echo       Print or return
     *
     * @return string
     */
    public function renderColor($field_name, $label, $tooltip = '', $echo = true)
    {
        $name  = esc_attr('params[' . $this->prefix . $field_name . ']');
        $value = isset($this->params[$this->prefix . $field_name]) ? esc_html($this->params[$this->prefix . $field_name]) : 0;
        $html = '<div class="control-group">';
        $html .= '
        <label class="control-label" title="'. $tooltip .'" for="' . $name . '">' . esc_html($label) . '</label>
        <div class="controls">';
        $html .= '<input title="" name="' . $name . '"
                   value="' . $value . '"
                   class="ju-input ju-primary-input minicolors wp-color-field ' . $field_name . '" type="text" aria-autocomplete="false">
        </div>';
        $html .= '</div>';

        if (true === $echo) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Print output html
            echo $html;
            return '';
        } else {
            return $html;
        }
    }

    /**
     * Generate input text
     *
     * @param string  $field_name Field name
     * @param string  $label      Field label
     * @param string  $tooltip    Field tooltip
     * @param boolean $echo       Print or return
     *
     * @return string
     */
    public function renderText($field_name, $label, $tooltip = '', $echo = true)
    {
        $name  = esc_attr('params[' . $this->prefix . $field_name . ']');
        $value = esc_html($this->params[$this->prefix . $field_name]);

        $html = '<div class="control-group">';
        $html .= '<label class="control-label" title="'. $tooltip .'" for="' . $name . '">' . esc_html($label) . '</label>
        <div class="controls">
            <input title="" name="' . $name . '" value="' . $value . '"
                   class="ju-input ju-primary-input" type="text">
        </div>';
        $html .= '</div>';

        if (true === $echo) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Print output html
            echo $html;
            return '';
        } else {
            return $html;
        }
    }

    /**
     * Helper functions to render a select field
     *
     * @param string  $field_name Option name
     * @param string  $label      Field label
     * @param string  $tooltip    Field tooltip
     * @param array   $options    Option list
     * @param boolean $echo       Print or return
     *
     * @return string
     */
    public function renderSelectOptions($field_name, $label, $tooltip = '', $options = array(), $echo = true)
    {
        $name  = esc_attr('params[' . $this->prefix . $field_name . ']');
        $value = isset($this->params[$this->prefix . $field_name]) ? $this->params[$this->prefix . $field_name] : '0';
        $html  = '<div class="control-group">';
        $html .= '<label class="control-label" title="'. $tooltip .'" for="' . $name . '">' . esc_html($label) . '</label>';
        $html .= '<div class="controls">';
        $html .= '<select name="' . $name . '" id="' . $name . '" class="ju-input ju-select">';

        foreach ($options as $key => $option) {
            $selected = ((int) $key === (int) $value) ? 'selected' : '';
            $html .= '<option value="' . esc_attr($key) . '" ' . $selected . '>' . esc_attr($option) . '</option>';
        }

        $html .= '</select>';

        $html .= '</div>';
        $html .= '</div>';

        if (true === $echo) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Print output html
            echo $html;
            return '';
        } else {
            return $html;
        }
    }

    /**
     * Helper functions to render a custom fields
     *
     * @param string  $field_name Field name
     * @param string  $label      Field label
     * @param string  $tooltip    Field tooltip
     * @param boolean $echo       Print or return
     *
     * @return string
     */
    public function renderCustomFields($field_name, $label, $tooltip = '', $echo = true)
    {
        $category = $this->category;
        $categoryId = isset($category->term_id) ? $category->term_id : 0;
        $categoryFrom = apply_filters('wpfdAddonCategoryFrom', $categoryId);
        if (function_exists('wpfdCheckInstallACFPlugins') && function_exists('wpfd_get_support_cloud')
            && wpfdCheckInstallACFPlugins() && !in_array($categoryFrom, wpfd_get_support_cloud())) {
            $name = esc_attr('params[' . $this->prefix . $field_name . ']');
            $groupName = esc_attr('params[' . $this->prefix .'groupcustomfields]');
            $value = (isset($this->params[$this->prefix . $field_name]) && ((int) $this->params[$this->prefix . $field_name]) === 1) ? '1' : '0';
            $checked = ($value === '1') ? ' checked' : '';
            $options = array('none' => esc_html__('-- Select ACF groups --', 'wpfd'));
            $optionVal = isset($this->params[$this->prefix . 'groupcustomfields']) ? $this->params[$this->prefix . 'groupcustomfields'] : 'none';
            $showSelect = intval($value) === 1 ? '' : 'display: none;';
            $fieldGroups = acf_get_field_groups(array( 'post_type' => 'wpfd_file' ));

            if (is_array($fieldGroups) && !empty($fieldGroups)) {
                $options['all_groups'] = esc_html__('All ACF groups', 'wpfd');
                foreach ($fieldGroups as $fieldGroup) {
                    $options[$fieldGroup['key']] = $fieldGroup['title'];
                }
            }

            $html = '<div class="control-group">';
            $html .= '<label class="control-label" title="'. $tooltip .'" for="ref_' . $name . '">' . esc_html($label) . '</label>
                        <div class="ju-switch-button">
                            <label class="switch">
                                <input type="checkbox" name="ref_' . $name . '" id="ref_' . $name . '" onChange="jQuery(this).parents(\'.control-group\').find(\'.show_custom_field_select_section\').slideToggle()" ' . esc_attr($checked) . ' />
                                <span class="slider"></span>
                            </label>
                            <input type="hidden" name="' . $name . '" value="' . esc_attr($value) . '" />
                        </div>';
            $html .= '<div class="show_custom_field_select_section" style="' . $showSelect . '; margin: 10px 0">';
            $html .= '<select name="' . $groupName . '" id="' . $groupName . '" class="ju-input ju-select" style="width: 100%; max-width: unset;">';
            foreach ($options as $key => $option) {
                $selected = (strval($key) === strval($optionVal)) ? 'selected' : '';
                $html .= '<option value="' . esc_attr($key) . '" ' . $selected . '>' . esc_attr($option) . '</option>';
            }
            $html .= '</select>';
            $html .= '</div>';
            $html .= '</div>';

            if (true === $echo) {
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Print output html
                echo $html;
                return '';
            } else {
                return $html;
            }
        } else {
            return '';
        }
    }

    /**
     * Generate theme node
     *
     * @param string $name    Theme name
     * @param string $checked Checked state
     *
     * @return string
     */
    public function themeNode($name, $checked = '')
    {
        $html = '<div ref="' . esc_attr($name) . '" class="wpfd-theme ' . esc_attr($checked) . '" title="' . esc_attr($name) . '">';
        $html .= '<p>' . esc_html(ucfirst($name)) . '</p>';
        $html .= '<div class="wpfd-theme-select-icon">';
        $html .= '<img src="' . esc_url($this->themeIcon($name)) . '" />';
        $html .= '</div>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Generate theme icon url
     *
     * @param string $name Theme name
     *
     * @return string
     */
    public function themeIcon($name)
    {
        $iconPath = wpfd_locate_theme($name, 'theme-icon.png');
        if (!file_exists($iconPath)) {
            return WPFD_PLUGIN_URL . 'app/admin/assets/ui/images/custom-theme-icon.png';
        }

        return wpfd_abs_path_to_url($iconPath);
    }
}
