<?php

namespace SPC_Pro\Modules;

use SPC_Pro\Constants;
use SPC\Modules\Module_Interface;
use SW_CLOUDFLARE_PAGECACHE;

class Fallback_Cache implements Module_Interface {
	/**
	 * Plugin instance.
	 *
	 * @var SW_CLOUDFLARE_PAGECACHE
	 */
	private SW_CLOUDFLARE_PAGECACHE $plugin;

	/**
	 * Drop-in file.
	 *
	 * @var string
	 */
	private $dropin_file = 'advanced-cache-pro-addon.php';

	/**
	 * Drop-in source.
	 *
	 * @var string
	 */
	private $dropin_source;

	/**
	 * Drop-in destination.
	 *
	 * @var string
	 */
	private $dropin_destination;

	/**
	 * Logger instance.
	 *
	 * @var \SWCFPC_Logs
	 */
	private $logger;

	/**
	 * Fallback_Cache constructor.
	 *
	 * @param SW_CLOUDFLARE_PAGECACHE $plugin
	 */
	public function __construct( SW_CLOUDFLARE_PAGECACHE $plugin ) {
		$this->plugin             = $plugin;
		$this->logger             = $plugin->get_logger();
		$this->dropin_source      = SWCFPC_PLUGIN_PATH . 'pro/assets/' . $this->dropin_file;
		$this->dropin_destination = WP_CONTENT_DIR . '/wp-cloudflare-super-page-cache/' . $this->dropin_file;
	}


	/**
	 * Initialize.
	 *
	 * @return void
	 */
	public function init() {
		if ( ! $this->is_fallback_cache_enabled() ) {
			return;
		}

		// when using advanced-cache.php, we need to add the dropin.
		add_action( 'swcfpc_advanced_cache_after_enable', [ $this, 'add_dropin' ] );
		add_action( 'swcfpc_advanced_cache_after_disable', [ $this, 'remove_dropin' ] );

		$this->load_dropin();
	}

	public function add_ignored_query_params( $params ) {
		if ( ! is_array( $params ) ) {
			return $params;
		}

		$params = array_unique( array_merge( $params, $this->plugin->get_single_config( Constants::SETTING_CACHE_EXCLUDED_URL_PARAMS, Constants::CACHE_EXCLUDED_URL_PARAMS_DEFAULT ) ) );

		return $params;
	}

	/**
	 * Load the dropin file when CURL for fallback cache is used.
	 *
	 * Otherwise it's loaded by the advanced-cache.php file.
	 */
	private function load_dropin() {
		if ( ! file_exists( $this->dropin_source ) || ! $this->is_curl_fallback_cache_enabled() ) {
			return;
		}

		global $swcfpc_config;

		$swcfpc_config = $this->plugin->get_config();

		require $this->dropin_source;
	}

	/**
	 * Add the dropin file.
	 */
	public function add_dropin() {
		if ( ! file_exists( $this->dropin_source ) ) {
			return;
		}

		if ( file_exists( $this->dropin_destination ) && ! @unlink( $this->dropin_destination ) ) {
			$this->logger->add_log( 'fallback_cache_pro::add_dropin', 'Failed to remove existing dropin file.' );
			return;
		}

		if ( file_put_contents( $this->dropin_destination, file_get_contents( $this->dropin_source ) ) === false ) {
			$this->logger->add_log( 'fallback_cache_pro::add_dropin', 'Failed to copy dropin file.' );
			return;
		}

		$this->logger->add_log( 'fallback_cache_pro::add_dropin', 'Dropin file added.' );
	}

	/**
	 * Remove the dropin file.
	 */
	public function remove_dropin() {
		if ( ! file_exists( $this->dropin_destination ) ) {
			$this->logger->add_log( 'fallback_cache_pro::remove_dropin', 'No file to remove. Dropin file does not exist.' );
			return;
		}

		if ( ! @unlink( $this->dropin_destination ) ) {
			$this->logger->add_log( 'fallback_cache_pro::remove_dropin', 'Failed to remove dropin file.' );
			return;
		}

		$this->logger->add_log( 'fallback_cache_pro::remove_dropin', 'Dropin file removed.' );
	}

	/**
	 * Check if the fallback cache is enabled.
	 */
	private function is_fallback_cache_enabled() {
		return (int) $this->plugin->get_single_config( 'cf_fallback_cache', 0 ) > 0;
	}

	/**
	 * Check if the fallback cache with CURL is enabled.
	 */
	private function is_curl_fallback_cache_enabled() {
		return (int) $this->plugin->get_single_config( 'cf_fallback_cache_curl', 0 ) > 0;
	}
}
