<?php

namespace SPC_Pro\Modules\PageProfiler\Storage;

/**
 * Class Transients
 *
 * Handles storage and retrieval of page profiler data using WordPress transients.
 * Transients provide a way to temporarily store cached data with an expiration time.
 *
 * @package SPC_Pro\Modules\PageProfiler\Storage
 */
class Transients extends Base {
	/**
	 * Prefix for all transient keys to avoid conflicts with other plugins.
	 */
	const PREFIX = '_sprof_';

	/**
	 * Default expiration time for transients (7 days in seconds).
	 */
	const EXPIRATION_TIME = 7 * DAY_IN_SECONDS;

	/**
	 * The actual expiration time to use, which can be filtered.
	 *
	 * @var int
	 */
	private $expiration_time;

	/**
	 * Constructor.
	 *
	 * Sets up the expiration time, allowing it to be modified via WordPress filters.
	 */
	public function __construct() {
		$this->expiration_time = apply_filters( 'spc_page_profiler_transient_expiration', self::EXPIRATION_TIME );
	}

	/**
	 * Generates the full transient key with prefix.
	 *
	 * @param string $key The base key to prefix.
	 * @return string The prefixed key.
	 */
	private function get_key( string $key ) {
		return self::PREFIX . $key;
	}

	/**
	 * Stores data in a transient.
	 *
	 * @param string $key The key to store the data under.
	 * @param array  $data The data to store.
	 * @return void
	 */
	public function store( string $key, array $data ) {
		set_transient( $this->get_key( $key ), $data, $this->expiration_time );
	}

	/**
	 * Retrieves data from a transient.
	 *
	 * @param string $key The key to retrieve data for.
	 * @return mixed The stored data or false if the transient doesn't exist or has expired.
	 */
	public function get( string $key ) {
		return get_transient( $this->get_key( $key ) );
	}

	/**
	 * Deletes a specific transient.
	 *
	 * @param string $key The key of the transient to delete.
	 * @return void
	 */
	public function delete( string $key ) {
		delete_transient( $this->get_key( $key ) );
	}

	/**
	 * Deletes all transients created by this class.
	 *
	 * @return void
	 */
	public function delete_all() {
		global $wpdb;
		$wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s", '_transient_' . self::PREFIX . '%', '_transient_timeout_' . self::PREFIX . '%' ) );
	}
}
