<?php
/**
 * BuddyPress expanding the functionality
 * 
 * @addon bp-social-network
 * @version 1.2
 *
 * @package ThemeREX Addons
 * @since v2.29.3
 */

if ( ! class_exists( 'TRX_Addons_BP_Social_Network_Activity' ) ) {

	class TRX_Addons_BP_Social_Network_Activity {

		private $options = array();

		public function __construct () {

			// Common Settings
			//=====================================
			// Add params to the ThemeREX Addons Options
			add_filter( 'trx_addons_filter_options', array( $this, 'trx_addons_bp_social_network_add_options' ) );

			// Load required styles and scripts for the frontend
			add_action( 'wp_enqueue_scripts', array( $this, 'trx_addons_bp_social_network_load_scripts_front' ), TRX_ADDONS_ENQUEUE_SCRIPTS_PRIORITY );

			// Merge styles to the single stylesheet
			add_filter( 'trx_addons_filter_merge_styles', array( $this, 'trx_addons_bp_social_network_merge_styles' ) );

			// Merge specific scripts to the single file
			add_action( 'trx_addons_filter_merge_scripts', array( $this, 'trx_addons_bp_social_network_merge_scripts' ) );

			// Add specific class to bp activity shortcode
			add_filter( 'bpas_activity_list_classes', array( $this, 'trx_addons_bp_social_network_add_shortcode_class' ) );



			// Functional
			//=====================================
			//add parameters in trx_addons_get_share_links
			add_action( 'trx_addons_filter_social_sharing_attr', array( $this, 'trx_addons_filter_social_sharing_attr' ), 10, 2 );

			//add share list
			add_action( 'bp_activity_entry_meta', array( $this, 'add_share_list_bp_activity_entry_meta' ) );

			//add reaction to activity
			add_action( 'bp_activity_entry_meta', array( $this, 'add_reaction_bp_activity_entry_meta' ) );

			//add reaction to activity comment
			add_action( 'bp_activity_comment_options', array( $this, 'add_reaction_bp_activity_comment_entry_meta' ) );

			//ajax add reaction
			add_action( 'wp_ajax_trx_addons_bp_social_network_post_add_reaction', array( $this, 'ajax_add_reaction' ) );
			add_action( 'wp_ajax_nopriv_trx_addons_bp_social_network_post_add_reaction', array( $this, 'ajax_add_reaction' ) );

			//ajax remove reaction
			add_action( 'wp_ajax_trx_addons_bp_social_network_post_remove_reaction', array( $this, 'ajax_remove_reaction' ) );
			add_action( 'wp_ajax_nopriv_trx_addons_bp_social_network_post_remove_reaction', array( $this, 'ajax_remove_reaction' ) );
		}

		// Common Settings
		//=====================================
		public function trx_addons_bp_social_network_add_options( $options ) {

			trx_addons_array_insert_before( $options, 'sc_section', apply_filters( 'trx_addons_filter_options_bp_social_network', array(
				// Section 'BP Extension'
				'bp_social_network_section' => array(
					'title' => esc_html__( 'BP Extension', 'trx_addons' ),
					'desc' => wp_kses_data( __( "Links to the social profiles and post's share settings", 'trx_addons' ) ),
					'icon' => 'trx_addons_icon-chat',
					'type' => 'panel'
				),

				// Share
				'bp_social_network_socials_section_share' => array(
					'title' => esc_html__( 'Share posts', 'trx_addons' ),
					'icon' => 'trx_addons_icon-share-2',
					'type' => 'section'
				),
				'bp_social_network_share_info' => array(
					'title' => esc_html__( 'URL to share posts', 'trx_addons' ) . trx_addons_get_theme_doc_link( '#themerex_addons_bp_share_posts' ),
					'desc' => wp_kses( __( "Specify URLs to share your posts on social media. If the URL field is left empty, the sharing button will not work.<br>You can use the following macros to include specific post parts into the URL:<br><br>{link} - post's URL,<br>{title} - title of the post,<br>{descr} - excerpt of the post,<br>{image} - post's featured image URL,<br>{id} - post's ID", 'trx_addons' ), 'trx_addons_kses_content' ),
					'type' => 'info'
				),
				'bp_social_network_share' => array(
					'title' => esc_html__( 'Share', 'trx_addons' ),
					'desc' => wp_kses(
								__( "Clone the field group, then select an icon/image, specify the social network's title, and the URL to share posts.", 'trx_addons' )
									. '<br>' . __( "Select an 'email' icon to add 'mailto:' link to the share buttons.", 'trx_addons' )
									. '<br>' . __( "Select a 'link' icon to add 'Copy to Clipboard' link to the share buttons.", 'trx_addons' ),
								'trx_addons_kses_content'
							),
					'translate' => true,
					'clone' => true,
					'std' => array(
						array( 
							'name' => 'icon-twitter',
							'url' => trx_addons_get_share_url( 'twitter' ),
						),
						array(
							'name' => 'icon-facebook',
							'url' => trx_addons_get_share_url( 'facebook' ),
						),
						array(
							'name' => 'icon-email',
							'url' => trx_addons_get_share_url( 'email' ),
						),
						array(
							'name' => 'icon-link',
							'url' => trx_addons_get_share_url( 'link' ),
						)
					),
					'type' => 'group',
					'fields' => array(
						'name' => array(
							'title' => esc_html__( 'Icon', 'trx_addons' ),
							'desc' => wp_kses_data( __( 'Select the network icon', 'trx_addons' ) ),
							'class' => 'trx_addons_column-1_6 trx_addons_new_row',
							'std' => '',
							'options' => array(),
							'style' => trx_addons_get_setting( 'socials_type' ),
							'type' => 'icons'
						),
						'title' => array(
							'title' => esc_html__( 'Title', 'trx_addons' ),
							'desc' => wp_kses_data( __( "Social network's name. If empty - icon's name will be used", 'trx_addons' ) ),
							'class' => 'trx_addons_column-2_6',
							'std' => '',
							'type' => 'text'
						),
						'url' => array(
							'title' => esc_html__( 'URL to sharer', 'trx_addons' ),
							'desc' => wp_kses_data( __( 'Specify URL to share your posts in this network', 'trx_addons' ) ),
							'class' => 'trx_addons_column-2_6',
							'std' => '',
							'type' => 'text'
						),
						'color' => array(
							"title" => esc_html__('Color', 'trx_addons'),
							"desc" => wp_kses_data( __("Color of the item", 'trx_addons') ),
							"class" => "trx_addons_column-1_6",
							"std" => "",
							"type" => "color"
						),
					)
				),

				// Reactions
				'bp_social_network_socials_section_reactions' => array(
					'title' => esc_html__( 'Reactions', 'trx_addons' ),
					'icon' => 'trx_addons_icon-thumbs-up',
					'type' => 'section'
				),
				'bp_social_network_reactions_info' => array(
					'title' => esc_html__( 'Reactions', 'trx_addons' ) . trx_addons_get_theme_doc_link( '#themerex_addons_reactions' ),
					'desc' => wp_kses_data( __( 'Create a set of reactions to engage more user interactions.', 'trx_addons' ) ),
					'type' => 'info'
				),
				'bp_social_network_reactions_allowed' => array(
					'title' => esc_html__( 'Allow extended reactions', 'trx_addons' ),
					'desc' => wp_kses_data( __( 'Allow extended reactions or use simple likes counter', 'trx_addons' ) ),
					'std' => '0',
					'type' => 'switch'
				),
				'bp_social_network_reactions' => array(
					'title' => esc_html__( 'Reactions List', 'trx_addons' ),
					'desc' => wp_kses(
								__( "Clone the fields group or press the 'Add New Item' button to add a new reaction.", 'trx_addons' )
									. '<br>' . __( "Select the image for this reaction, and finally, add a title if you want it to be displayed.", 'trx_addons' )
									. '<br>' . __( "Please note that choosing an image for reaction will override the icon if chosen.", 'trx_addons' ),
								'trx_addons_kses_content'
							),
					'translate' => true,
					'clone' => true,
					'std' => array( array() ),
					'type' => 'group',
					'dependency' => array(
						'bp_social_network_reactions_allowed' => array( '1' )
					),
					'fields' => array(
						'name' => array(
							'title' => esc_html__( 'Icon', 'trx_addons' ),
							'desc' => wp_kses_data( __( 'Select icon of this reaction', 'trx_addons' ) ),
							'class' => 'trx_addons_column-1_6 trx_addons_new_row',
							'std' => '',
							'options' => array(),
							'style' => trx_addons_get_setting('icons_type' ),
							'type' => 'icons'
						),
						'color' => array(
							"title" => esc_html__('Color', 'trx_addons'),
							"desc" => wp_kses_data( __("Color of the item", 'trx_addons') ),
							"class" => "trx_addons_column-1_6",
							"std" => "",
							"type" => "color"
						),
						'image' => array(
							'title' => esc_html__( 'Image', 'trx_addons' ),
							'desc' => wp_kses_data( __('Select image of this reaction', 'trx_addons') ),
							'class' => 'trx_addons_column-1_6',
							'std' => '',
							'type' => 'image'
						),
						'title' => array(
							'title' => esc_html__( 'Title', 'trx_addons' ),
							'desc' => wp_kses_data( __( "Reaction's name. If empty - icon's name will be used", 'trx_addons' ) ),
							'class' => 'trx_addons_column-3_6',
							'std' => '',
							'type' => 'text'
						),
					)
				),
				'bp_social_network_section_end' => array(
					'type' => 'panel_end'
				)
			) ) );

			return $options;
		}

		public function trx_addons_bp_social_network_load_scripts_front () {
			if ( trx_addons_is_on( trx_addons_get_option( 'debug_mode' ) ) ) {
				wp_enqueue_style( 'trx_addons_bp_social_network', trx_addons_get_file_url( TRX_ADDONS_PLUGIN_ADDONS . 'bp-social-network/bp-social-network.css' ), array(), null );
				wp_enqueue_script( 'trx_addons_bp_social_network', trx_addons_get_file_url( TRX_ADDONS_PLUGIN_ADDONS . 'bp-social-network/bp-social-network.js' ), array( 'jquery' ), null, true );
			}
		}

		public function trx_addons_bp_social_network_merge_styles ( $list ) {
			$list[ TRX_ADDONS_PLUGIN_ADDONS . 'bp-social-network/bp-social-network.css' ] = true;
			return $list;
		}

		public function trx_addons_bp_social_network_merge_scripts ( $list ) {
			$list[ TRX_ADDONS_PLUGIN_ADDONS . 'bp-social-network/bp-social-network.js' ] = true;
			return $list;
		}

		public function trx_addons_bp_social_network_add_shortcode_class ( $classes ) {
			$classes .= ' trx_addons_BP_Social_Network_activity_shortcode';
			return $classes;
		}

		// Reaction AJAX Operations
		//=====================================
		public function ajax_add_reaction () {

			trx_addons_verify_nonce();
			$response = array( 'error' => '' );

			if ( ! isset( $_POST['post_id'] ) || empty( $_POST['post_id'] ) ) {
				$response['error'] = esc_html__( 'Not Found post_id!', 'trx_addons' );
				trx_addons_ajax_response( $response );
			}

			if ( ! isset( $_POST['user_id'] ) || empty( $_POST['user_id'] ) ) {
				$response['error'] = esc_html__( 'Not Found user_id!', 'trx_addons' );
				trx_addons_ajax_response( $response );
			}

			if ( ! isset( $_POST['reaction'] ) ) {
				$response['error'] = esc_html__( 'Not Found reaction!', 'trx_addons' );
				trx_addons_ajax_response( $response );
			}

			$this->set_post_default_reactions_options( array(
				'post_id' => (int) $_POST['post_id'],
				'user_id' => (int) $_POST['user_id'],
			) );

			$this->remove_reactions_activity_user();

			$list_reactions_slug_index = array_search( $_POST['reaction'], array_column( $this->options['list_reactions'], 'slug' ) );
			if ( false !== $list_reactions_slug_index ) {

				$this->options['meta_reactions'][] = array(
					'user_id' => $this->options['user_id'],
					'slug' => $_POST['reaction'],
					'date' => time(),
				);
			}

			$this->set_post_option_post_reactions();

			$this->set_post_option_likes();

			bp_activity_update_meta( $this->options['post_id'], 'trx_addons_bp_social_network_post_reactions', $this->options['meta_reactions'] );

			$response['reaction_content'] = apply_filters( 'trx_addons_bp_social_network_post_reactions_ajax_add_reaction_content', $this->get_post_likes_content(), $this->options );

			$response['history'] = apply_filters( 'trx_addons_bp_social_network_post_reactions_ajax_add_history', $this->get_post_likes_history_list(), $this->options );

			trx_addons_ajax_response( $response );
		}

		public function ajax_remove_reaction () {

			trx_addons_verify_nonce();
			$response = array( 'error' => '' );

			if ( ! isset( $_POST['post_id'] ) || empty( $_POST['post_id'] ) ) {
				$response['error'] = esc_html__( 'Not Found post_id!', 'trx_addons' );
				trx_addons_ajax_response( $response );
			}

			if ( ! isset( $_POST['user_id'] ) || empty( $_POST['user_id'] ) ) {
				$response['error'] = esc_html__( 'Not Found user_id!', 'trx_addons' );
				trx_addons_ajax_response( $response );
			}

			if ( ! isset( $_POST['reaction'] ) ) {
				$response['error'] = esc_html__( 'Not Found reaction!', 'trx_addons' );
				trx_addons_ajax_response( $response );
			}

			$this->set_post_default_reactions_options( array(
				'post_id' => (int) $_POST['post_id'],
				'user_id' => (int) $_POST['user_id'],
			) );

			$this->remove_reactions_activity_user();

			$this->set_post_option_post_reactions();

			$this->set_post_option_likes();

			bp_activity_update_meta( $this->options['post_id'], 'trx_addons_bp_social_network_post_reactions', $this->options['meta_reactions'] );

			$response['reaction_content'] = apply_filters( 'trx_addons_bp_social_network_post_reactions_ajax_remove_reaction_content', $this->get_post_likes_content(), $this->options );

			$response['history'] = apply_filters( 'trx_addons_bp_social_network_post_reactions_ajax_remove_history', $this->get_post_likes_history_list(), $this->options );

			trx_addons_ajax_response( $response );
		}

		// Reaction Settings
		//=====================================
		private function remove_reactions_activity_user () {

			if ( ! $this->options['meta_reactions'] ) {
				return false;
			}

			foreach ( $this->options['meta_reactions'] as $reaction_key => $reaction_item ) {
				if ( $this->options['user_id'] === $reaction_item['user_id'] ) {
					unset( $this->options['meta_reactions'][$reaction_key] );
					break;
				}
			}
		}

		private function set_post_default_reactions_options ( $data = array() ) {

			$this->options = apply_filters( 'trx_addons_bp_social_network_post_reactions_default_options', array_merge( array(
				'post_id'             => null,
				'count_post_likes'    => 0,
				'reactions_allowed'   => trx_addons_is_on( trx_addons_get_option( 'bp_social_network_reactions_allowed' ) ),
				'list_reactions'      => array(),
				'post_reactions'      => array(),
				'meta_reactions'      => array(),
				'user_id'             => null,
				'user_reactions'      => array(),
			), $data ) );

			if ( ! $this->options['post_id'] ) {
				$this->options['post_id'] = bp_get_activity_comment_id();

				if ( ! $this->options['post_id'] ) {
					$this->options['post_id'] = bp_get_activity_id();
				}
			}

			if ( ! $this->options['user_id'] ) {
				$this->options['user_id'] = get_current_user_id();
			}

			if ( ! $this->options['post_id'] || ! $this->options['user_id'] ) {
				return false;
			}

			$list_reactions = array();

			if ( $this->options['reactions_allowed'] ) {
				$list_reactions = trx_addons_get_option( 'bp_social_network_reactions' );
			}

			array_unshift( $list_reactions, array(
				'name' => '',
				'title' => esc_html( 'Like', 'trx_addons' ),
				'slug' => '',
				'style' => 'svg',
				'image' => trx_addons_get_svg_from_file( trx_addons_get_file_dir( TRX_ADDONS_PLUGIN_ADDONS . 'bp-social-network/images/like-icon.svg' ) ),
			) );

			if ( ! empty( $list_reactions ) ) {
				foreach ( $list_reactions as $reaction_item ) {

					$reaction_time_item = $reaction_item;

					if ( 'none' == $reaction_time_item['name'] ) {
						$reaction_time_item['name'] = '';
					}

					if ( ! isset( $reaction_item['slug'] ) || '' !== $reaction_item['slug'] ) {
						if ( ! empty( $reaction_item['name'] ) ) {

							if ( ! trx_addons_is_url( $reaction_item['name'] ) ) {
								$reaction_time_item['finename'] = str_replace( array( 'icon-', 'trx_addons_icon-' ), '', $reaction_item['name'] );
							} else {
								$reaction_time_item['finename'] = trx_addons_get_file_name( $reaction_item['name'] );
							}

							if ( trx_addons_is_url( $reaction_item['image'] ) ) {
								$reaction_time_item['style'] = 'images';
							} else {
								$reaction_time_item['style'] = 'icons';
							}

							if ( empty( $reaction_item['title'] ) ) {
								$reaction_time_item['title'] = $reaction_time_item['finename'];
							}

						} elseif ( ! empty( $reaction_item['title'] ) && ! empty( $reaction_item['image'] ) ) {

							$reaction_time_item['style'] = 'images';

						} else {

							continue;

						}

						if ( ! isset( $reaction_time_item['slug'] ) ) {
							$reaction_time_item['slug'] = strtolower( sanitize_title( $reaction_time_item['title'] ) );
						} 

						if ( ! empty( $reaction_item['image'] ) ) {
							$reaction_time_item['image'] = trx_addons_get_attachment_url( $reaction_item['image'], 'thumbnail' );
						} else {
							$reaction_time_item['image'] = '';
						}
					}

					$reaction_time_item['count'] = 0;
					$reaction_time_item['users'] = array();

					$this->options['list_reactions'][] = $reaction_time_item;
				}
			}

			$post_reactions = bp_activity_get_meta( $this->options['post_id'], 'trx_addons_bp_social_network_post_reactions', true );

			if ( $post_reactions == '' ) {
				bp_activity_delete_meta( $this->options['post_id'], 'trx_addons_bp_social_network_post_reactions' );
				bp_activity_add_meta( $this->options['post_id'], 'trx_addons_bp_social_network_post_reactions', array() );
			}

			if (  is_array( $post_reactions ) && ! empty( $post_reactions ) ) {
				foreach ( $post_reactions as $reaction_item ) {

					if ( ! isset( $reaction_item['user_id'] )  ) {
						continue;
					}

					$this->options['meta_reactions'][] = $reaction_item;
				}
			}
		}

		private function set_post_option_post_reactions () {

			if ( ! $this->options['meta_reactions'] ) {
				return false;
			}

			foreach ( $this->options['meta_reactions'] as $reaction_key => $reaction_item ) {

				$list_reactions_slug_index = array_search( $reaction_item['slug'], array_column( $this->options['list_reactions'], 'slug' ) );

				if ( false === $list_reactions_slug_index ) {
					continue;
				}

				$reaction_time_item = $reaction_item;
				$post_reaction_slug_index = array_search( $reaction_item['slug'], array_column( $this->options['post_reactions'], 'slug' ) );

				if ( $this->options['reactions_allowed'] || '' === $reaction_item['slug'] ) {
					if ( false === $post_reaction_slug_index ) {

						$reaction_time_item['count'] = 1;
						$reaction_time_item['users'] = array( $reaction_item['user_id'] );
						$reaction_time_item = array_merge( $this->options['list_reactions'][$list_reactions_slug_index], $reaction_time_item );

						$this->options['post_reactions'][] = $reaction_time_item;
						$this->options['list_reactions'][$list_reactions_slug_index] = $reaction_time_item;
					} else {

						$this->options['post_reactions'][$post_reaction_slug_index]['count']++;
						$this->options['post_reactions'][$post_reaction_slug_index]['users'][] = $reaction_item['user_id'];

						$this->options['list_reactions'][$list_reactions_slug_index]['count']++;
						$this->options['list_reactions'][$list_reactions_slug_index]['users'][] = $reaction_item['user_id'];
					}
				}
			}
		}

		private function set_post_option_likes () {

			if ( ! $this->options['post_reactions'] ) {
				return false;
			}

			$this->options['count_post_likes'] = 0;

			foreach ( $this->options['post_reactions'] as $reaction_item ) {
				if ( $this->options['reactions_allowed'] || '' === $reaction_item['slug'] ) {
					$this->options['count_post_likes'] += $reaction_item['count'];

					if ( in_array( $this->options['user_id'], $reaction_item['users'] ) ) {
						$this->options['user_reactions'][] = $reaction_item;
					}
				}
			}
		}

		private function get_count_post_likes_k_format ( $count ) {

			if ( ! $count ) {
				return 0;
			}

			return $count >= 1000 ? round( $count / 1000 ) . 'K' : $count;
		}

		// Reaction Renders
		//=====================================
		private function render_post_reactions_content_wrapper_start () {

			$wrapper_class = array( 
				'trx_addons_bp_social_network_post_reactions-wrapper',
				'trx_addons_bp_social_network_post_reactions-wrapper-js',
			);

			?>
				<div 
					class="<?php echo implode( ' ', $wrapper_class ); ?>"
					data-post_id ="<?php echo esc_attr( $this->options['post_id'] ); ?>"
					data-user_id ="<?php echo esc_attr( $this->options['user_id'] ); ?>"
				>
			<?php
		}

		private function render_post_reactions_content_wrapper_end () {
			?>
				</div>
			<?php
		}

		private function render_post_reactions_content_btn_likes () {

			$is_reacted = ! empty( $this->options['user_reactions'] );

			if ( $is_reacted ) {
				$reaction = end( $this->options['user_reactions'] );
			} else {
				$reaction = $this->options['list_reactions'][0];
			}

			if ( empty( $reaction ) ) {
				return false;
			}

			$btn_class = array(
				'trx_addons_bp_social_network_post_reaction_btn',
				'trx_addons_bp_social_network_post_reaction_btn-js',
			);

			$btn_icon = '';

			if ( '' == $reaction['slug'] && $is_reacted ) {

				$reaction['image'] = trx_addons_get_svg_from_file( trx_addons_get_file_dir( TRX_ADDONS_PLUGIN_ADDONS . 'bp-social-network/images/like-active-icon.svg' ) );
			} else {

				$btn_class[] = 'trx_addons_bp_social_network_post_reaction_btn_not_default';

				if ( ! empty( $reaction['slug'] ) ) {
					$btn_class[] = 'trx_addons_bp_social_network_post_reactions_btn_item_icon-' . esc_attr( $reaction['slug'] );
				}

				$btn_icon = $reaction['name'];
			}

			?>
				<a 
					href="javascript:void(0);"
					role="button"
					class="<?php echo implode( ' ', $btn_class ); ?>"
					data-is_reacted="<?php echo esc_attr( $is_reacted ); ?>"
					data-reaction="<?php echo esc_attr( $reaction['slug'] ); ?>"
					<?php echo ( ! empty( $reaction['color'] ) && empty( $reaction['image'] ) ) ? ' style="--var-bp_social_network-reaction-color: ' . $reaction['color'] . '"' : ''; ?>
				>
					<span class="trx_addons_bp_social_network_post_reaction_image-outer">
						<?php $this->render_reaction_image( array( 'src' => $reaction['image'], 'style' => $reaction['style'], 'icon' => $btn_icon ) ); ?>
					</span>
					<span class="trx_addons_bp_social_network_post_reactions_label"><?php echo esc_attr( $reaction['title'] ); ?></span>
					<?php if ( ! $this->options['reactions_allowed'] && $is_reacted ) { ?>
						<span class="trx_addons_bp_social_network_post_reactions_count"><?php echo "&nbsp;" . $this->get_count_post_likes_k_format( $this->options['count_post_likes'] ); ?></span>
					<?php } ?>
				</a>
			<?php
		}

		private function render_post_likes_content_reaction_item () {

			if ( ! isset( $this->options['current_reaction']['slug']  ) ) {
				return false;
			}

			$btn_icon = '';

			$reaction_class = array(
				'trx_addons_bp_social_network_post_reactions_item_btn',
				'trx_addons_bp_social_network_post_reactions_item_btn-js',
				'trx_addons_bp_social_network_post_reactions_item_btn_icon-' . esc_attr( $this->options['current_reaction']['slug'] ),
				'sc_icon_type_' . esc_attr( $this->options['current_reaction']['style'] ),
			);

			if ( '' === $this->options['current_reaction']['slug'] ) {
				$reaction_class[] = 'trx_addons_bp_social_network_post_reactions_item_btn_prime';
			} else if ( 'icons' == $this->options['current_reaction']['style'] && ! $this->options['current_reaction']['image'] ) {
				$btn_icon = $this->options['current_reaction']['name'];
			}

			?>
				<div class="trx_addons_bp_social_network_post_reactions_item">
					<a 
						href="javascript:void(0);"
						role="button"
						class="<?php echo esc_attr( implode( ' ', $reaction_class ) ); ?>"
						data-is_reacted="<?php echo in_array( $this->options['user_id'], $this->options['current_reaction']['users'] ); ?>"
						data-reaction="<?php echo esc_attr( $this->options['current_reaction']['slug'] ); ?>"
						<?php echo ( ! empty( $this->options['current_reaction']['color'] ) && empty( $this->options['current_reaction']['image'] ) ) ? ' style="--var-bp_social_network-reaction-color: ' . $this->options['current_reaction']['color'] . '"' : ''; ?>
					>
						<span class="trx_addons_bp_social_network_post_reaction_image-outer">
							<?php $this->render_reaction_image( array( 'src' => $this->options['current_reaction']['image'], 'style' => $this->options['current_reaction']['style'], 'icon' => $btn_icon ) ); ?>
						</span>
						<span class="trx_addons_bp_social_network_post_reactions_item_btn-label"><?php echo esc_html( $this->options['current_reaction']['title'] ); ?></span>
					</a>
				</div>
			<?php
		}

		private function render_post_reactions_content_reaction_list () {

			if ( ! $this->options['reactions_allowed'] || ! $this->options['list_reactions'] || count( $this->options['list_reactions'] ) <= 1 ) {
				return false;
			}

			?>
				<div class="trx_addons_bp_social_network_post_list_reactions">
			<?php

			foreach ( $this->options['list_reactions'] as $reaction_key => $reaction_item ) {

				$this->options['current_reaction'] = $reaction_item;
				$this->render_post_likes_content_reaction_item();
			}

			unset( $this->options['current_reaction'] );

			?>
				</div>
			<?php
		}

		private function render_post_likes_history_reaction_item () {

			if ( ! isset( $this->options['current_reaction']['slug'] ) || ! $this->options['current_reaction']['users'] ) {
				return false;
			}

			$reaction_class = array(
				'trx_addons_bp_social_network_post_history_reaction_item_btn',
				'trx_addons_bp_social_network_post_history_reaction_item_btn-js',
				'sc_icon_type_' . esc_attr( $this->options['current_reaction']['style'] ),
			);

			if ( ! empty( $this->options['current_reaction']['slug'] ) ) {
				$reaction_class[] = 'trx_addons_bp_social_network_post_history_reaction_item_btn_icon-' . esc_attr( $this->options['current_reaction']['slug'] );
			}

			$btn_icon = '';

			if ( '' === $this->options['current_reaction']['slug'] ) {

				$reaction_class[] = 'trx_addons_bp_social_network_post_history_reaction_item_btn_prime';
				$this->options['current_reaction']['image'] = trx_addons_get_svg_from_file( trx_addons_get_file_dir( TRX_ADDONS_PLUGIN_ADDONS . 'bp-social-network/images/like-active-icon.svg' ) );

			} else if ( 'icons' == $this->options['current_reaction']['style'] && ! $this->options['current_reaction']['image'] ) {
				$btn_icon = $this->options['current_reaction']['name'];
			}

			$usernames = array();
			foreach ( $this->options['current_reaction']['users'] as $user_id ) {

				$username = bp_core_get_user_displayname( $user_id );
				if ( $username ) {
					$usernames[] = $username;
				}

			}

			?>
				<div class="trx_addons_bp_social_network_post_history_reaction_item">
					<a 
						href="javascript:void(0);"
						role="button"
						class="<?php echo esc_attr( implode( ' ', $reaction_class ) ); ?>"
						data-is_reacted="<?php echo in_array( $this->options['user_id'], $this->options['current_reaction']['users'] ); ?>"
						data-reaction="<?php echo esc_attr( $this->options['current_reaction']['slug'] ); ?>"
						<?php echo ( ! empty( $this->options['current_reaction']['color'] ) && empty( $this->options['current_reaction']['image'] ) ) ? ' style="--var-bp_social_network-reaction-color: ' . $this->options['current_reaction']['color'] . '"' : ''; ?>
					>
						<span class="trx_addons_bp_social_network_post_reaction_image-outer">
							<?php $this->render_reaction_image( array( 'src' => $this->options['current_reaction']['image'], 'style' => $this->options['current_reaction']['style'], 'icon' => $btn_icon ) ); ?>
							<span class="trx_addons_bp_social_network_post_history_reaction_item_username">
								<?php echo esc_html__( sprintf(
									'%s reacted with: ' . $this->options['current_reaction']['title'] . ':',
									implode( ', ', $usernames )
								), 'trx_addons' ); ?>
							</span>
						</span>
						<span class="trx_addons_bp_social_network_post_reactions_number"><?php echo $this->get_count_post_likes_k_format( $this->options['current_reaction']['count'] ); ?></span>
					</a>
				</div>
			<?php
		}

		private function render_reaction_image ( $image = array() ) {

			if ( ! $image['icon'] && ( ! isset( $image['src'] ) || empty( $image['src'] ) ) ) {
				return '';
			}

			$is_svg = 'svg' == $image['style'];

			$image = array_merge( array(
				'src'    => null,
				'class'  => array(),
			), $image );

			$image['class'][] = 'trx_addons_bp_social_network_post_reaction_image';

			if ( $image['icon'] && empty( $image['src'] ) ) {
				$image['class'][] = 'sc_icon_type_icons';
				$image['class'][] = $image['icon'];
			} elseif ( $is_svg ) {
				$image['class'][] = 'trx_addons_bp_social_network_post_reaction_image-svg';
			} else {
				$image['class'][] = 'trx_addons_bp_social_network_post_reaction_image-image';
			}

			unset( $image['style'] );
			unset( $image['icon'] );

			$image_attrs = '';

			foreach ( $image as $image_attr_key => $image_attr_val ) {

				if ( 'src' === $image_attr_key ) {
					if ( ! empty( $image_attr_val ) && ! $is_svg ) {
						$image_attrs .= ' style="--var-trx_addons_bp_social_network_post_reaction_image-url: url(' . esc_attr( $image_attr_val ) . ');"';
					}
					continue;
				}

				if ( 'class' === $image_attr_key && ! empty( $image_attr_val ) ) {
					$image_attr_val = implode( ' ', $image_attr_val );
				}

				$image_attrs .= ' ' . $image_attr_key . '="' . esc_attr( $image_attr_val ) . '"';
			}

			?>
				<span <?php echo $image_attrs; ?>>
					<?php if ( $is_svg ) { ?>
						<?php echo $image['src']; ?>
					<?php } ?>
				</span>
			<?php
		}

		// Reaction Outputs
		//=====================================
		private function get_post_likes_history_list () {

			if ( ! $this->options['reactions_allowed'] || ! $this->options['post_reactions'] || ! $this->options['list_reactions'] || count( $this->options['list_reactions'] ) <= 1 ) {
				return '';
			}

			$wrapper_class = array( 
				'trx_addons_bp_social_network_post_history_reactions',
				'trx_addons_bp_social_network_post_history_reactions-js',
			);

			ob_start();

			?>
				<div
					class="<?php echo implode( ' ', $wrapper_class ); ?>"
				>
			<?php

			foreach ( $this->options['post_reactions'] as $reaction_key => $reaction_item ) {

				if ( '' !== $reaction_item['slug'] && ! $this->options['reactions_allowed'] ) {
					continue;
				}

				$this->options['current_reaction'] = $reaction_item;
				$this->render_post_likes_history_reaction_item();
			}

			unset( $this->options['current_reaction'] );

			?>
				</div>
			<?php

			$content = ob_get_clean();

			return apply_filters( 'trx_addons_bp_social_network_post_reactions_history', $content, $this->options );
		}

		private function get_post_likes_content () {

			ob_start();

			$this->render_post_reactions_content_wrapper_start();

			$this->render_post_reactions_content_btn_likes();

			$this->render_post_reactions_content_reaction_list();

			$this->render_post_reactions_content_wrapper_end();

			$content = ob_get_clean();

			return apply_filters( 'trx_addons_bp_social_network_post_reactions_content', $content, $this->options );
		}

		private function render_post_reactions_content () {

			$this->set_post_default_reactions_options();

			$this->set_post_option_post_reactions();

			$this->set_post_option_likes();

			trx_addons_show_layout( $this->get_post_likes_content() );

			trx_addons_show_layout( $this->get_post_likes_history_list() );
		}

		public function add_reaction_bp_activity_entry_meta () {
			if ( is_user_logged_in() ) {
				$this->render_post_reactions_content();
			}
		}

		public function add_reaction_bp_activity_comment_entry_meta () {
			if ( is_user_logged_in() ) {
				$this->render_post_reactions_content();
			}
		}

		// Shares Settings
		//=====================================
		private function set_post_default_share_options (  ) {

			$this->options = apply_filters( 'trx_addons_bp_social_network_post_shares_default_options', array(
				'post_id'             => null,
				'user_id'             => null,
			) );

			if ( ! $this->options['post_id'] ) {
				$this->options['post_id'] = bp_get_activity_id();
			}

			if ( ! $this->options['user_id'] ) {
				$this->options['user_id'] = bp_get_activity_user_id();
			}

			if ( ! $this->options['post_id'] || ! $this->options['user_id'] ) {
				return false;
			}

			$this->options['post_link'] = bp_activity_get_permalink( $this->options['post_id'] );
			$this->options['post_title'] = bp_core_get_user_displayname( $this->options['post_id'] );
			$this->options['post_descr'] = bp_activity_get_meta( $this->options['post_id'], 'bp_activity_text' );
			$this->options['post_thumb'] = bp_core_fetch_avatar( array( 
				'item_id' => $this->options['user_id'],
				'html' => false,
			) );
		}

		public function trx_addons_filter_social_sharing_attr ( $params, $social ) {

			if (
				isset( $social['bp_social_network'] ) && ! empty( $social['bp_social_network'] )
				&& ! empty( $social['color'] )
				&& ( ! empty( $params ) || strpos( $params, 'style=' ) === false )
			) {
				$params .= ' style="--var-bp_social_network-share-color: ' . $social['color'] . ';"';
			}

			return $params;
		}

		private function get_post_share_list_content () {

			if ( ! $this->options['post_id'] || ! $this->options['user_id'] ) {
				return '';
			}

			$shared_list = trx_addons_get_option( 'bp_social_network_share' );

			if ( ! empty( $shared_list ) ) {
				foreach ( $shared_list as &$item ) {
					$item['bp_social_network'] = true;
				}
			}

			ob_start();

			?>
				<span class="trx_addons_bp_social_network_post_share_list-wrapper">
					<?php 
						trx_addons_get_share_links( array_merge( $this->options, array(
							'type' => 'drop',
							'popup' => true,
							'caption' => esc_html__( 'Share', 'trx_addons' ),
							'wrap' => 'div',
							'echo' => true,
						) ), $shared_list );
					?>
				</span>
			<?php

			$content = ob_get_clean();

			return apply_filters( 'trx_addons_bp_social_network_post_share_list_content', $content, $this->options );
		}

		private function render_post_share_list_content () {

			$this->set_post_default_share_options();

			trx_addons_show_layout( $this->get_post_share_list_content() );
		}

		public function add_share_list_bp_activity_entry_meta () {
			if ( is_user_logged_in() ) {
				$this->render_post_share_list_content();
			}
		}
	}

	new TRX_Addons_BP_Social_Network_Activity();
}