(function ($) {
  "use strict";

  // Check for required global variables
  if (typeof felan_template_vars === 'undefined' || typeof felan_payout_vars === 'undefined') {
    console.error('Required global variables (felan_template_vars, felan_payout_vars) are not loaded.');
    return; // Exit early to prevent errors
  }

  var ajax_url = felan_template_vars.ajax_url || '',
    custom_field_payout = felan_payout_vars.custom_field_payout || [];

  $(document).ready(function () {
    initializePayoutForm();
    initializeSelect2();
    initializeEventHandlers();
    updateFieldRequirements();
  });

  function initializePayoutForm() {
    $(".felan-payout-dashboard li.payout-item").each(function () {
      var value = $(this).attr("id");
      $("." + value).appendTo("#" + value + " .content");

      // Hide content for disabled payment methods
      var $checkbox = $(this).find('input.payment-enable');
      if ($checkbox.length > 0 && !$checkbox.is(':checked')) {
        $(this).find('.content').hide();
      }
    });
  }

  function initializeSelect2() {
    setTimeout(function () {
      $('.felan-payout-dashboard .felan-select2').each(function () {
        var $this = $(this);
        var $formGroup = $this.closest('.form-group');

        if ($this.hasClass('select2-hidden-accessible')) {
          $this.select2('destroy');
        }

        try {
          $this.select2({
            placeholder: 'Select Country',
            allowClear: true,
            width: '100%',
            dropdownAutoWidth: false,
            dropdownParent: $formGroup.length ? $formGroup : $this.parent()
          });
        } catch (e) {
          // Do nothing
        }
      });
    }, 500);
  }

  function initializeEventHandlers() {
    $("#payout-bank-country").on("change", function () {
      clearAllErrors();
      updateFieldRequirements();
    });

    $('#payout-bank-account-holder, #payout-bank-name, #payout-bank-account-number, #payout-bank-address, #payout-bank-iban, #payout-bank-swift-bic, #payout-bank-routing-number, #payout-paypal, #payout-stripe').on('input', function () {
      clearFieldError($(this));
    });

    // Payment enable/disable toggles
    $('.payment-enable').on('change', function () {
      var $checkbox = $(this);
      var $payoutItem = $checkbox.closest('.payout-item');
      var $content = $payoutItem.find('.content');
      var isEnabled = $checkbox.is(':checked');

      if (isEnabled) {
        $content.slideDown(300);
      } else {
        $content.slideUp(300);
        // Clear all fields in this payment method when disabled
        $content.find('input, select').val('');
        clearAllErrors();
      }
    });

    $(document).on("click", "#btn-submit-payout", handleFormSubmit);
  }

  function handleFormSubmit(e) {
    e.preventDefault();

    try {
      if (typeof felan_payout_vars === 'undefined') {
        throw new Error('Required variables are not loaded. Please refresh the page and try again.');
      }

      var form_payout = $(".felan-payout-dashboard");
      if (form_payout.length === 0) {
        throw new Error('Payout form not found on page.');
      }
      clearAllErrors();

      var formData = {
        payout_paypal: form_payout.find('input[name="payout_paypal"]').val(),
        payout_stripe: form_payout.find('input[name="payout_stripe"]').val(),
        payout_bank_country: form_payout.find('select[name="payout_bank_country"]').val(),
        payout_bank_account_holder: form_payout.find('input[name="payout_bank_account_holder"]').val(),
        payout_bank_name: form_payout.find('input[name="payout_bank_name"]').val(),
        payout_bank_account_number: form_payout.find('input[name="payout_bank_account_number"]').val(),
        payout_bank_address: form_payout.find('textarea[name="payout_bank_address"]').val(),
        payout_bank_iban: form_payout.find('input[name="payout_bank_iban"]').val(),
        payout_bank_swift_bic: form_payout.find('input[name="payout_bank_swift_bic"]').val(),
        payout_bank_routing_number: form_payout.find('input[name="payout_bank_routing_number"]').val()
      };

      // Check which payment methods are enabled
      var paypalEnabled = form_payout.find('input[name="enable_paypal"]').is(':checked');
      var stripeEnabled = form_payout.find('input[name="enable_stripe"]').is(':checked');
      var bankEnabled = form_payout.find('input[name="enable_bank"]').is(':checked');

      if (!paypalEnabled && !stripeEnabled && !bankEnabled) {
        var errorMessage = felan_payout_vars.enable_payment_method || 'Please enable at least one payment method.';
        showGeneralError(errorMessage);
        form_payout.find(".btn-loading").fadeOut();
        return false;
      }

      var validationErrors = [];

      if (paypalEnabled) {
        if (!formData.payout_paypal || formData.payout_paypal.trim() === '') {
          validationErrors.push({
            field: 'paypal',
            message: felan_payout_vars.paypal_required
          });
        } else if (!isValidEmail(formData.payout_paypal)) {
          validationErrors.push({
            field: 'paypal',
            message: felan_payout_vars.paypal_invalid_email
          });
        }
      }

      if (stripeEnabled) {
        if (!formData.payout_stripe || formData.payout_stripe.trim() === '') {
          validationErrors.push({
            field: 'stripe',
            message: felan_payout_vars.stripe_required
          });
        }
      }

      if (bankEnabled) {
        var bankErrors = validateBankTransferFields(formData);
        validationErrors = validationErrors.concat(bankErrors);
      }

      var custom_field = {};
      var customFieldErrors = [];

      $.each(custom_field_payout, function (index, value) {
        if (value.id) {
          var fieldValue = $("input[name=" + value.id + "]").val();
          custom_field[value.id] = fieldValue;

          var fieldElement = $("input[name=" + value.id + "]");
          if (fieldElement.attr('required') && (!fieldValue || fieldValue.trim() === '')) {
            customFieldErrors.push({
              field: value.id,
              message: felan_payout_vars.custom_field_required + ' ' + (value.label || value.name) + '.'
            });
          }
        }
      });

      if (customFieldErrors.length > 0) {
        validationErrors = validationErrors.concat(customFieldErrors);
      }

      if (validationErrors.length > 0) {
        showErrors(validationErrors);
        form_payout.find(".btn-loading").fadeOut();
        return false;
      }

      submitPayoutData(formData, custom_field, paypalEnabled, stripeEnabled, bankEnabled);
    } catch (error) {
      console.error('Payout form submission error:', error);
      var errorMessage = felan_payout_vars.unexpected_error || 'An unexpected error occurred while processing your request.';
      if (error.message) {
        errorMessage += ' ' + error.message;
      }
      errorMessage += ' ' + (felan_payout_vars.contact_support || 'Please try again or contact support if the problem persists.');
      showGeneralError(errorMessage);
      var form_payout = $(".felan-payout-dashboard");
      form_payout.find(".btn-loading").fadeOut();
    }
  }

  function validateStripe(stripeValue) {
    if (!stripeValue || stripeValue.trim() === '') {
      showError('stripe', felan_payout_vars.stripe_account_required || felan_payout_vars.stripe_default_required || 'Please enter your Stripe account ID.');
      $('.felan-payout-dashboard').find(".btn-loading").fadeOut();
      return false;
    }
    return true;
  }

  function validateBankTransferFields(formData) {
    var errors = [];

    // Country - Required
    if (!formData.payout_bank_country || formData.payout_bank_country.trim() === '') {
      errors.push({
        field: 'bank-country',
        message: felan_payout_vars.country_required || 'Please select a country.'
      });
    }

    // Account Number - Required
    if (!formData.payout_bank_account_number || formData.payout_bank_account_number.trim() === '') {
      errors.push({
        field: 'bank-account-number',
        message: felan_payout_vars.account_number_required || 'Please enter account number.'
      });
    } else if (formData.payout_bank_account_number.trim().length < 4) {
      errors.push({
        field: 'bank-account-number',
        message: felan_payout_vars.account_number_min_length || 'Account number must be at least 4 characters.'
      });
    } else if (!/^[0-9]+$/.test(formData.payout_bank_account_number.trim())) {
      errors.push({
        field: 'bank-account-number',
        message: felan_payout_vars.account_number_invalid || 'Account number must contain only numbers.'
      });
    }

    // Account Holder - Required
    if (!formData.payout_bank_account_holder || formData.payout_bank_account_holder.trim() === '') {
      errors.push({
        field: 'bank-account-holder',
        message: felan_payout_vars.account_name_required || 'Please enter account holder name.'
      });
    } else if (formData.payout_bank_account_holder.trim().length < 2) {
      errors.push({
        field: 'bank-account-holder',
        message: felan_payout_vars.account_name_min_length || 'Account holder name must be at least 2 characters.'
      });
    }

    // Bank Name - Required
    if (!formData.payout_bank_name || formData.payout_bank_name.trim() === '') {
      errors.push({
        field: 'bank-name',
        message: felan_payout_vars.bank_name_required || 'Please enter bank name.'
      });
    } else if (formData.payout_bank_name.trim().length < 2) {
      errors.push({
        field: 'bank-name',
        message: felan_payout_vars.bank_name_min_length || 'Bank name must be at least 2 characters.'
      });
    }

    // SWIFT/BIC - Required
    if (!formData.payout_bank_swift_bic || formData.payout_bank_swift_bic.trim() === '') {
      errors.push({
        field: 'bank-swift-bic',
        message: felan_payout_vars.swift_required || 'Please enter SWIFT/BIC code.'
      });
    } else if (!isValidSWIFT(formData.payout_bank_swift_bic.trim())) {
      errors.push({
        field: 'bank-swift-bic',
        message: felan_payout_vars.swift_invalid || 'Please enter a valid SWIFT/BIC code (8-11 characters).'
      });
    }

    // Bank Address - Required
    if (!formData.payout_bank_address || formData.payout_bank_address.trim() === '') {
      errors.push({
        field: 'bank-address',
        message: felan_payout_vars.bank_address_required || 'Please enter bank address.'
      });
    } else if (formData.payout_bank_address.trim().length < 10) {
      errors.push({
        field: 'bank-address',
        message: felan_payout_vars.bank_address_min_length || 'Bank address must be at least 10 characters.'
      });
    }

    // IBAN - Required for EU countries
    var euCountries = ['DE', 'FR', 'IT', 'ES', 'NL', 'AT', 'BE', 'FI', 'IE', 'LU', 'PT', 'GR', 'CY', 'MT', 'SK', 'SI', 'EE', 'LV', 'LT'];
    if (euCountries.includes(formData.payout_bank_country)) {
      if (!formData.payout_bank_iban || formData.payout_bank_iban.trim() === '') {
        errors.push({
          field: 'bank-iban',
          message: felan_payout_vars.iban_required || 'IBAN is required for EU countries.'
        });
      } else if (!isValidIBAN(formData.payout_bank_iban.trim())) {
        errors.push({
          field: 'bank-iban',
          message: felan_payout_vars.iban_invalid || 'Invalid IBAN format. Must start with 2-letter country code (e.g., GB29NWBK60161331926819).'
        });
      }
    } else if (formData.payout_bank_iban && formData.payout_bank_iban.trim() !== '') {
      // Validate IBAN format if provided for non-EU countries
      if (!isValidIBAN(formData.payout_bank_iban.trim())) {
        errors.push({
          field: 'bank-iban',
          message: felan_payout_vars.iban_invalid || 'Invalid IBAN format. Must start with 2-letter country code (e.g., GB29NWBK60161331926819).'
        });
      }
    }

    // Routing Number - Required for US
    if (formData.payout_bank_country === 'US') {
      if (!formData.payout_bank_routing_number || formData.payout_bank_routing_number.trim() === '') {
        errors.push({
          field: 'bank-routing-number',
          message: felan_payout_vars.routing_required || 'Routing number is required for US banks.'
        });
      } else if (!/^[0-9]{9}$/.test(formData.payout_bank_routing_number.trim())) {
        errors.push({
          field: 'bank-routing-number',
          message: felan_payout_vars.routing_invalid || 'Please enter a valid 9-digit routing number.'
        });
      }
    } else if (formData.payout_bank_routing_number && formData.payout_bank_routing_number.trim() !== '') {
      // Validate routing number format if provided for non-US countries
      if (!/^[0-9]{9}$/.test(formData.payout_bank_routing_number.trim())) {
        errors.push({
          field: 'bank-routing-number',
          message: felan_payout_vars.routing_invalid || 'Routing number must be exactly 9 digits.'
        });
      }
    }

    return errors;
  }

  function validateBankTransfer(formData) {
    var errors = validateBankTransferFields(formData);

    if (errors.length > 0) {
      showErrors(errors);
      $('.felan-payout-dashboard').find(".btn-loading").fadeOut();
      return false;
    }

    return true;
  }

  function submitPayoutData(formData, custom_field, paypalEnabled, stripeEnabled, bankEnabled) {
    try {
      if (typeof ajax_url === 'undefined' || !ajax_url) {
        throw new Error(felan_payout_vars.ajax_url_undefined || 'AJAX URL is not defined. Please refresh the page and try again.');
      }

      var form = $('.felan-payout-dashboard');

      $.ajax({
      type: "POST",
      dataType: "json",
      url: ajax_url,
      data: {
        action: "felan_update_payout_ajax",
        nonce: felan_payout_vars.nonce,
        enable_paypal: paypalEnabled ? '1' : '0',
        enable_stripe: stripeEnabled ? '1' : '0',
        enable_bank: bankEnabled ? '1' : '0',
        payout_paypal: formData.payout_paypal,
        payout_stripe: formData.payout_stripe,
        payout_bank_country: formData.payout_bank_country,
        payout_bank_account_holder: formData.payout_bank_account_holder,
        payout_bank_name: formData.payout_bank_name,
        payout_bank_account_number: formData.payout_bank_account_number,
        payout_bank_address: formData.payout_bank_address,
        payout_bank_iban: formData.payout_bank_iban,
        payout_bank_swift_bic: formData.payout_bank_swift_bic,
        payout_bank_routing_number: formData.payout_bank_routing_number,
        custom_field: custom_field
      },
        beforeSend: function () {
          form.find(".btn-loading").fadeIn();
        },
        success: function (response) {
          if (response.success) {
            window.location.reload();
          } else {
            showGeneralError(response.message);
          }
          form.find(".btn-loading").fadeOut();
        },
        error: function (xhr, status, error) {
          showGeneralError(felan_payout_vars.ajax_error);
          form.find(".btn-loading").fadeOut();
        }
      });
    } catch (error) {
      console.error('Payout data submission error:', error);
      var errorMessage = felan_payout_vars.submit_failed || 'Failed to submit payout data.';
      if (error.message && error.message !== 'undefined') {
        errorMessage += ' ' + error.message;
      }
      errorMessage += ' ' + (felan_payout_vars.try_again || 'Please try again.');
      showGeneralError(errorMessage);
      $('.felan-payout-dashboard').find(".btn-loading").fadeOut();
    }
  }

  function isValidEmail(email) {
    var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
  }

  function isValidIBAN(iban) {
    var cleanIban = iban.replace(/\s/g, '').toUpperCase();
    return cleanIban.length >= 15 && cleanIban.length <= 34 && /^[A-Z]{2}[0-9]{2}[A-Z0-9]+$/.test(cleanIban);
  }

  function isValidSWIFT(swift) {
    var cleanSwift = swift.replace(/\s/g, '').toUpperCase();
    return cleanSwift.length >= 8 && cleanSwift.length <= 11 && /^[A-Z]{4}[A-Z]{2}[A-Z0-9]{2}([A-Z0-9]{3})?$/.test(cleanSwift);
  }

  function showError(field, message) {
    var errorElement = $('#error-' + field);
    var inputElement = field === 'stripe' ? $('#payout-stripe') : $('#payout-' + field);

    if (errorElement.length && inputElement.length) {
      errorElement.text(message).addClass('show');
      inputElement.closest('.form-group').addClass('error');
      scrollToElement(inputElement);
    }
  }

  function showErrors(errors) {
    var firstError = null;

    errors.forEach(function (error) {
      var errorElement = $('#error-' + error.field);
      var inputElement = $('#payout-' + error.field);

      if (errorElement.length && inputElement.length) {
        errorElement.text(error.message).addClass('show');
        inputElement.closest('.form-group').addClass('error');

        if (!firstError) {
          firstError = inputElement;
        }
      }
    });

    if (firstError) {
      scrollToElement(firstError);
    }
  }

  function clearFieldError(element) {
    var fieldId = element.attr('id');
    if (!fieldId) return;

    // Convert ID format: payout-bank-country -> bank-country
    var errorId = fieldId.replace('payout-', '');
    var errorElement = $('#error-' + errorId);

    if (errorElement.length) {
      errorElement.removeClass('show').text('');
    }
    element.closest('.form-group').removeClass('error');
  }

  function clearAllErrors() {
    $('.error-message').removeClass('show').text('');
    $('.form-group').removeClass('error');
    $('.general-error-message').remove();
  }

  function showGeneralError(message) {
    $('.general-error-message').remove();
    var errorHtml = '<div class="general-error-message" style="color: #ef4444;  margin-bottom: 20px; font-weight: 500;">' + message + '</div>';
    $('.felan-payout-dashboard').before(errorHtml);

    setTimeout(function () {
      $('.general-error-message').fadeOut(300, function () {
        $(this).remove();
      });
    }, 5000);
  }

  function scrollToElement(element) {
    $('html, body').animate({
      scrollTop: element.offset().top - 100
    }, 500);
  }

  function updateFieldRequirements() {
    var selectedCountry = $("#payout-bank-country").val();
    var euCountries = ['DE', 'FR', 'IT', 'ES', 'NL', 'AT', 'BE', 'FI', 'IE', 'LU', 'PT', 'GR', 'CY', 'MT', 'SK', 'SI', 'EE', 'LV', 'LT'];

    var ibanLabel = $('label[for="payout-bank-iban"]');
    var ibanRequiredSpan = ibanLabel.find('.required');

    if (euCountries.includes(selectedCountry)) {
      if (ibanRequiredSpan.length === 0) {
        ibanLabel.append(' <span class="required">*</span>');
      }
      $('#payout-bank-iban').attr('required', true);

      var helpText = ibanLabel.siblings('.help-text');
      if (helpText.length > 0) {
        helpText.text(felan_payout_vars.iban_required_eu);
      }
    } else {
      ibanRequiredSpan.remove();
      $('#payout-bank-iban').removeAttr('required');

      var helpText = ibanLabel.siblings('.help-text');
      if (helpText.length > 0) {
        helpText.text(felan_payout_vars.iban_optional);
      }
    }

    var swiftLabel = $('label[for="payout-bank-swift-bic"]');
    var swiftRequiredSpan = swiftLabel.find('.required');

    if (selectedCountry === 'US') {
      swiftRequiredSpan.remove();
      $('#payout-bank-swift-bic').removeAttr('required');
    } else if (selectedCountry && !euCountries.includes(selectedCountry)) {
      if (swiftRequiredSpan.length === 0) {
        swiftLabel.append(' <span class="required">*</span>');
      }
      $('#payout-bank-swift-bic').attr('required', true);
    } else {
      swiftRequiredSpan.remove();
      $('#payout-bank-swift-bic').removeAttr('required');
    }
  }

})(jQuery);
