/**
 * Project Alerts Main Entry Point
 * Initializes all modules and coordinates between them
 */
(function ($) {
    'use strict';

    if (typeof $ === 'undefined') {
        return;
    }

    var retryCount = 0;
    var MAX_RETRIES = 10;

    function initializeProjectAlerts() {
        var $form = $('#project-alerts-form');
        if (!$form.length) {
            return;
        }

        var vars = typeof felan_project_alerts_vars !== 'undefined' ? felan_project_alerts_vars : {};

        // Check if all required modules are loaded
        var requiredModules = ['ProjectAlertsProgress', 'ProjectAlertsValidation', 'ProjectAlertsForm', 'ProjectAlertsList'];
        var missingModules = requiredModules.filter(function(module) {
            return !window[module];
        });

        if (missingModules.length > 0) {
            if (retryCount < MAX_RETRIES) {
                retryCount++;
                setTimeout(initializeProjectAlerts, 100);
            }
            return;
        }

        retryCount = 0;

        // Initialize all modules in correct order
        if (window.ProjectAlertsProgress && window.ProjectAlertsProgress.init) {
            window.ProjectAlertsProgress.init(vars, $form);
        }
        if (window.ProjectAlertsValidation && window.ProjectAlertsValidation.init) {
            window.ProjectAlertsValidation.init(vars, $form);
        }
        if (window.ProjectAlertsForm && window.ProjectAlertsForm.init) {
            window.ProjectAlertsForm.init(vars, $form);
        }
        if (window.ProjectAlertsList && window.ProjectAlertsList.init) {
            window.ProjectAlertsList.init(vars);
        }
        if (window.ProjectAlertsPackage && window.ProjectAlertsPackage.init) {
            window.ProjectAlertsPackage.init(vars);
        }

        // Handle cancel edit button
        $(document).on('click', '#cancel-edit-btn', function(e) {
            e.preventDefault();
            if (window.ProjectAlertsList && window.ProjectAlertsList.cancelEdit) {
                window.ProjectAlertsList.cancelEdit();
            }
        });

        // Check URL for edit parameter on page load
        var urlParams = new URLSearchParams(window.location.search);
        var editId = urlParams.get('edit');
        if (editId && window.ProjectAlertsList && window.ProjectAlertsList.editAlert) {
            // Use retry mechanism to ensure form is ready
            var editRetryCount = 0;
            var maxEditRetries = 20; // 20 retries = 2 seconds max

            function tryEditAlert() {
                var $formWrapper = $('#project-alerts-form-wrapper');
                var $form = $('#project-alerts-form');

                // Check if form is ready
                if ($form.length && $formWrapper.length &&
                    window.ProjectAlertsForm &&
                    window.ProjectAlertsForm.showForm &&
                    window.ProjectAlertsList.editAlert) {

                    // Form is ready, trigger edit
                    window.ProjectAlertsList.editAlert(parseInt(editId), function() {
                        // Success callback - ensure form is visible
                        if ($formWrapper.is(':hidden')) {
                            window.ProjectAlertsForm.showForm(true);
                        }
                    });
                } else {
                    // Form not ready yet, retry
                    if (editRetryCount < maxEditRetries) {
                        editRetryCount++;
                        setTimeout(tryEditAlert, 100);
                    } else {
                        // Max retries reached, try one more time with longer delay
                        setTimeout(function() {
                            if (window.ProjectAlertsList && window.ProjectAlertsList.editAlert) {
                                window.ProjectAlertsList.editAlert(parseInt(editId));
                            }
                        }, 500);
                    }
                }
            }

            // Start trying after initial delay
            setTimeout(tryEditAlert, 300);
        }
    }

    $(document).ready(function() {
        setTimeout(initializeProjectAlerts, 50);
    });
})(jQuery);
