/**
 * Project Alerts Form Module
 * Handles form submission, show/hide, and form state management
 */
(function ($) {
    'use strict';

    var ProjectAlertsForm = {
        vars: {},
        $form: null,
        ajaxUrl: '',
        nonce: '',

        /**
         * Initialize form module
         */
        init: function(vars, $form) {
            this.vars = vars || {};
            this.$form = $form || $('#project-alerts-form');
            this.ajaxUrl = vars.ajax_url || (typeof ajaxurl !== 'undefined' ? ajaxurl : '/wp-admin/admin-ajax.php');
            this.nonce = vars.nonce || '';

            if (!this.$form.length) {
                return;
            }

            // Initialize Select2
            this.initSelect2();

            // Setup sticky button
            this.setupStickyButton();

            // Auto-generate alert name from email
            this.setupAutoNameGeneration();

            // Clear errors on field change
            this.setupErrorClearing();

            // Form submission
            this.setupFormSubmission();
        },

        /**
         * Initialize Select2
         */
        initSelect2: function() {
            var self = this;
            if ($.fn.select2) {
                this.$form.find('.felan-select2').each(function() {
                    var $select = $(this);
                    var isMultiple = $select.prop('multiple');
                    var placeholder = $select.data('placeholder') || (self.vars.select_option || 'Select an option');

                    // For single select with placeholder, need allowClear: true
                    var select2Options = {
                        width: '100%',
                        placeholder: placeholder,
                        // Ensure Select2 creates inputs with proper attributes
                        minimumResultsForSearch: Infinity // Disable search for better performance and to avoid creating search input without id/name
                    };

                    // For multiple selects, enable search but ensure proper attributes
                    if (isMultiple) {
                        select2Options.minimumResultsForSearch = 0; // Enable search for multiple selects
                    }

                    if (!isMultiple && placeholder) {
                        // Single select with placeholder needs allowClear to show placeholder
                        select2Options.allowClear = true;
                    } else {
                        // Multi-select or single without placeholder
                        select2Options.allowClear = false;
                    }

                    $select.select2(select2Options);

                    // Ensure Select2 elements have proper attributes for accessibility
                    var $select2Container = $select.next('.select2-container');
                    if ($select2Container.length) {
                        // Get the original label for this select
                        var $label = $select.closest('.form-field').find('label[for="' + $select.attr('id') + '"]');
                        var labelText = $label.length ? $label.text().replace(/\s*\(Optional\)\s*/g, '').replace(/\s*\*\s*/g, '').trim() : '';

                        // Ensure Select2 search input has proper attributes if it exists
                        var $searchInput = $select2Container.find('input.select2-search__field');
                        if ($searchInput.length) {
                            if (!$searchInput.attr('id')) {
                                var searchId = $select.attr('id') + '-search';
                                $searchInput.attr('id', searchId);
                            }
                            if (!$searchInput.attr('name')) {
                                $searchInput.attr('name', $searchInput.attr('id'));
                            }
                            if (!$searchInput.attr('autocomplete')) {
                                $searchInput.attr('autocomplete', 'off');
                            }
                            if (!$searchInput.attr('aria-label')) {
                                var searchLabel = labelText ? (self.vars.search || 'Search') + ' ' + labelText : ($select.attr('aria-label') || $select.attr('data-placeholder') || (self.vars.search || 'Search'));
                                $searchInput.attr('aria-label', searchLabel);
                            }
                        }

                        // Ensure Select2 selection has proper attributes
                        var $selection = $select2Container.find('.select2-selection');
                        if ($selection.length && !$selection.attr('aria-labelledby') && $label.length) {
                            $selection.attr('aria-labelledby', $label.attr('id') || $select.attr('id') + '-label');
                        }
                    }

                    // Listen to Select2 events
                    var selectId = $select.attr('id');

                    if (isMultiple) {
                        $select.on('select2:select select2:unselect select2:clear change', function() {
                            var $field = $select.closest('.form-field');
                            if ($field.hasClass('has-error')) {
                                $field.removeClass('has-error');
                                $field.find('.field-error').remove();
                            }
                            if (window.ProjectAlertsProgress && window.ProjectAlertsProgress.updateProgress) {
                                setTimeout(function() {
                                    window.ProjectAlertsProgress.updateProgress();
                                }, 100);
                            }
                        });
                    } else {
                        $select.on('change', function() {
                            var $field = $select.closest('.form-field');
                            if ($field.hasClass('has-error')) {
                                $field.removeClass('has-error');
                                $field.find('.field-error').remove();
                            }
                            if (window.ProjectAlertsProgress && window.ProjectAlertsProgress.updateProgress) {
                                setTimeout(function() {
                                    window.ProjectAlertsProgress.updateProgress();
                                }, 100);
                            }
                        });
                    }
                });
            }
        },

        /**
         * Setup auto-generate alert name from email
         */
        setupAutoNameGeneration: function() {
            var self = this;
            this.$form.find('#alert-email').on('input blur', function() {
                var $field = $(this).closest('.form-field');
                var email = $(this).val();
                var $nameField = self.$form.find('#alert-name');
                var currentName = $nameField.val();

                // If name is empty and email is valid, auto-generate name
                if (!currentName && email && self.isValidEmail(email)) {
                    var defaultName = (self.vars.create_alert || 'Project Alert') + ' - ' + email;
                    $nameField.val(defaultName);
                }

                if (email && self.isValidEmail(email) && $field.hasClass('has-error')) {
                    $field.removeClass('has-error');
                    $field.find('.field-error').remove();
                }
            });

            // Clear auto-generated name if user starts typing
            this.$form.find('#alert-name').on('input focus', function() {
                var $nameField = $(this);
                var $emailField = self.$form.find('#alert-email');
                var email = $emailField.val();
                var currentName = $nameField.val();

                // If name matches auto-generated pattern, clear it when user focuses
                if (currentName && email && currentName === (self.vars.create_alert || 'Project Alert') + ' - ' + email) {
                    if ($(document.activeElement)[0] === $nameField[0]) {
                        $nameField.val('');
                    }
                }
            });
        },

        /**
         * Setup error clearing on field change
         */
        setupErrorClearing: function() {
            var self = this;
            // Clear errors for multi-select taxonomies
            this.$form.find('select[multiple]').on('change', function() {
                var $field = $(this).closest('.form-field');
                var selected = $(this).val();
                var hasValue = false;

                if (selected && Array.isArray(selected) && selected.length > 0) {
                    hasValue = true;
                } else if (selected && !Array.isArray(selected) && selected !== '' && selected !== null) {
                    hasValue = true;
                }

                if (hasValue && $field.hasClass('has-error')) {
                    // Check if this is the first taxonomy field with error
                    var $firstError = self.$form.find('select[multiple]').closest('.form-field.has-error').first();
                    if ($firstError.length && $firstError[0] === $field[0]) {
                        $field.removeClass('has-error');
                        $field.find('.field-error').remove();
                    }
                }
            });

            // Fallback for non-Select2 selects
            this.$form.find('select:not([multiple])').on('change', function () {
                var $select = $(this);
                var selectId = $select.attr('id');
                if ((selectId === 'alert-frequency' || selectId === 'alert-match-type') && !$select.hasClass('select2-hidden-accessible')) {
                    if (window.ProjectAlertsProgress && window.ProjectAlertsProgress.updateProgress) {
                        window.ProjectAlertsProgress.updateProgress();
                    }
                }
            });
        },

        /**
         * Email validation helper
         */
        isValidEmail: function(email) {
            var re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return re.test(email);
        },

        /**
         * Show notice
         */
        showNotice: function(message, type) {
            var $notice = $('#form-notice');
            $notice.removeClass('success error warning').addClass(type);
            $notice.html(message).stop(true, true).hide().fadeIn();

            // Scroll to notice if it's an error
            if (type === 'error' && $notice.is(':visible')) {
                $('html, body').animate({
                    scrollTop: $notice.offset().top - 100
                }, 500);
            }

            // Auto hide after 5 seconds
            setTimeout(function () {
                $notice.fadeOut();
            }, 5000);
        },

        /**
         * Hide notice
         */
        hideNotice: function() {
            $('#form-notice').fadeOut();
        },

        /**
         * Setup form submission
         */
        setupFormSubmission: function() {
            var self = this;
            this.$form.on('submit', function (e) {
                e.preventDefault();

                // Validate
                if (window.ProjectAlertsValidation && window.ProjectAlertsValidation.validateForm) {
                    var validation = window.ProjectAlertsValidation.validateForm();
                    if (!validation.isValid) {
                        var $firstError = self.$form.find('.has-error').first();
                        if ($firstError.length) {
                            $('html, body').animate({
                                scrollTop: $firstError.offset().top - 100
                            }, 500);
                        }
                        return;
                    }
                }

                // Collect form data
                var getSelectValue = function(selector, isMultiple) {
                    var $select = self.$form.find(selector);
                    if (!$select.length) {
                        return isMultiple ? [] : '';
                    }

                    var val = $select.val();

                    if (isMultiple) {
                        // Ensure it's an array
                        if (!val) return [];
                        if (Array.isArray(val)) return val;
                        if (val !== '' && val !== null) return [val];
                        return [];
                    } else {
                        // Single select
                        return val || '';
                    }
                };

                var editId = $('#submit-alert').data('edit-id');
                var formData = {
                    action: 'felan_project_alerts_action',
                    nonce: self.nonce,
                    name: self.$form.find('#alert-name').val() || '',
                    email: self.$form.find('#alert-email').val() || '',
                    categories: getSelectValue('#alert-categories', true) || [],
                    skills: getSelectValue('#alert-skills', true) || [],
                    language: getSelectValue('#alert-language', true) || [],
                    career: getSelectValue('#alert-career', true) || [],
                    location: getSelectValue('#alert-location', true) || [],
                    state: getSelectValue('#alert-state', true) || [],
                    match_type: getSelectValue('#alert-match-type', false) || 'OR',
                    frequency: getSelectValue('#alert-frequency', false) || 'daily'
                };

                // Add alert_id if editing
                if (editId) {
                    formData.alert_id = editId;
                }

                // Show loading state
                var $submitBtn = $('#submit-alert');
                var $btnText = $submitBtn.find('.btn-text');
                var $btnLoading = $submitBtn.find('.btn-loading');
                $submitBtn.prop('disabled', true);
                $btnText.hide();
                $btnLoading.show();

                // Hide previous notices
                self.hideNotice();

                // AJAX request
                $.ajax({
                    type: 'POST',
                    url: self.ajaxUrl,
                    data: formData,
                    dataType: 'json',
                    success: function (response) {
                        $submitBtn.prop('disabled', false);
                        $btnText.show();
                        $btnLoading.hide();
                        if (self.syncStickyButton) {
                            self.syncStickyButton();
                        }

                        if (response.success) {
                            var isEdit = $('#submit-alert').data('edit-id');
                            var successMsg = isEdit
                                ? (self.vars.update_success || 'Alert updated successfully!')
                                : (response.message || self.vars.success_message || 'Project alert created successfully!');

                            self.showNotice(successMsg, 'success');

                            // Reset edit mode
                            $('#submit-alert').data('edit-id', '');
                            $('#submit-alert .btn-text').text(self.vars.create_alert || 'Create Project Alert');
                            if (self.syncStickyButton) {
                                self.syncStickyButton();
                            }
                            $('#edit-mode-banner').slideUp(300);
                            // Show "Back to List" button when not editing
                            $('#btn-back-to-list').show();
                            $('#form-title-text').text(self.vars.create_alert || 'Create Project Alert');
                            // Update icon for create mode
                            var $formTitleIcon = $('#form-title-icon');
                            if ($formTitleIcon.length) {
                                $formTitleIcon.removeClass('fa-edit').addClass('fa-plus-circle');
                            }
                            $('.felan-project-alerts-title h2').text(self.vars.title || 'Project Alerts');

                            // Reload page to get fresh server-rendered list
                            if (window.ProjectAlertsList && window.ProjectAlertsList.reloadAlertsList) {
                                setTimeout(function() {
                                    window.ProjectAlertsList.reloadAlertsList();
                                }, 500);
                            } else {
                                // Fallback: reload page
                                setTimeout(function() {
                                    window.location.reload();
                                }, 500);
                            }
                        } else {
                            self.showNotice(response.message || self.vars.error_message || 'An error occurred. Please try again.', 'error');
                        }
                    },
                    error: function (xhr, status, error) {
                        $submitBtn.prop('disabled', false);
                        $btnText.show();
                        $btnLoading.hide();
                        if (self.syncStickyButton) {
                            self.syncStickyButton();
                        }

                        var errorMsg = self.vars.error_message || 'An error occurred. Please try again.';

                        // Try to parse JSON response
                        try {
                            var response = JSON.parse(xhr.responseText);
                            if (response.message) {
                                errorMsg = response.message;
                            }
                        } catch (e) {
                            // Handle different error types
                            if (xhr.status === 0) {
                                errorMsg = self.vars.network_error || 'Network error. Please check your connection and try again.';
                            } else if (xhr.status === 500) {
                                errorMsg = self.vars.server_error || 'Server error. Please try again later.';
                            } else if (xhr.responseText) {
                                errorMsg = self.vars.generic_error || 'An error occurred. Please refresh the page and try again.';
                            }
                        }

                        self.showNotice(errorMsg, 'error');
                    }
                });
            });
        },

        /**
         * Show form
         */
        showForm: function(isEdit) {
            var $formWrapper = $('#project-alerts-form-wrapper');
            var $formTitle = $('#form-title-text');
            var $alertsList = $('#project-alerts-list');
            var $backBtn = $('#btn-back-to-list');

            if (isEdit) {
                $formTitle.text(this.vars.edit_alert || 'Edit Project Alert');
                // Hide "Back to List" button when editing
                $backBtn.hide();
            } else {
                $formTitle.text(this.vars.create_alert || 'Create Project Alert');
                // Show "Back to List" button when creating
                $backBtn.show();
            }

            // Trigger package edit mode
            if (window.ProjectAlertsPackage && typeof window.ProjectAlertsPackage.setEditMode === 'function') {
                window.ProjectAlertsPackage.setEditMode(isEdit);
            }

            // Hide alerts list completely
            $alertsList.hide();
            // Show form
            $formWrapper.slideDown(300);
            $('html, body').animate({
                scrollTop: $formWrapper.offset().top - 100
            }, 500);
        },

        /**
         * Hide form
         */
        hideForm: function() {
            var $formWrapper = $('#project-alerts-form-wrapper');
            var $alertsList = $('#project-alerts-list');
            var isEditing = $('#submit-alert').data('edit-id');

            $formWrapper.slideUp(300, function() {
                $alertsList.show();
            });

            // Show alerts list immediately to prevent race conditions
            $alertsList.show();

            // Reset form if not editing (to avoid infinite loop with cancelEdit)
            if (!isEditing) {
                this.$form[0].reset();

                // Reset package edit mode to false (create mode)
                if (window.ProjectAlertsPackage && typeof window.ProjectAlertsPackage.setEditMode === 'function') {
                    window.ProjectAlertsPackage.setEditMode(false);
                }

                if ($.fn.select2) {
                    this.$form.find('.felan-select2').val(null).trigger('change');
                }
                if (window.ProjectAlertsProgress && typeof window.ProjectAlertsProgress.updateProgress === 'function') {
                    window.ProjectAlertsProgress.updateProgress();
                }
            }
        },

        /**
         * Reset form
         */
        resetForm: function() {
            this.$form[0].reset();
            if ($.fn.select2) {
                this.$form.find('.felan-select2').val(null).trigger('change');
            }
            if (window.ProjectAlertsProgress && window.ProjectAlertsProgress.updateProgress) {
                window.ProjectAlertsProgress.updateProgress();
            }
        },

        /**
         * Populate form with alert data
         */
        populateForm: function(alert) {
            var fieldMap = {
                'name': '#alert-name',
                'email': '#alert-email',
                'categories': '#alert-categories',
                'skills': '#alert-skills',
                'language': '#alert-language',
                'career': '#alert-career',
                'location': '#alert-location',
                'state': '#alert-state',
                'match_type': '#alert-match-type',
                'frequency': '#alert-frequency'
            };

            // Handle name field separately (skip if auto-generated)
            if (alert.name && alert.name !== (this.vars.create_alert || 'Project Alert') + ' - ' + alert.email) {
                this.$form.find(fieldMap.name).val(alert.name);
            } else {
                this.$form.find(fieldMap.name).val('');
            }

            // Populate other fields
            var self = this;
            Object.keys(fieldMap).forEach(function(key) {
                if (key === 'name') return; // Already handled
                var $field = self.$form.find(fieldMap[key]);
                // Only populate if field exists (may be hidden if taxonomy not allowed)
                if ($field.length && alert[key]) {
                    var value = alert[key];
                    if (Array.isArray(value) && value.length > 0) {
                        $field.val(value).trigger('change');
                    } else if (!Array.isArray(value) && value !== '') {
                        $field.val(value).trigger('change');
                    }
                }
            });
        },

        /**
         * Setup sticky button functionality
         */
        setupStickyButton: function() {
            var self = this;
            var $submitBtn = $('#submit-alert');
            var $stickySubmitBtn = $('#submit-alert-sticky');
            var $stickyWrapper = $('#form-submit-sticky');
            var $formWrapper = $('#project-alerts-form-wrapper');

            // Function to check if submit button is in viewport
            function isSubmitButtonInViewport() {
                if (!$submitBtn.length || !$formWrapper.is(':visible')) {
                    return true; // If form is hidden, don't show sticky
                }
                var rect = $submitBtn[0].getBoundingClientRect();
                return rect.top >= 0 && rect.bottom <= window.innerHeight;
            }

            // Function to update sticky button visibility
            function updateStickyButton() {
                if (!$formWrapper.is(':visible')) {
                    $stickyWrapper.hide();
                    return;
                }

                if (isSubmitButtonInViewport()) {
                    $stickyWrapper.slideUp(200);
                } else {
                    $stickyWrapper.slideDown(200);
                }
            }

            // Sync sticky button state with original button
            self.syncStickyButton = function() {
                var $originalText = $submitBtn.find('.btn-text');
                var $originalLoading = $submitBtn.find('.btn-loading');
                var $stickyText = $stickySubmitBtn.find('.btn-text');
                var $stickyLoading = $stickySubmitBtn.find('.btn-loading');

                // Sync text
                $stickyText.text($originalText.text());

                // Sync disabled state
                $stickySubmitBtn.prop('disabled', $submitBtn.prop('disabled'));

                // Sync loading state
                if ($originalLoading.is(':visible')) {
                    $stickyText.hide();
                    $stickyLoading.show();
                } else {
                    $stickyText.show();
                    $stickyLoading.hide();
                }
            };

            // Handle sticky button click
            $stickySubmitBtn.on('click', function(e) {
                e.preventDefault();
                // Trigger form submit
                self.$form.trigger('submit');
            });

            // Update sticky button on scroll
            var scrollTimeout;
            $(window).on('scroll', function() {
                clearTimeout(scrollTimeout);
                scrollTimeout = setTimeout(function() {
                    updateStickyButton();
                    self.syncStickyButton();
                }, 100);
            });

            // Update sticky button on resize
            var resizeTimeout;
            $(window).on('resize', function() {
                clearTimeout(resizeTimeout);
                resizeTimeout = setTimeout(function() {
                    updateStickyButton();
                }, 100);
            });

            // Update sticky button when form visibility changes
            var observer = new MutationObserver(function() {
                updateStickyButton();
                self.syncStickyButton();
            });

            if ($formWrapper.length) {
                observer.observe($formWrapper[0], {
                    attributes: true,
                    attributeFilter: ['style', 'class']
                });
            }

            // Initial check
            setTimeout(function() {
                updateStickyButton();
                self.syncStickyButton();
            }, 500);

            // Sync sticky button when original button state changes
            var originalButtonObserver = new MutationObserver(function() {
                self.syncStickyButton();
            });

            if ($submitBtn.length) {
                originalButtonObserver.observe($submitBtn[0], {
                    childList: true,
                    subtree: true,
                    attributes: true,
                    attributeFilter: ['disabled', 'class']
                });
            }
        }
    };

    // Export to global namespace
    window.ProjectAlertsForm = ProjectAlertsForm;

})(jQuery);

