/**
 * Project Alerts List Module
 * Handles alerts list loading, rendering, and actions (edit/delete)
 */
(function ($) {
    'use strict';

    var ProjectAlertsList = {
        vars: {},
        ajaxUrl: '',
        nonce: '',

        /**
         * Initialize list module
         */
        init: function(vars) {
            this.vars = vars || {};
            this.ajaxUrl = vars.ajax_url || (typeof ajaxurl !== 'undefined' ? ajaxurl : '/wp-admin/admin-ajax.php');
            this.nonce = vars.nonce || '';

            // Setup button handlers
            this.setupButtonHandlers();

            // Attach event handlers for server-rendered list
            this.attachAlertListHandlers();
        },

        /**
         * Setup button handlers
         */
        setupButtonHandlers: function() {
            var self = this;
            // Create alert button handlers
            $(document).on('click', '#btn-create-alert, #btn-create-alert-empty', function(e) {
                e.preventDefault();
                if (window.ProjectAlertsForm && window.ProjectAlertsForm.showForm) {
                    window.ProjectAlertsForm.showForm(false);
                }
            });

            // Back to list button
            $(document).on('click', '#btn-back-to-list', function(e) {
                e.preventDefault();
                if (window.ProjectAlertsForm && window.ProjectAlertsForm.hideForm) {
                    window.ProjectAlertsForm.hideForm();
                }
            });
        },

        /**
         * Helper function to escape HTML
         */
        escapeHtml: function(text) {
            if (!text) return '';
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return String(text).replace(/[&<>"']/g, function(m) { return map[m]; });
        },

        /**
         * Format date for display
         */
        formatDate: function(dateString) {
            if (!dateString) return '';
            var date = new Date(dateString);
            if (isNaN(date.getTime())) return dateString;

            var now = new Date();
            var diff = now - date;
            var seconds = Math.floor(diff / 1000);
            var minutes = Math.floor(seconds / 60);
            var hours = Math.floor(minutes / 60);
            var days = Math.floor(hours / 24);

            if (seconds < 60) {
                return self.vars.just_now || 'Just now';
            } else if (minutes < 60) {
                return minutes + ' ' + (minutes > 1 ? (self.vars.minutes || 'minutes') : (self.vars.minute || 'minute')) + ' ' + (self.vars.ago || 'ago');
            } else if (hours < 24) {
                return hours + ' ' + (hours > 1 ? (self.vars.hours || 'hours') : (self.vars.hour || 'hour')) + ' ' + (self.vars.ago || 'ago');
            } else if (days < 7) {
                return days + ' ' + (days > 1 ? (self.vars.days || 'days') : (self.vars.day || 'day')) + ' ' + (self.vars.ago || 'ago');
            } else {
                // Format as date
                var options = { year: 'numeric', month: 'short', day: 'numeric' };
                return date.toLocaleDateString('en-US', options);
            }
        },

        /**
         * Reload alerts list (server-side rendered)
         * Reloads the page to get fresh server-rendered content
         */
        reloadAlertsList: function() {
            // Remove edit parameter from URL if exists
            var url = window.location.pathname;
            var urlParams = new URLSearchParams(window.location.search);
            urlParams.delete('edit');
            var queryString = urlParams.toString();
            if (queryString) {
                url += '?' + queryString;
            }
            window.location.href = url;
        },

        /**
         * Render alerts list
         */
        renderAlertsList: function(alerts) {
            var self = this;
            var $listContent = $('#alerts-list-content');
            var html = '<div class="alerts-grid">';

            alerts.forEach(function(alert) {
                var isEditing = $('#submit-alert').data('edit-id') == alert.id;
                html += '<div class="alert-item' + (isEditing ? ' alert-item-editing' : '') + '" data-id="' + alert.id + '">';
                html += '<div class="alert-item-header">';
                html += '<div class="alert-item-title-wrapper">';
                // Get alert title and decode HTML entities properly
                var alertTitle = alert.name || alert.email || '';
                html += '<h4 class="alert-item-title">' + self.escapeHtml(alertTitle) + '</h4>';
                if (isEditing) {
                    html += '<span class="alert-item-badge editing-badge"><i class="far fa-edit" aria-hidden="true"></i> ' + (self.vars.editing_alert || 'Editing') + '</span>';
                }
                html += '</div>';
                html += '<div class="alert-item-actions">';
                html += '<button type="button" class="btn-edit-alert' + (isEditing ? ' active' : '') + '" data-id="' + alert.id + '" aria-label="' + (self.vars.edit_alert || 'Edit Alert') + '">';
                html += '<i class="far fa-edit" aria-hidden="true"></i> <span class="btn-text">' + (self.vars.edit_alert || 'Edit') + '</span>';
                html += '</button>';
                html += '<button type="button" class="btn-delete-alert" data-id="' + alert.id + '" aria-label="' + (self.vars.delete_alert || 'Delete') + '">';
                html += '<i class="far fa-trash-alt" aria-hidden="true"></i>';
                html += '</button>';
                html += '</div>';
                html += '</div>';
                html += '<div class="alert-item-body">';
                html += '<div class="alert-item-meta">';
                html += '<span class="alert-meta-item"><i class="far fa-envelope" aria-hidden="true"></i> ' + self.escapeHtml(alert.email) + '</span>';
                html += '<span class="alert-meta-item"><i class="far fa-filter" aria-hidden="true"></i> ' + self.escapeHtml(alert.match_type || 'OR') + '</span>';
                html += '<span class="alert-meta-item"><i class="far fa-clock" aria-hidden="true"></i> ' + self.escapeHtml(alert.frequency || 'daily') + '</span>';
                html += '</div>';

                // Show time log - only show updated_at if exists and different from created_at, otherwise show created_at
                html += '<div class="alert-item-timestamps">';
                var hasUpdate = alert.updated_at && alert.updated_at !== alert.created_at;

                if (hasUpdate) {
                    // Show updated_at only
                    html += '<span class="alert-timestamp-item" title="' + self.escapeHtml(alert.updated_at) + '">';
                    html += '<i class="far fa-edit" aria-hidden="true"></i> ';
                    html += '<span class="timestamp-label">Updated:</span> ';
                    html += '<span class="timestamp-value">' + self.formatDate(alert.updated_at) + '</span>';
                    html += '</span>';
                } else if (alert.created_at) {
                    // Show created_at only
                    html += '<span class="alert-timestamp-item" title="' + self.escapeHtml(alert.created_at) + '">';
                    html += '<i class="far fa-calendar-plus" aria-hidden="true"></i> ';
                    html += '<span class="timestamp-label">Created:</span> ';
                    html += '<span class="timestamp-value">' + self.formatDate(alert.created_at) + '</span>';
                    html += '</span>';
                }
                html += '</div>';

                // Show selected taxonomies with better formatting
                var taxonomyConfig = [
                    { key: 'categories', icon: 'far fa-folder', label: self.vars.label_categories || 'Categories' },
                    { key: 'skills', icon: 'far fa-tags', label: self.vars.label_skills || 'Skills' },
                    { key: 'language', icon: 'far fa-globe', label: self.vars.label_language || 'Languages' },
                    { key: 'career', icon: 'far fa-briefcase', label: self.vars.label_career || 'Careers' },
                    { key: 'location', icon: 'far fa-map-marker-alt', label: self.vars.label_location || 'Location' },
                    { key: 'state', icon: 'far fa-map', label: self.vars.label_state || 'State/Province' }
                ];
                var taxonomies = [];
                var totalCount = 0;

                taxonomyConfig.forEach(function(config) {
                    if (alert[config.key] && alert[config.key].length > 0) {
                        taxonomies.push('<span class="taxonomy-badge"><i class="' + config.icon + '" aria-hidden="true"></i> ' + self.escapeHtml(config.label) + ': <strong>' + alert[config.key].length + '</strong></span>');
                        totalCount += alert[config.key].length;
                    }
                });

                if (taxonomies.length > 0) {
                    html += '<div class="alert-item-taxonomies">';
                    html += '<div class="taxonomy-badges">' + taxonomies.join('') + '</div>';
                    html += '<div class="taxonomy-summary">Total: <strong>' + totalCount + '</strong> items selected</div>';
                    html += '</div>';
                }
                html += '</div>';
                html += '</div>';
            });

            html += '</div>';
            $listContent.html(html);

            // Attach event handlers
            this.attachAlertListHandlers();
        },

        /**
         * Attach handlers for alert list actions
         */
        attachAlertListHandlers: function() {
            var self = this;
            var $confirmModal = $('#felan-confirm-delete-alert');
            var pendingDeleteId = null;

            // Edit alert
            $(document).off('click', '.btn-edit-alert').on('click', '.btn-edit-alert', function(e) {
                e.preventDefault();
                var $btn = $(this);
                var alertId = $btn.data('id');
                var $alertItem = $btn.closest('.alert-item');

                // Prevent multiple clicks
                if ($btn.hasClass('loading')) {
                    return;
                }

                // Show loading state
                $btn.addClass('loading').prop('disabled', true);
                var originalHtml = $btn.html();
                $btn.html('<i class="far fa-spinner fa-spin" aria-hidden="true"></i> <span class="btn-text">' + (self.vars.loading || 'Loading...') + '</span>');

                // Hide editing badge if exists
                $alertItem.find('.editing-badge').hide();

                self.editAlert(alertId, function() {
                    $btn.removeClass('loading').prop('disabled', false).html(originalHtml);
                }, function() {
                    $btn.removeClass('loading').prop('disabled', false).html(originalHtml);
                    $alertItem.find('.editing-badge').show();
                });
            });

            // Delete alert - show confirm modal
            $(document).off('click', '.btn-delete-alert').on('click', '.btn-delete-alert', function(e) {
                e.preventDefault();
                var alertId = $(this).data('id');
                pendingDeleteId = alertId;

                if ($confirmModal.length) {
                    $confirmModal.css({ opacity: '1', visibility: 'visible', display: 'block' });
                    $('body').addClass('popup-open');
                } else {
                    // Fallback to confirm if modal not found
                    if (confirm(self.vars.delete_confirm || 'Are you sure you want to delete this alert?')) {
                        self.deleteAlert(alertId);
                    }
                }
            });

            // Confirm delete action
            $(document).off('click', '#felan-confirm-delete-alert .btn-confirm-delete-alert').on('click', '#felan-confirm-delete-alert .btn-confirm-delete-alert', function(e) {
                e.preventDefault();
                var $btn = $(this);

                // Prevent multiple clicks
                if ($btn.hasClass('loading') || $btn.prop('disabled')) {
                    return;
                }

                if (pendingDeleteId) {
                    // Show loading state
                    $btn.addClass('loading').prop('disabled', true);
                    var originalHtml = $btn.html();
                    $btn.html('<i class="far fa-spinner fa-spin" aria-hidden="true"></i> ' + (self.vars.loading || 'Loading...'));

                    // Disable cancel button too
                    var $cancelBtn = $confirmModal.find('.btn-cancel-delete-alert');
                    $cancelBtn.prop('disabled', true);

                    // Call delete with callback to restore button state
                    self.deleteAlert(pendingDeleteId, function() {
                        // Restore button state on success
                        $btn.removeClass('loading').prop('disabled', false).html(originalHtml);
                        $cancelBtn.prop('disabled', false);
                        $confirmModal.css({ opacity: '0', visibility: 'hidden' });
                        $('body').removeClass('popup-open');
                    }, function() {
                        // Restore button state on error
                        $btn.removeClass('loading').prop('disabled', false).html(originalHtml);
                        $cancelBtn.prop('disabled', false);
                    });

                    pendingDeleteId = null;
                } else {
                    // No pending delete, just close modal
                    $confirmModal.css({ opacity: '0', visibility: 'hidden' });
                    $('body').removeClass('popup-open');
                }
            });

            // Cancel delete action
            $(document).off('click', '#felan-confirm-delete-alert .btn-cancel-delete-alert, #felan-confirm-delete-alert .btn-close').on('click', '#felan-confirm-delete-alert .btn-cancel-delete-alert, #felan-confirm-delete-alert .btn-close', function(e) {
                e.preventDefault();
                pendingDeleteId = null;
                $confirmModal.css({ opacity: '0', visibility: 'hidden' });
                $('body').removeClass('popup-open');
            });

            // Close modal when clicking overlay
            $(document).off('click', '#felan-confirm-delete-alert .bg-overlay').on('click', '#felan-confirm-delete-alert .bg-overlay', function(e) {
                e.preventDefault();
                pendingDeleteId = null;
                $confirmModal.css({ opacity: '0', visibility: 'hidden' });
                $('body').removeClass('popup-open');
            });
        },

        /**
         * Edit alert - populate form with alert data
         */
        editAlert: function(alertId, onSuccess, onError) {
            var self = this;
            $.ajax({
                type: 'POST',
                url: this.ajaxUrl,
                data: {
                    action: 'felan_get_project_alert',
                    nonce: this.nonce,
                    alert_id: alertId
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success && response.data) {
                        var alert = response.data;

                        if (window.ProjectAlertsForm && window.ProjectAlertsForm.populateForm) {
                            window.ProjectAlertsForm.populateForm(alert);
                        }

                        $('#submit-alert .btn-text').text(self.vars.save_changes || 'Save Changes');
                        $('#submit-alert').data('edit-id', alertId);
                        if (window.ProjectAlertsForm && window.ProjectAlertsForm.syncStickyButton) {
                            window.ProjectAlertsForm.syncStickyButton();
                        }

                        var alertName = alert.name || alert.email || (self.vars.create_alert || 'Alert') + ' #' + alertId;
                        $('#editing-alert-name').text(alertName);
                        $('#edit-mode-banner').slideDown(300);
                        $('#btn-back-to-list').hide();
                        $('#form-title-text').text(self.vars.edit_alert || 'Edit Project Alert');
                        // Update icon for edit mode
                        var $formTitleIcon = $('#form-title-icon');
                        if ($formTitleIcon.length) {
                            $formTitleIcon.removeClass('fa-plus-circle').addClass('fa-edit');
                        }
                        $('.felan-project-alerts-title h2').text(self.vars.edit_alert || 'Edit Project Alert');

                        // Ensure form is visible before populating
                        var $formWrapper = $('#project-alerts-form-wrapper');
                        if ($formWrapper.length && $formWrapper.is(':hidden')) {
                            $formWrapper.show();
                        }

                        if (window.ProjectAlertsForm && window.ProjectAlertsForm.showForm) {
                            window.ProjectAlertsForm.showForm(true);
                        }

                        // Wait a bit for form to be visible, then populate
                        setTimeout(function() {
                            if (window.ProjectAlertsProgress && window.ProjectAlertsProgress.updateProgress) {
                                window.ProjectAlertsProgress.updateProgress();
                            }

                            // Scroll to form
                            var $form = $('#project-alerts-form');
                            if ($form.length) {
                                $('html, body').animate({
                                    scrollTop: $form.offset().top - 100
                                }, 500);
                            }
                        }, 100);

                        // Update URL without reload
                        if (history.pushState) {
                            var newUrl = window.location.pathname + '?edit=' + alertId;
                            window.history.pushState({path: newUrl}, '', newUrl);
                        }

                        if (onSuccess) {
                            onSuccess();
                        }
                    } else {
                        if (window.ProjectAlertsForm && window.ProjectAlertsForm.showNotice) {
                            window.ProjectAlertsForm.showNotice(self.vars.error_message || 'Failed to load alert data', 'error');
                        }
                        if (onError) {
                            onError();
                        }
                    }
                },
                error: function() {
                    if (window.ProjectAlertsForm && window.ProjectAlertsForm.showNotice) {
                        window.ProjectAlertsForm.showNotice(self.vars.error_message || 'Failed to load alert data', 'error');
                    }
                    if (onError) {
                        onError();
                    }
                }
            });
        },

        /**
         * Cancel edit mode
         */
        cancelEdit: function() {
            if (window.ProjectAlertsForm && window.ProjectAlertsForm.resetForm) {
                window.ProjectAlertsForm.resetForm();
            }

            $('#submit-alert .btn-text').text(this.vars.create_alert || 'Create Project Alert');
            $('#submit-alert').data('edit-id', '');
            if (window.ProjectAlertsForm && window.ProjectAlertsForm.syncStickyButton) {
                window.ProjectAlertsForm.syncStickyButton();
            }

            // Hide edit banner
            $('#edit-mode-banner').slideUp(300);

            // Show "Back to List" button when not editing
            $('#btn-back-to-list').show();

            // Reset form title
            $('#form-title-text').text(this.vars.create_alert || 'Create Project Alert');
            // Update icon for create mode
            var $formTitleIcon = $('#form-title-icon');
            if ($formTitleIcon.length) {
                $formTitleIcon.removeClass('fa-edit').addClass('fa-plus-circle');
            }
            $('.felan-project-alerts-title h2').text(this.vars.title || 'Project Alerts');

            // Update URL
            if (history.pushState) {
                var newUrl = window.location.pathname;
                window.history.pushState({path: newUrl}, '', newUrl);
            }

            // Reload page to get fresh server-rendered list
            this.reloadAlertsList();
        },

        /**
         * Delete alert
         * @param {number} alertId - Alert ID to delete
         * @param {function} onSuccess - Callback on success
         * @param {function} onError - Callback on error
         */
        deleteAlert: function(alertId, onSuccess, onError) {
            var self = this;
            var $confirmModal = $('#felan-confirm-delete-alert');

            $.ajax({
                type: 'POST',
                url: this.ajaxUrl,
                data: {
                    action: 'felan_delete_project_alert',
                    nonce: this.nonce,
                    alert_id: alertId
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        if (window.ProjectAlertsForm && window.ProjectAlertsForm.showNotice) {
                            window.ProjectAlertsForm.showNotice(response.message || self.vars.delete_success || 'Alert deleted successfully', 'success');
                        }

                        // Call success callback if provided
                        if (onSuccess) {
                            onSuccess();
                        }

                        // Reload page to get fresh server-rendered list
                        setTimeout(function() {
                            self.reloadAlertsList();
                        }, 500);
                    } else {
                        // Close modal on error
                        if ($confirmModal.length) {
                            $confirmModal.css({ opacity: '0', visibility: 'hidden' });
                            $('body').removeClass('popup-open');
                        }

                        if (window.ProjectAlertsForm && window.ProjectAlertsForm.showNotice) {
                            window.ProjectAlertsForm.showNotice(response.message || self.vars.error_message || 'Failed to delete alert', 'error');
                        }

                        // Call error callback if provided
                        if (onError) {
                            onError();
                        }
                    }
                },
                error: function() {
                    // Close modal on error
                    if ($confirmModal.length) {
                        $confirmModal.css({ opacity: '0', visibility: 'hidden' });
                        $('body').removeClass('popup-open');
                    }

                    if (window.ProjectAlertsForm && window.ProjectAlertsForm.showNotice) {
                        window.ProjectAlertsForm.showNotice(self.vars.error_message || 'Failed to delete alert', 'error');
                    }

                    // Call error callback if provided
                    if (onError) {
                        onError();
                    }
                }
            });
        }
    };

    // Export to global namespace
    window.ProjectAlertsList = ProjectAlertsList;

})(jQuery);

