/**
 * Project Alerts Package Integration Module
 * Handles package limits checking and UI updates based on package
 */
(function ($) {
    'use strict';

    var ProjectAlertsPackage = {
        vars: {},
        packageLimits: null,

        /**
         * Initialize package module
         */
        init: function(vars) {
            this.vars = vars || {};
            this.packageLimits = this.vars.package_limits || null;

            if (!this.packageLimits) {
                return;
            }

            // Setup package UI
            this.setupPackageUI();
        },

        /**
         * Set Edit Mode
         * @param {boolean} isEdit
         */
        setEditMode: function(isEdit) {
            var $form = $('#project-alerts-form');
            var $submitBtn = $('.form-submit button');

            if (isEdit) {
                // In edit mode: enable fields and button, hide warning
                $form.removeClass('limit-reached');
                $form.find('input, select, textarea').prop('disabled', false);
                $submitBtn.prop('disabled', false).find('.btn-text').text(this.vars.save_changes || 'Save Changes');
                $('#form-notice').hide();
                // Show package limit warnings for taxonomies (skip alert limit check)
                this.setupPackageUI(true); // true = skip alert limit check
            } else {
                // In create mode: re-check limits
                $submitBtn.find('.btn-text').text(this.vars.create_alert || 'Create Project Alert');
                this.setupPackageUI(false); // false = check alert limit
            }
        },

        /**
         * Check if user can create more alerts (UX only - server will verify)
         * This is only for UI display purposes, not security
         */
        canCreateAlert: function() {
            // Note: This is UX only. Server-side validation is the source of truth.
            if (!this.packageLimits || !this.packageLimits.alerts) {
                return true; // If no data, allow UI to show (server will verify)
            }

            var alerts = this.packageLimits.alerts;

            // Check if alerts are enabled (from server)
            if (alerts.enabled === false) {
                return false;
            }

            // Check limit (UX only)
            if (alerts.limit !== 'unlimited' && alerts.current >= alerts.limit) {
                return false;
            }

            return true;
        },

        /**
         * Get current alert usage info
         */
        getAlertUsage: function() {
            if (!this.packageLimits || !this.packageLimits.alerts) {
                return null;
            }
            return this.packageLimits.alerts;
        },

        /**
         * Get allowed taxonomies for current package
         */
        getAllowedTaxonomies: function() {
            // Need to parse from packageLimits.taxonomies
            // The structure is taxonomies: { slug: { limit: '...' }, ... }
            if (!this.packageLimits || !this.packageLimits.taxonomies) {
                return [];
            }
            return Object.keys(this.packageLimits.taxonomies);
        },

        /**
         * Get max terms per taxonomy
         */
        getMaxTermsPerTaxonomy: function(taxonomy) {
            if (!this.packageLimits || !this.packageLimits.taxonomies) {
                return 'unlimited';
            }

            // Map form field data-taxonomy to PHP array keys if they differ
            var mapping = {
                'categories': 'project_cat',
                'skills': 'project_skill',
                'language': 'project-language',
                'career': 'project-career',
                'location': 'felan_location',
                'state': 'felan_state'
            };

            var phpKey = mapping[taxonomy] || taxonomy;

            var limitData = this.packageLimits.taxonomies[phpKey] || this.packageLimits.taxonomies[taxonomy];

            if (limitData) {
                // PHP returns array('enabled' => true, 'max_terms' => ...)
                if (typeof limitData === 'object' && limitData.max_terms !== undefined) {
                    return limitData.max_terms;
                }
                return limitData;
            }

            return 'unlimited';
        },

        /**
         * Validate alert against package limits (UX only - server will verify)
         * This is only for user experience, not security. Server-side validation is the source of truth.
         */
        validateAlert: function(formData) {
            var self = this;
            var result = { isValid: true, errors: [] };

            // If no package limits data, skip validation (server will handle it)
            if (!this.packageLimits) {
                return result;
            }

            // Check if alerts are enabled (from server data)
            if (this.packageLimits.alerts && this.packageLimits.alerts.enabled === false) {
                // This should not happen if form is shown, but check anyway for UX
                result.isValid = false;
                result.errors.push({
                    element: null,
                    message: 'Project alerts are not enabled for your package.'
                });
                return result;
            }

            // Validate Taxonomies (UX only - server will verify)
            $('.form-field select[data-taxonomy]').each(function() {
                var $select = $(this);
                var taxonomy = $select.data('taxonomy');
                var selected = $select.val() || [];
                var max = self.getMaxTermsPerTaxonomy(taxonomy);

                if (max !== 'unlimited' && selected.length > max) {
                    result.isValid = false;
                    var label = $select.closest('.form-field').find('label').first().clone().children().remove().end().text().trim();
                    result.errors.push({
                        element: $select,
                        message: self.vars.text_limit_exceeded
                            ? self.vars.text_limit_exceeded.replace('%s', label).replace('%d', max)
                            : (self.vars.text_limit_exceeded || 'You can only select up to %d %s. Please reduce your selection.').replace('%s', label).replace('%d', max)
                    });
                }
            });

            return result;
        },

        /**
         * Setup package UI (disable fields, show limits, etc.)
         * Note: This is UX only. Server-side validation is the source of truth.
         * @param {boolean} skipAlertLimitCheck - If true, skip the overall alert limit check (for edit mode)
         */
        setupPackageUI: function(skipAlertLimitCheck) {
            var self = this;

            if (!this.packageLimits) return;

            // 1. Check if alerts are enabled (from server data)
            var alerts = this.packageLimits.alerts;
            if (alerts && alerts.enabled === false) {
                // This should not happen if form is shown (server should hide form),
                // but check anyway for UX consistency
                $('#project-alerts-form').addClass('limit-reached');
                $('.form-submit button').prop('disabled', true).find('.btn-text').text('Project Alerts Not Enabled');

                var upgradeUrl = this.vars.upgrade_url || '#';
                var msg = '<div class="alert alert-warning">' +
                    'Project alerts are not enabled for your package.' +
                    ' <a href="'+ upgradeUrl +'" class="alert-link" style="margin-left: 15px;">'+ (this.vars.upgrade_package || 'Upgrade Package') +'</a>' +
                    '</div>';
                $('#form-notice').html(msg).show();

                // Disable all fields
                $('#project-alerts-form').find('input, select, textarea').prop('disabled', true);
                return;
            }

            // 2. Alert Quota Warning (skip in edit mode)
            if (!skipAlertLimitCheck) {
                if (alerts && alerts.limit !== 'unlimited') {
                    var remaining = alerts.limit - alerts.current;
                    if (remaining <= 0) {
                         // Show warning (UX only - server will verify)
                         $('#project-alerts-form').addClass('limit-reached');
                         $('.form-submit button').prop('disabled', true).find('.btn-text').text(this.vars.alert_limit_reached || 'Alert Limit Reached');

                         var upgradeUrl = this.vars.upgrade_url || '#';
                         var msg = '<div class="alert alert-warning">' +
                             (this.vars.text_limit_reached || 'You have reached your alert limit.') +
                             ' <a href="'+ upgradeUrl +'" class="alert-link" style="margin-left: 15px;">'+ (this.vars.upgrade_package || 'Upgrade Package') +'</a>' +
                             '</div>';
                         $('#form-notice').html(msg).show();

                         // Disable all fields (UX only - server will verify)
                         $('#project-alerts-form').find('input, select, textarea').prop('disabled', true);
                         return;
                    }
                }
            }

            // 3. Taxonomy Limits (UX only - server will verify)
            $('.form-field select[data-taxonomy]').each(function() {
                var $select = $(this);
                var taxonomy = $select.data('taxonomy');
                var max = self.getMaxTermsPerTaxonomy(taxonomy);
                var $label = $select.closest('.form-field').find('label');

                // Helper to update label
                var updateLabel = function() {
                    var selected = $select.val() || [];
                    var current = selected.length;

                    // Find or create badge
                    var $badge = $label.find('.package-limit-info');
                    if ($badge.length === 0) {
                        $badge = $('<span class="package-limit-info ms-2"></span>');
                        $label.append($badge);
                    }

                    if (max !== 'unlimited') {
                        var countText = ' (' + current + '/' + max + ')';

                        // Reset styles first
                        $badge.attr('style', '').css({
                            'font-size': '0.9em',
                            'margin-left': '5px',
                            'color': '#666',
                            'display': 'inline-block'
                        }).text(countText);

                        // Add warning color if limit reached/exceeded
                        if (current > max) {
                            $badge.css({
                                'color': '#dc3545',
                                'border': '1px solid #dc3545',
                                'padding': '2px 4px',
                                'border-radius': '4px',
                                'line-height': '1.2'
                            });
                        } else if (current === max) {
                            $badge.css('color', '#f59e0b'); // Amber 500
                        }
                    } else {
                        // Hide badge if unlimited
                        $badge.hide();
                    }
                };

                // Initial update
                updateLabel();

                // Attach change listener
                $select.on('change', function() {
                    updateLabel();
                    // Additional visual feedback
                    var selected = $select.val() || [];
                    var $field = $select.closest('.form-field');

                    if (max !== 'unlimited' && selected.length > max) {
                        $field.addClass('has-error');
                        // Explicitly target Select2 container if present
                        if ($select.hasClass('select2-hidden-accessible')) {
                            $select.next('.select2-container').find('.select2-selection').css('border-color', '#ef4444');
                        }
                    } else {
                        $field.removeClass('has-error');
                         if ($select.hasClass('select2-hidden-accessible')) {
                            // Reset border color (empty string usually reverts to CSS)
                            $select.next('.select2-container').find('.select2-selection').css('border-color', '');
                        }
                    }
                });
            });
        }
    };

    // Export to global namespace
    window.ProjectAlertsPackage = ProjectAlertsPackage;

})(jQuery);
