/**
 * Project Alerts Progress Indicator Module
 * Handles progress indicator display and interactions
 */
(function ($) {
    'use strict';

    var ProjectAlertsProgress = {
        vars: {},
        $form: null,

        /**
         * Initialize progress module
         */
        init: function(vars, $form) {
            this.vars = vars || {};
            this.$form = $form || $('#project-alerts-form');

            if (!this.$form.length) {
                return;
            }

            // Initial update
            this.updateProgress();

            // Attach handlers initially
            this.attachRemoveHandlers();
            this.attachClearAllHandlers();
        },

        /**
         * Get field label (localized)
         */
        getFieldLabel: function(selectId) {
            // Try to get from localized strings first
            var labelKey = 'label_' + selectId.replace('alert-', '').replace('-', '_');
            if (this.vars[labelKey]) {
                return this.vars[labelKey];
            }

            // Fallback to default labels
            var labels = {
                'alert-categories': this.vars.label_categories || 'Categories',
                'alert-skills': this.vars.label_skills || 'Skills',
                'alert-language': this.vars.label_language || 'Languages',
                'alert-career': this.vars.label_career || 'Careers',
                'alert-location': this.vars.label_location || 'Location',
                'alert-state': this.vars.label_state || 'State/Province',
                'alert-match-type': 'Match Type',
                'alert-frequency': 'Notification Frequency'
            };
            return labels[selectId] || selectId.replace('alert-', '').replace(/-/g, ' ');
        },

        /**
         * Get field icon
         */
        getFieldIcon: function(selectId) {
            var icons = {
                'alert-categories': 'far fa-folder',
                'alert-skills': 'far fa-tags',
                'alert-language': 'far fa-globe',
                'alert-career': 'far fa-briefcase',
                'alert-location': 'far fa-map-marker-alt',
                'alert-state': 'far fa-map',
                'alert-match-type': 'far fa-filter',
                'alert-frequency': 'far fa-clock'
            };
            return icons[selectId] || 'far fa-check';
        },

        /**
         * Helper function to decode HTML entities
         */
        decodeHtmlEntities: function(text) {
            if (!text) return '';
            var textarea = document.createElement('textarea');
            textarea.innerHTML = text;
            return textarea.value;
        },

        /**
         * Helper function to escape HTML
         */
        escapeHtml: function(text) {
            if (!text) return '';
            // First decode any existing HTML entities to avoid double encoding
            var decoded = this.decodeHtmlEntities(String(text));
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return decoded.replace(/[&<>"']/g, function(m) { return map[m]; });
        },

        /**
         * Update progress indicator with details
         */
        updateProgress: function() {
            var self = this;
            var count = 0;
            var details = [];
            var fieldOrder = ['alert-categories', 'alert-skills', 'alert-language', 'alert-career', 'alert-location', 'alert-state', 'alert-match-type', 'alert-frequency'];

            // Process each field in order
            fieldOrder.forEach(function(selectId) {
                var $select = $('#' + selectId);
                if (!$select.length) {
                    return;
                }

                var selected = $select.val();
                var items = [];
                var isMultiple = $select.prop('multiple');

                if (isMultiple) {
                    // Multi-select
                    if (selected && Array.isArray(selected) && selected.length > 0) {
                        selected.forEach(function(value) {
                            var $option = $select.find('option[value="' + value + '"]');
                            var text = $option.text() || value;
                            if (text && text.trim() !== '') {
                                items.push({
                                    value: value,
                                    text: self.escapeHtml(text)
                                });
                                count++;
                            }
                        });
                    }
                } else {
                    // Single select (match_type and frequency)
                    if (selected && selected !== '' && selected !== null) {
                        var $option = $select.find('option[value="' + selected + '"]');
                        var text = $option.text() || selected;
                        if (text && text.trim() !== '') {
                            items.push({
                                value: selected,
                                text: self.escapeHtml(text)
                            });
                            count++;
                        }
                    }
                }

                // Check limits if package module is available
                var isLimitExceeded = false;
                var maxLimit = 'unlimited';

                if (window.ProjectAlertsPackage && typeof window.ProjectAlertsPackage.getMaxTermsPerTaxonomy === 'function') {
                   var taxonomy = $select.data('taxonomy');
                   if (taxonomy) {
                       maxLimit = window.ProjectAlertsPackage.getMaxTermsPerTaxonomy(taxonomy);
                       if (maxLimit !== 'unlimited' && items.length > maxLimit) {
                           isLimitExceeded = true;
                       }
                   }
                }

                // Add to details if has items
                if (items.length > 0) {
                    details.push({
                        id: selectId,
                        label: self.getFieldLabel(selectId),
                        icon: self.getFieldIcon(selectId),
                        items: items,
                        count: items.length,
                        isExceeded: isLimitExceeded,
                        maxLimit: maxLimit
                    });
                }
            });

            // Update count
            var $countEl = $('#selected-count');
            if ($countEl.length) {
                $countEl.text(count);
            }

            // Update details and show/hide form-progress
            var $detailsEl = $('#progress-details');
            var $formProgress = $('#form-progress');
            if ($detailsEl.length) {
                if (details.length === 0) {
                    $detailsEl.html('<div class="progress-empty">' + (this.vars.no_taxonomies_selected || 'No taxonomies selected yet') + '</div>');
                    // Hide form-progress when no taxonomies selected
                    if ($formProgress.length) {
                        $formProgress.hide();
                    }
                } else {
                    // Show form-progress when taxonomies are selected
                    if ($formProgress.length) {
                        $formProgress.show();
                    }
                    var html = '';
                    details.forEach(function(detail) {
                        var taxonomyClass = 'progress-taxonomy';
                        var countStyle = '';
                        var labelStyle = '';

                        if (detail.isExceeded) {
                             taxonomyClass += ' limit-exceeded';
                             countStyle = 'color: #dc3545; font-weight: bold;';
                             labelStyle = 'color: #dc3545;';
                        }

                        html += '<div class="' + taxonomyClass + '" data-field="' + detail.id + '" style="' + (detail.isExceeded ? 'border-left: 3px solid #dc3545; padding-left: 10px;' : '') + '">';
                        html += '<div class="taxonomy-header">';
                        html += '<i class="' + detail.icon + '" aria-hidden="true" style="' + labelStyle + '"></i>';
                        html += '<span class="taxonomy-label" style="' + labelStyle + '">' + self.escapeHtml(detail.label) + '</span>';

                        var countText = '(' + detail.count + ')';
                        if (detail.isExceeded) {
                            countText = '(' + detail.count + '/' + detail.maxLimit + ')';
                        }
                        html += '<span class="taxonomy-count" style="' + countStyle + '">' + countText + '</span>';

                        if (detail.isExceeded) {
                             html += '<span class="badge badge-danger ms-2" style="background: #dc3545; color: white; padding: 2px 6px; border-radius: 4px; font-size: 10px; margin-left: 5px;">Exceeded</span>';
                        }

                        html += '<button type="button" class="taxonomy-clear-all" data-field="' + detail.id + '" aria-label="Clear all ' + self.escapeHtml(detail.label) + '">';
                        html += '<i class="far fa-times-circle" aria-hidden="true"></i>';
                        html += '<span class="clear-all-text">Clear All</span>';
                        html += '</button>';
                        html += '</div>';
                        html += '<div class="taxonomy-items">';
                        detail.items.forEach(function(item) {
                            html += '<span class="taxonomy-item" data-value="' + self.escapeHtml(String(item.value)) + '" data-field="' + detail.id + '">';
                            html += '<span class="taxonomy-item-text">' + item.text + '</span>';
                            html += '<button type="button" class="taxonomy-item-remove" aria-label="Remove ' + item.text + '">';
                            html += '<i class="far fa-times" aria-hidden="true"></i>';
                            html += '</button>';
                            html += '</span>';
                        });
                        html += '</div>';
                        html += '</div>';
                    });
                    $detailsEl.html(html);

                    // Attach remove handlers
                    self.attachRemoveHandlers();

                    // Attach clear all handlers
                    self.attachClearAllHandlers();
                }
            }
        },

        /**
         * Attach remove handlers for taxonomy items
         */
        attachRemoveHandlers: function() {
            var self = this;
            $('#progress-details').off('click', '.taxonomy-item-remove').on('click', '.taxonomy-item-remove', function(e) {
                e.preventDefault();
                e.stopPropagation();

                var $item = $(this).closest('.taxonomy-item');
                var value = $item.data('value');
                var fieldId = $item.data('field');
                var $select = $('#' + fieldId);

                if (!$select.length) {
                    return;
                }

                // Remove the value from current selection
                var currentValues = $select.val() || [];
                if (!Array.isArray(currentValues)) {
                    currentValues = [currentValues];
                }
                var newValues = currentValues.filter(function(v) {
                    return String(v) !== String(value);
                });

                // Update Select2
                $select.val(newValues).trigger('change');

                setTimeout(function() {
                    self.updateProgress();
                }, 150);
            });
        },

        /**
         * Attach clear all handlers for taxonomy groups
         */
        attachClearAllHandlers: function() {
            var self = this;
            $('#progress-details').off('click', '.taxonomy-clear-all').on('click', '.taxonomy-clear-all', function(e) {
                e.preventDefault();
                e.stopPropagation();

                var $button = $(this);
                var fieldId = $button.data('field');
                var $select = $('#' + fieldId);

                if (!$select.length) {
                    return;
                }

                // Clear all values
                $select.val($select.prop('multiple') ? [] : null).trigger('change');

                setTimeout(function() {
                    self.updateProgress();
                }, 150);
            });
        }
    };

    // Export to global namespace
    window.ProjectAlertsProgress = ProjectAlertsProgress;

})(jQuery);

