/**
 * Project Alerts Validation Module
 * Handles form validation logic
 */
(function ($) {
    'use strict';

    var ProjectAlertsValidation = {
        vars: {},
        $form: null,

        /**
         * Initialize validation module
         */
        init: function(vars, $form) {
            this.vars = vars || {};
            this.$form = $form || $('#project-alerts-form');

            if (!this.$form.length) {
                return;
            }
        },

        /**
         * Helper function to escape HTML
         */
        escapeHtml: function(text) {
            if (!text) return '';
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return String(text).replace(/[&<>"']/g, function(m) { return map[m]; });
        },

        /**
         * Clear all field errors
         */
        clearFieldErrors: function() {
            this.$form.find('.form-field').removeClass('has-error');
            // Hide and clear all error messages (including placeholders)
            this.$form.find('.field-error').hide().html('');
        },

        /**
         * Show field error
         */
        showFieldError: function($field, message) {
            $field.addClass('has-error');
            var $error = $field.find('.field-error');
            if ($error.length) {
                $error.html(this.escapeHtml(message)).show();
            } else {
                $error = $('<div class="field-error" role="alert">' + this.escapeHtml(message) + '</div>');
                // Append to the form-field container (after input/select)
                $field.append($error);
            }
        },

        /**
         * Email validation helper
         */
        isValidEmail: function(email) {
            var re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return re.test(email);
        },

        /**
         * Form validation with error placement
         */
        validateForm: function() {
            var self = this;
            var isValid = true;
            this.clearFieldErrors();

            // Validate email
            var $emailField = this.$form.find('#alert-email').closest('.form-field');
            var email = this.$form.find('#alert-email').val();
            if (!email || !this.isValidEmail(email)) {
                isValid = false;
                this.showFieldError($emailField, this.vars.validation_email || 'Please enter a valid email address');
            }

            // Validate at least one taxonomy
            var hasTaxonomy = false;
            this.$form.find('select[multiple]').each(function () {
                var selected = $(this).val();
                if (selected && Array.isArray(selected) && selected.length > 0) {
                    hasTaxonomy = true;
                } else if (selected && !Array.isArray(selected) && selected !== '' && selected !== null) {
                    hasTaxonomy = true;
                }
            });

            if (!hasTaxonomy) {
                isValid = false;
                // Show error on first taxonomy field
                var $firstTaxonomyField = this.$form.find('select[multiple]').first().closest('.form-field');
                if ($firstTaxonomyField.length) {
                    this.showFieldError($firstTaxonomyField, this.vars.validation_taxonomy || 'Please select at least one taxonomy');
                } else if (window.ProjectAlertsForm && window.ProjectAlertsForm.showNotice) {
                    window.ProjectAlertsForm.showNotice(this.vars.validation_taxonomy || 'Please select at least one taxonomy', 'error');
                }
            }

            // Validate Match Type
            var $matchTypeField = this.$form.find('#alert-match-type').closest('.form-field');
            var matchType = this.$form.find('#alert-match-type').val();
            if (!matchType || matchType === '' || matchType === null) {
                isValid = false;
                this.showFieldError($matchTypeField, this.vars.validation_match_type || 'Please select a match type');
            }

            // Validate Notification Frequency
            var $frequencyField = this.$form.find('#alert-frequency').closest('.form-field');
            var frequency = this.$form.find('#alert-frequency').val();
            if (!frequency || frequency === '' || frequency === null) {
                isValid = false;
                this.showFieldError($frequencyField, this.vars.validation_frequency || 'Please select a notification frequency');
            }

            // Package Validation (UX only - server will verify on submit)
            // Note: This is for user experience only. Server-side validation is the source of truth.
            if (window.ProjectAlertsPackage && window.ProjectAlertsPackage.validateAlert) {
                var pkgResult = window.ProjectAlertsPackage.validateAlert();
                if (!pkgResult.isValid) {
                    isValid = false;
                    $.each(pkgResult.errors, function(i, error) {
                        if (error.element && error.message) {
                            self.showFieldError(error.element.closest('.form-field'), error.message);
                        } else if (error.message) {
                            // Show general error if no specific element
                            if (window.ProjectAlertsForm && window.ProjectAlertsForm.showNotice) {
                                window.ProjectAlertsForm.showNotice(error.message, 'error');
                            }
                        }
                    });
                }
            }

            return {
                isValid: isValid,
                errors: []
            };
        }
    };

    // Export to global namespace
    window.ProjectAlertsValidation = ProjectAlertsValidation;

})(jQuery);

