/**
 * Wire Transfer Modal Handler
 * Reusable JavaScript for wire transfer payment modal
 */

var FELAN_WIRE_TRANSFER_MODAL = FELAN_WIRE_TRANSFER_MODAL || {};

(function ($) {
  "use strict";

  FELAN_WIRE_TRANSFER_MODAL = {
    /**
     * Initialize modal functionality
     */
    init: function () {
      this.bindEvents();
    },

    /**
     * Show wire transfer modal
     * @param {string|number} itemId - Package/Service/Project ID
     * @param {function} onConfirm - Callback function when user confirms transfer
     */
    show: function (itemId, onConfirm) {
      var $modal = $("#felan-wire-transfer-modal");

      if ($modal.length === 0) {
        console.error("Wire transfer modal not found");
        return;
      }

      // Store item ID and callback
      $modal.data("item-id", itemId);
      $modal.data("confirm-callback", onConfirm);

      // Show modal
      $modal.fadeIn(300);
      $("body").addClass("modal-open");
    },

    /**
     * Hide wire transfer modal
     */
    hide: function () {
      $("#felan-wire-transfer-modal").fadeOut(300);
      $("body").removeClass("modal-open");
    },

    /**
     * Bind all event handlers
     */
    bindEvents: function () {
      var self = this;

      // Close button click
      $(document).on("click", ".felan-modal-close", function () {
        self.hide();
      });

      // Overlay click
      $(document).on("click", ".felan-modal-overlay", function () {
        self.hide();
      });

      // Confirm transfer button
      $(document).on("click", ".felan-confirm-wire-transfer", function () {
        var $modal = $("#felan-wire-transfer-modal");
        var itemId = $modal.data("item-id");
        var callback = $modal.data("confirm-callback");

        // Hide modal
        self.hide();

        // Execute callback if provided
        if (typeof callback === "function") {
          callback(itemId);
        }
      });

      // Copy button functionality
      $(document).on("click", ".copy-btn", function (e) {
        e.preventDefault();
        self.copyToClipboard($(this));
      });

      // ESC key to close
      $(document).on("keydown", function (e) {
        if (e.key === "Escape" || e.keyCode === 27) {
          if ($("#felan-wire-transfer-modal").is(":visible")) {
            self.hide();
          }
        }
      });
    },

    /**
     * Copy text to clipboard
     * @param {jQuery} $btn - Copy button element
     */
    copyToClipboard: function ($btn) {
      var textToCopy = $btn.data("copy");

      // Try modern clipboard API first
      if (navigator.clipboard && navigator.clipboard.writeText) {
        navigator.clipboard.writeText(textToCopy).then(
          function () {
            this.showCopyFeedback($btn);
          }.bind(this),
          function () {
            // Fallback to old method
            this.copyToClipboardFallback($btn, textToCopy);
          }.bind(this)
        );
      } else {
        // Fallback for older browsers
        this.copyToClipboardFallback($btn, textToCopy);
      }
    },

    /**
     * Fallback copy method for older browsers
     * @param {jQuery} $btn - Copy button element
     * @param {string} text - Text to copy
     */
    copyToClipboardFallback: function ($btn, text) {
      var $temp = $("<input>");
      $("body").append($temp);
      $temp.val(text).select();

      try {
        document.execCommand("copy");
        this.showCopyFeedback($btn);
      } catch (err) {
        console.error("Failed to copy text:", err);
      }

      $temp.remove();
    },

    /**
     * Show visual feedback after copy
     * @param {jQuery} $btn - Copy button element
     */
    showCopyFeedback: function ($btn) {
      var originalHtml = $btn.html();
      $btn.html('<i class="fas fa-check"></i>');
      $btn.addClass("copied");

      setTimeout(function () {
        $btn.html(originalHtml);
        $btn.removeClass("copied");
      }, 1500);
    },
  };

  // Initialize on document ready
  $(document).ready(function () {
    FELAN_WIRE_TRANSFER_MODAL.init();
  });
})(jQuery);
