(function ($) {
  "use strict";
  $(document).ready(function () {
    var ajax_url = felan_template_vars.ajax_url,
      $form_popup = $("#form-invite-project-popup"),
      $projectList = $form_popup.find('.project-list'),
      $messageContainer = $form_popup.find('.invite-message');

    // Search functionality - filter projects (only if search box exists)
    var $searchInput = $form_popup.find('.project-search-input');

    if ($searchInput.length > 0) {
      function filterProjects(searchTerm) {
        searchTerm = (searchTerm || '').toLowerCase();

        // Remove no-results message first
        $projectList.find('.no-results').remove();

        // Filter projects
        var visibleCount = 0;
        $projectList.find('li').each(function () {
          var $li = $(this);
          // Skip no-results item
          if ($li.hasClass('no-results')) {
            return;
          }

          var projectTitle = $li.data('project-title') || '';
          if (projectTitle.indexOf(searchTerm) !== -1) {
            $li.show();
            visibleCount++;
          } else {
            $li.hide();
          }
        });

        // Show "No results" message if needed
        if (visibleCount === 0 && searchTerm.length > 0) {
          if ($projectList.find('.no-results').length === 0) {
            $projectList.append('<li class="no-results"><span>' + (felan_template_vars.no_results_found || 'No projects found') + '</span></li>');
          }
        }
      }

      // Search on keyup
      $searchInput.on('keyup', function () {
        var searchTerm = $(this).val();
        filterProjects(searchTerm);
      });

      // Search on focus - show all projects immediately
      $searchInput.on('focus', function () {
        var searchTerm = $(this).val();
        // If there's a search term, filter; otherwise show all
        if (searchTerm.length > 0) {
          filterProjects(searchTerm);
        } else {
          // Show all projects when focus
          $projectList.find('li').not('.no-results').show();
          $projectList.find('.no-results').remove();
        }
      });
    }

    // Clear search on clear button
    $(".felan-clear-project-invite").on("click", function (e) {
      e.preventDefault();
      $form_popup.find('input[type="checkbox"]').prop("checked", false);
      // Only clear search if search input exists
      if ($searchInput.length > 0) {
        $searchInput.val('').trigger('keyup');
      }
    });

    // Helper function to show message
    function showMessage(message, type) {
      type = type || 'error'; // 'error' or 'success'
      $messageContainer
        .removeClass('error success')
        .addClass(type)
        .html('<span>' + message + '</span>')
        .fadeIn();

      // Auto hide after 5 seconds
      setTimeout(function () {
        $messageContainer.fadeOut();
      }, 5000);
    }

    // Helper function to hide message
    function hideMessage() {
      $messageContainer.fadeOut();
    }

    // Helper function to show field error
    function showFieldError(fieldName, message) {
      var $field = $form_popup.find('.project-selection-field');
      var $errorContainer = $field.find('.project-selection-error');

      $field.addClass('has-error');
      $errorContainer
        .html(message)
        .fadeIn();
    }

    // Helper function to clear field errors
    function clearFieldErrors() {
      var $field = $form_popup.find('.project-selection-field');
      $field.removeClass('has-error');
      $field.find('.project-selection-error').fadeOut().html('');
    }

    // Clear error when user selects/deselects a project
    $form_popup.on('change', 'input[name="projects_invite[]"]', function() {
      clearFieldErrors();
    });

    // Handle form submission
    $form_popup.each(function () {
      var $btn_submit = $form_popup.find("#btn-saved-project-invite");
      $btn_submit.on("click", function (e) {
        e.preventDefault();
        var $this = $(this),
          $form = $form_popup.find('form'),
          freelancer_id = $form.find('input[name="freelancer_id"]').val(),
          author_id = $form.find('input[name="author_id"]').val(),
          list_projects = $form.find('input[name="list_projects"]').val(),
          nonce = $form.find('input[name="nonce"]').val(),
          projects_id = $form.find('input[name="projects_invite[]"]:checked')
            .map(function () {
              return $(this).val();
            })
            .get();

        // Hide previous messages and clear field errors
        hideMessage();
        clearFieldErrors();

        // Manual Validation - Client Side
        var validationErrors = [];

        // 1. Validate nonce
        if (!nonce || nonce.trim() === '') {
          console.error('Nonce not found in form');
          validationErrors.push({
            field: 'nonce',
            message: 'Security token missing. Please refresh the page and try again.'
          });
        }

        // 2. Validate required fields
        if (!freelancer_id || freelancer_id <= 0 || isNaN(parseInt(freelancer_id))) {
          validationErrors.push({
            field: 'freelancer_id',
            message: 'Invalid freelancer information. Please refresh the page and try again.'
          });
        }

        if (!author_id || author_id <= 0 || isNaN(parseInt(author_id))) {
          validationErrors.push({
            field: 'author_id',
            message: 'Invalid freelancer information. Please refresh the page and try again.'
          });
        }

        // 3. Validate list_projects is valid JSON
        var list_projects_array = [];
        if (list_projects) {
          try {
            list_projects_array = JSON.parse(list_projects);
            if (!Array.isArray(list_projects_array)) {
              validationErrors.push({
                field: 'list_projects',
                message: 'Invalid project list format. Please refresh the page and try again.'
              });
            }
          } catch (e) {
            console.error('Invalid JSON in list_projects:', e);
            validationErrors.push({
              field: 'list_projects',
              message: 'Invalid project list format. Please refresh the page and try again.'
            });
          }
        }

        // 4. Validate projects_id array
        if (!Array.isArray(projects_id)) {
          projects_id = [];
        }

        // 5. Validate project IDs are positive integers
        var invalidProjects = projects_id.filter(function(id) {
          return !id || isNaN(id) || parseInt(id) <= 0;
        });

        if (invalidProjects.length > 0) {
          validationErrors.push({
            field: 'projects_invite',
            message: 'Invalid project selected. Please refresh the page and try again.'
          });
        }

        // 6. Validate at least one project selected (most important validation)
        // Check if there are existing invites to determine if this is a "remove all" action
        var hasExistingInvites = false;
        if (list_projects_array.length > 0) {
          // Check if any projects in the list are currently checked (already invited)
          $form.find('input[name="projects_invite[]"]').each(function() {
            if ($(this).is(':checked')) {
              hasExistingInvites = true;
              return false; // break
            }
          });
        }

        // If no projects selected and no existing invites to remove, show error
        if (projects_id.length === 0 && !hasExistingInvites) {
          validationErrors.push({
            field: 'projects_invite',
            message: 'Please select at least one project to invite.'
          });
        }

        // Display validation errors
        if (validationErrors.length > 0) {
          // Show field-specific errors first
          var projectSelectionError = validationErrors.find(function(err) {
            return err.field === 'projects_invite';
          });

          if (projectSelectionError) {
            showFieldError('projects_invite', projectSelectionError.message);
          }

          // Show general errors for other fields
          var generalErrors = validationErrors.filter(function(err) {
            return err.field !== 'projects_invite';
          });

          if (generalErrors.length > 0) {
            var generalMessage = generalErrors.map(function(err) {
              return err.message;
            }).join(' ');
            showMessage(generalMessage, 'error');
          }

          // Scroll to first error
          var $firstErrorField = $form_popup.find('.has-error');
          if ($firstErrorField.length) {
            $form_popup.find('.custom-scrollbar').animate({
              scrollTop: $firstErrorField.offset().top - $form_popup.find('.invite-popup').offset().top - 20
            }, 300);
          }

          return;
        }

        $.ajax({
          type: "POST",
          url: ajax_url,
          dataType: "json",
          data: {
            action: "felan_add_to_project_invite",
            freelancer_id: freelancer_id,
            author_id: author_id,
            list_projects: list_projects,
            projects_id: projects_id,
            nonce: nonce,
          },
          beforeSend: function () {
            $this.find(".btn-loading").fadeIn();
            $this.prop("disabled", true);
          },
          success: function (data) {
            $this.find(".btn-loading").fadeOut();
            $this.prop("disabled", false);

            if (data.success == true) {
              // Show success message
              if (data.message) {
                showMessage(data.message, 'success');
                // Close popup after 1.5 seconds
                setTimeout(function () {
                  $form_popup.removeClass("open");
                  location.reload();
                }, 1500);
              } else {
                // Close popup immediately if no message
                $form_popup.removeClass("open");
                location.reload();
              }
            } else {
              // Show error message
              var errorMessage = data.message || 'An error occurred. Please try again.';
              showMessage(errorMessage, 'error');
            }
          },
          error: function (xhr, status, error) {
            $this.find(".btn-loading").fadeOut();
            $this.prop("disabled", false);
            showMessage('An error occurred. Please try again.', 'error');
          },
        });
      });
    });
  });
})(jQuery);

